#!/usr/bin/perl 
#===============================================================================
# Copyright 2012 K.K.DNAFORM
# This file is part of idr_paraclu program.
# Idr_paraclu is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, any later version.
#
# Idr_paraclu is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Foobar. If not, see <http://www.gnu.org/licenses/>.
#
#         FILE:  convertIDRtoBED.pl
#
#        USAGE:  convertIDRtoBED.pl input idr bedprefix 
#					input				input file (IDR's result)
#					idr					IDR value as a threshold
#					bedprefix			output BED file's prefix
#
#  DESCRIPTION:  Converting the IDR's result file to BED and bedGraph file.
#				 The following 6 files are created.
#					BED file for replicate1
#					bedGraph file on the forward strand for replicate1 
#					bedGraph file on the reverse strand for replicate1
#					BED file for replicate2
#					bedGraph file on the forward strand for replicate2
#					bedGraph file on the reverse strand for replicate2
#
# REQUIREMENTS:  ---
#        NOTES:  ---
#       AUTHOR:  Ohmiya
#      CREATED:  2011/10/27
#     REVISION:  ---
#===============================================================================

use strict;
use warnings;

## bedGraph options
my $bedTemplate = "track type=bed name=\"%s at replicate%d (%s)\" description=\"%s at replicate%d (%s)\" visibility=2 itemRgb=On\n"; 
my $bedGraphTemplate = "track type=bedGraph name=\"%s at replicate%d (%s)\" description=\"%s at replicate%d (%s)\" visibility=full color=200,100,0 altColor=0,100,200\n"; 

## Color definition for stability
my $stable = "0,0,102";
my $unstable = "0,153,153";

if( $#ARGV != 2 ) {
	print "Usage : convertIDRtoBED.pl input idr bedprefix\n";
	exit 1;
}

chomp( my $inputFileName = $ARGV[0] );
chomp( my $idr = $ARGV[1] );
chomp( my $bed = $ARGV[2] );
my $bedFileName = $bed . ".bed";
my $bedGraphForward1 = $bed . "_forward_replicate1.bedgraph";
my $bedGraphReverse1 = $bed . "_reverse_replicate1.bedgraph";
my $bedGraphForward2 = $bed . "_forward_replicate2.bedgraph";
my $bedGraphReverse2 = $bed . "_reverse_replicate2.bedgraph";

open INPUT, "<$inputFileName" or die "Can't open $inputFileName : $!";
open BED, ">$bedFileName" or die "Can't open $bedFileName : $!";
open FORWARD1, ">$bedGraphForward1" or die "Can't open $bedGraphForward1 : $!";
open REVERSE1, ">$bedGraphReverse1" or die "Can't open $bedGraphReverse1 : $!";
open FORWARD2, ">$bedGraphForward2" or die "Can't open $bedGraphForward2 : $!";
open REVERSE2, ">$bedGraphReverse2" or die "Can't open $bedGraphReverse2 : $!";

## -----------------------------------------
## Output the track line
## -----------------------------------------
my $data = sprintf( $bedGraphTemplate, $bed, 1, "forward strand", $bed, 1, "forward strand" );
print FORWARD1 "$data";
$data = sprintf( $bedGraphTemplate, $bed, 2, "forward strand", $bed, 2, "forward strand" );
print FORWARD2 "$data";
$data = sprintf( $bedGraphTemplate, $bed, 1, "reverse strand", $bed, 1, "reverse strand" );
print REVERSE1 "$data";
$data = sprintf( $bedGraphTemplate, $bed, 2, "reverse strand", $bed, 2, "reverse strand" );
print REVERSE2 "$data";


## -----------------------------------------
## Read the input file
## -----------------------------------------
## Ignore the header
<INPUT>;
while( my $inputdata = <INPUT> ) {
	chomp( $inputdata );
	my @inputList = split( /\s+/, $inputdata );
	if( $inputList[5] >= $idr ) {
		next;
	}
	my $idrID = substr( $inputList[1], 1, length( $inputList[1] ) - 2 );
	my @idrIDList = split( /_/, $idrID );
	my $chr = $idrIDList[0];
	my $start = $idrIDList[1];
	my $end = $idrIDList[2];
	my $strand = $idrIDList[3];

	## Output the data to the BED file
	$data = "$chr\t$start\t$end\t$inputList[2]\t$inputList[5]\t$strand\n";
	print BED "$data";

	my $stability = $inputList[2];
	my $color = $unstable;
	if( $stability >= 2.0 ) {
		$color = $stable;
	}

	## Output the data to the bedGraph file for replicate1
	my $value;
	if( $strand eq "+" ) {
		$value = $stability;
		$data = "$chr\t$start\t$end\t$value\n";
		print FORWARD1 "$data";
	} else {
		$value = 0 - $stability;
		$data = "$chr\t$start\t$end\t$value\n";
		print REVERSE1 "$data";
	}

	$stability = $inputList[4];
	$color = $unstable;
	if( $stability >= 2.0 ) {
		$color = $stable;
	}

	## Output the data to the bedGraph file for replicate2
	if( $strand eq "+" ) {
		$value = $stability;
		$data = "$chr\t$start\t$end\t$value\n";
		print FORWARD2 "$data";
	} else {
		$value = 0 - $stability;
		$data = "$chr\t$start\t$end\t$value\n";
		print REVERSE2 "$data";
	}
}
close( INPUT );
close( BED );
close( FORWARD1 );
close( REVERSE1 );
close( FORWARD2 );
close( REVERSE2 );

