/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.server.nexaweb.web.filter;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import jp.co.fujitsu.reffi.common.nexaweb.constant.GlobalConstants;

import org.apache.commons.lang.StringUtils;

/**
 * <p>[概 要]</p>
 * 非同期連続通信用 動的URLフィルタクラス。
 * <p>[詳 細]</p>
 * 動的にIDが付加されたURL情報からアクセスするサーバ側リソースの解析、転送を行います。
 * 
 * <p>[備 考]</p>
 * Nexawebでは同一URLへの多重アクセスに制限がかかっているため、
 * 本フィルターを利用することで同一URLへの多重アクセスを実現します。
 * 
 * <b>使用例）</b><br>
 * ・web.xmlへフィルターの設定が必要です。
 * <pre class="samplecode">
 *  &lt;filter&gt;
 *     &lt;filter-name&gt;DynamicUrlFilter&lt;/filter-name&gt;
 *     &lt;filter-class&gt;jp.co.fujitsu.reffi.server.filter.DynamicUrlFilter&lt;/filter-class&gt;
 *  &lt;/filter&gt;
 *  &lt;filter-mapping&gt;
 *     &lt;filter-name&gt;DynamicUrlFilter&lt;/filter-name&gt;
 *     &lt;url-pattern&gt;/AsyncReq/*&lt;/url-pattern&gt;
 *  &lt;/filter-mapping&gt;
 * </pre>

 * ・アクションでの使用例
 * <pre class="samplecode">
 * 
 *    public class OpenPostalWindowAction extends BaseAction {
 *    
 *        // ①モデルを複数(例として10個)登録します
 *        &#064;Override
 *        protected void reserveModels(List<Class<? extends Model>> models) {
 *          for (int i = 0; i < 10; i++) {
 *            models.add(HTTPRequestCore.class);
 *          }
 *        }
 *
 *        // ②非同期通信&非同期連続通信フラグをtrueに設定
 *        &#064;Override
 *        public void nextModel(int index, Model prev, Model next) throws Exception{
 *          if (next == null) {
 *            return true;
 *          }
 *
 *          ((HTTPRequestCore) next).setRequestUrl("/DemoMultiWindow.do");
 *          //非同期通信フラグtrue
 *          ((HTTPRequestCore) next).setAsync(true);
 *          //非同期連続通信フラグtrue
 *          ((HTTPRequestCore) next).setDynamicUrl(true);
 *        }
 *
 *        // ③正常終了処理結果をハンドリングします
 *        &#064;Override
 *        public void successForward(int index, Model model, Object result) throws Exception {
 *            System.out.println("postal.xal受信、レンダリング正常終了");
 *        }
 *    
 *    }
 *
 * </pre>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * @author Project Reffi
 */
public class DynamicUrlFilter implements Filter {

	/** パスセパレータ */
	private static String SEPARATOR = "/";
	/** QueryString接頭辞 */
	private static String QUERY_HEAD = "?";

	@Override
	public void init(FilterConfig config) throws ServletException {
		/** */
	}

	@Override
	public void doFilter(ServletRequest request, ServletResponse response,
			FilterChain chain) throws IOException, ServletException {
		HttpServletRequest req = (HttpServletRequest) request;
		HttpServletResponse res = (HttpServletResponse) response;

		//コンテキストパスの取得
		String contextPath = req.getContextPath();
		if (contextPath != null && contextPath.equals(SEPARATOR)) {
			contextPath = "";
		}
		//パスの取得
		String path = getPath(req);
		
		if (!StringUtils.isBlank(path)
				&& path.contains(GlobalConstants.DYNAMIC_URL_PATH)) {
			String[] names = StringUtils.split(path, SEPARATOR);
			StringBuilder realPath = new StringBuilder(getActionPath(names));
			if (!StringUtils.isBlank(req.getQueryString())) {
				res.sendRedirect(contextPath + realPath.toString() + QUERY_HEAD
						+ req.getQueryString());
				return;
			} else {
				forward(req, res, realPath.toString());
				return;
			}
		} else {
			chain.doFilter(request, response);
		}
	}

	@Override
	public void destroy() {
		/** */
	}

	/**
	 * サーブレットにフォワードします。
	 * 
	 * @param request リクエスト
	 * @param response レスポンス
	 * @param actionPath アクションパス
	 * @throws IOException IO例外が発生した場合
	 * @throws ServletException サーブレット例外が発生した場合
	 */
	private void forward(HttpServletRequest request,
			HttpServletResponse response, String actionPath)
			throws IOException, ServletException {
		String forwardPath = actionPath;
		request.getRequestDispatcher(forwardPath).forward(request, response);
	}

	/**
	 * パスを返します。
	 * 
	 * @param request リクエスト
	 * @return パス
	 */
	private String getPath(HttpServletRequest request) {
		String path = request.getPathInfo();
		if (StringUtils.isEmpty(path)) {
			path = request.getServletPath();
		}
		return path;
	}

	/**
	 * アクションのパスを返します。
	 * 
	 * @param names パスを/で区切った配列
	 * @param index インデックス
	 * @return アクションのパス
	 */
	private String getActionPath(String[] names) {
		StringBuilder sb = new StringBuilder();
		for (int i = 2; i < names.length; i++) {
			sb.append(SEPARATOR).append(names[i]);
		}
		return sb.toString();
	}
}
