/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.action;

import java.awt.Dimension;
import java.awt.Rectangle;
import java.awt.Window;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.Vector;

import jp.co.fujitsu.reffi.client.nexaweb.controller.BaseController;
import jp.co.fujitsu.reffi.client.nexaweb.controller.ClientConfig;
import jp.co.fujitsu.reffi.client.nexaweb.controller.ParameterMapping;
import jp.co.fujitsu.reffi.client.nexaweb.listener.SubscribeManager;
import jp.co.fujitsu.reffi.client.nexaweb.parser.ElementValueParser;
import jp.co.fujitsu.reffi.client.nexaweb.parser.ElementValues;
import jp.co.fujitsu.reffi.client.nexaweb.parser.Parser;
import jp.co.fujitsu.reffi.client.nexaweb.validator.CustomValidator;
import jp.co.fujitsu.reffi.client.nexaweb.validator.ValidateError;
import jp.co.fujitsu.reffi.client.nexaweb.validator.ValidateErrors;
import jp.co.fujitsu.reffi.common.exception.CoreLogicException;

import com.nexaweb.client.ClientEvent;
import com.nexaweb.client.displayservice.DisplayService;
import com.nexaweb.client.netservice.HttpResponse;
import com.nexaweb.plugin.data.DataFramework;
import com.nexaweb.plugin.data.DataService;
import com.nexaweb.plugin.data.datasource.DataSourceContainer;
import com.nexaweb.plugin.data.datasource.ObjectDataSource;
import com.nexaweb.plugin.validation.exceptions.ValidationException;
import com.nexaweb.util.Log;
import com.nexaweb.xml.Document;
import com.nexaweb.xml.Element;
import com.nexaweb.xml.ParserException;
import com.nexaweb.xml.xpath.XPathFactory;

/**
 * <p>[概 要] </p>
 * 全アクションクラスの抽象基底アクションクラスです。
 * 
 * <p>[詳 細] </p>
 * アクションクラスとして動作する為の必要最低限機能と、
 * ウィンドウ操作やエレメント取得の為のAPIを提供します。
 * 
 * <p>[備 考] </p>
 * 
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public abstract class AbstractAction implements Action {

	/** MVC各レイヤを伝播するパラメータオブジェクトです。 */
	private ParameterMapping parameterMapping;

	/** このアクションを起動したコントローラです。 */
	private BaseController controller;

	/**
	 * <p>[概 要] </p>
	 * MVC各レイヤを伝播するパラメータオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * parameterMappingフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return MVC各レイヤを伝播するパラメータオブジェクト
	 */
	public ParameterMapping getParameterMapping() {
		return this.parameterMapping;
	}

	/**
	 * <p>[概 要] </p>
	 * MVC各レイヤを伝播するパラメータオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * parameterMappingフィールドを引数parameterMappingで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param parameterMapping
	 */
	public void setParameterMapping(ParameterMapping parameterMapping) {
		this.parameterMapping = parameterMapping;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return コントローラオブジェクト
	 */
	public BaseController getController() {
		return controller;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラオブジェクトを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * controllerフィールドを引数controllerで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param controller コントローラオブジェクト
	 */
	public void setController(BaseController controller) {
		this.controller = controller;
	}

	/**
	 * <p>[概 要] </p>
	 * コントローラにコールされるアクションの主幹メソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 引数parameterMappingをフィールドに保存し、このアクションを起こした
	 * エレメントが所属するウィンドウレベルエレメントをparameterMappingに設定します。
	 * 
	 * <p>[備 考] </p>
	 * このメソッドをオーバーライドして新たなアクション基底クラスを作成する場合、
	 * super.run(parameterMapping);を記述する必要が有ります。
	 * 
	 * @param parameterMapping MVC各レイヤを伝播するパラメータオブジェクト
	 */
	public ParameterMapping run(ParameterMapping parameterMapping)
			throws Exception {
		setParameterMapping(parameterMapping);
		// イベント発生元ウィンドウレベルエレメントをパラメータオブジェクトに設定
		parameterMapping.setEventSourceWindow(getOwnWindow());

		return parameterMapping;
	}

	/**
	 * <p>[概 要] </p>
	 * バリデーションを行うメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * validatorsメソッドで登録されたValidatorの数分、バリデーションを行います。
	 * バリデーションエラーが発生した場合、エラー情報がValidateErrorオブジェクトに設定され、
	 * 戻り値であるValidateErrorsオブジェクトに追加されます。<br>
	 * 
	 * <p>[備 考] </p>
	 * ValidateErrors返却後、{@link BaseAction#execute(ClientEvent, ParameterMapping)}は
	 * {@link BaseAction#validationFault(ValidateErrors)}メソッドをテンプレートコールします。<br>
	 * ValidateErrorをハンドリングする場合は、validationFaultメソッドをオーバーライドして下さい。
	 * 
	 * @param validators validatorsメソッドで設定されたバリデータオブジェクト群
	 * @return バリデーションエラー保持リストオブジェクト
	 * @throws Exception
	 */
	protected ValidateErrors validate(List<CustomValidator> validators)
			throws Exception {
		ValidateErrors validateErrors = new ValidateErrors();

		try {
			for (CustomValidator validator : validators) {
				Element element = validator.getElement();
				returnElementStatusBeforeError(element);
			}

			for (CustomValidator validator : validators) {
				validator.setSession(getController().getSession());

				// バリデート実行
				if (!validator.execute()) {
					validateErrors.addError(new ValidateError(validator
							.getElement(), validator.getErrorMessage(),
							validator.getHeadWord()));
				}
			}
		} catch (ValidationException e) {
			throw new CoreLogicException("EFC0009");
		}

		return validateErrors;
	}

	/**
	 * <p>[概 要] </p>
	 * バリデーションエラーが発生した時にコールされるメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 一つでもバリデーションエラーが発生した場合に呼び出されます。<br>
	 * ClientConfigオブジェクトの設定によってデフォルト処理が異なります。<p>
	 * 
	 * 【ClientConfig#isEnableValidationFaultProcessing()がtrue】<br>
	 * デフォルトエラーハンドリング処理を行います。<p>
	 * 
	 * 【ClientConfig#isComponentColorAndTipChangeOnValidationFault()がtrue】<br>
	 * ・エラーコンポーネントの背景色をcomponent.color.on.validation.faultの値で変更します。<br>
	 * ・エラーコンポーネントのツールチップをエラーメッセージに変更します。<p>
	 * 
	 * 【ClientConfig#isDisplayDialogOnValidationFault()がtrue】<br>
	 * 全エラーメッセージをダイアログ表示します。
	 * 
	 * <p>[備 考] </p>
	 * デフォルトエラーハンドリング処理を行わない場合、
	 * ClientConfig#setEnableValidationFaultProcessing(false)を実行して下さい。
	 * 
	 * @see BaseController#initialize()
	 * @param errors {@link #validate(List)}で生成されたエラーリストオブジェクト
	 */
	public void validationFault(ValidateErrors errors) {
		ClientConfig config = getController().getClientConfig();
		if (config.isEnableValidationFaultProcessing()) {
			// バリデーションエラーが一個でも有った場合
			if (errors.hasError()) {
				// 複数個のバリデーションエラー文言を纏める
				StringBuilder messageBuilder = new StringBuilder();
				// コンフィグで設定したエラー表示数分またはエラー数分ループ
				for (int i = 0; i < errors.size()
						&& i < config.getMaxDisplayCountOnValidationFault(); i++) {
					ValidateError error = errors.getError(i);
					Element element = error.getElement();
					String headWord = error.getHeadWord();
					String message = error.getMessage();

					if (errors.size() > 1) {
						messageBuilder.append((i + 1) + ". ");
					}
					messageBuilder.append(headWord);
					messageBuilder.append(" : ");
					messageBuilder.append(message);
					messageBuilder.append(System.getProperty("line.separator"));

					if (config.isComponentColorAndTipChangeOnValidationFault()) {
						// エラーの有ったエレメントのクローンを採る
						// クローンはエラーから復帰した時の為に使用する
						getController().getErrorElementSnapshot().snap(element);
						element.setAttribute("backgroundColor", config
								.getComponentColorOnValidationFault());
						element.setAttribute("tooltip", message);
					}
				}

				if (config.isDisplayDialogOnValidationFault()) {
					alertDialog("ValidationError", messageBuilder.toString());
				}
			}
		}
	}

	/**
	 * <p>[概 要] </p>
	 * エレメントの背景色とチップをバリデーションエラー発生前の状態に戻します。
	 * 
	 * <p>[詳 細] </p>
	 * BaseControllerに保存されているElementSnapshotオブジェクトから
	 * 引数で指定されたエレメントのエラー前状態エレメントを取得します。
	 * エラー前状態エレメントからbackgroundColor、tooltip属性を取り出し、現在の
	 * エレメントに属性セットします。
	 * ElementSnapshotオブジェクトに引数指定されたエレメントが存在しなかった場合は、
	 * 処理を行わずにfalseを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param element 状態を元に戻すエレメント
	 * @return true : 状態復帰
	 */
	protected boolean returnElementStatusBeforeError(Element element) {
		boolean ret = false;

		// バリデーションエラー前のエレメントを取得
		Element previousErrorElement = getController()
				.getErrorElementSnapshot().fetch(element);

		// バリデーションエラー前のエレメントが有った場合
		if (previousErrorElement != null) {
			// 背景色を元に戻す
			String backgroundColor = previousErrorElement
					.getAttribute("backgroundColor");
			if (backgroundColor != null) {
				element.setAttribute("backgroundColor", backgroundColor);
			} else {
				element.removeAttribute("backgroundColor");
			}
			// ツールチップを元に戻す
			String tooltip = previousErrorElement.getAttribute("tooltip");
			if (tooltip != null) {
				element.setAttribute("tooltip", tooltip);
			} else {
				element.removeAttribute("tooltip");
			}
			// 状態復帰処理成功
			ret = true;
		}

		return ret;
	}

	/**
	 * <p>[概 要] </p>
	 * このActionを発生させたコンポーネントが属するウィンドウレベルコンポーネントから、
	 * 引数nameをname属性値として持つエレメントを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 自ウィンドウエレメントを求め、{@link #getElementByNameFromWindow(Element, String)}
	 * メソッドを呼び出します。<br>
	 * 引数nameをname属性値として持つエレメントが自ウィンドウ内に無かった場合はnullを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 同じxal、同じname属性のエレメントが複数有る場合でも、
	 * このメソッドを使用することでユニークに取得出来ます。
	 * 
	 * @param name 取得するエレメントのname属性値
	 * @return 自ウィンドウ内で引数nameをnameとして持つエレメント
	 */
	protected Element getElementByNameFromOwnWindow(String name) {
		Element window = getOwnWindow();

		Element element = getElementByNameFromWindow(window, name);

		return element;
	}

	/**
	 * <p>[概 要] </p>
	 * 引数windowで指定されたウィンドウレベルコンポーネントから、
	 * 引数nameをname属性値として持つエレメントを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 第一引数エレメントの子孫エレメントの中から、引数nameをname属性として持つエレメント
	 * を返却します。
	 * 
	 * <p>[備 考] </p>
	 * Aウィンドウで発生したActionの中からBウィンドウのエレメント値を参照する場合等に使用します。
	 * 
	 * @param window 取得したいエレメントが存在するウィンドウレベルエレメント
	 * @param name 取得するエレメントのname属性値
	 * @return 他ウィンドウ内で引数nameをnameとして持つエレメント
	 */
	protected Element getElementByNameFromWindow(Element window, String name) {
		Element element = XPathFactory.createXPath(
				"descendant::*[@name='" + name + "']")
				.evaluateAsElement(window);

		return element;
	}

	/**
	 * <p>[概 要] </p>
	 * Actionを発生させたソースエレメントが属する、ウィンドウレベルエレメントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * このアクションを発生させたエレメントを取得し、
	 * 	<OL>
	 * 		<LI>rootPane</LI>
	 * 		<LI>window</LI>
	 * 		<LI>dialog</LI>
	 * 		<LI>messageDialog</LI>
	 * 	</OL>
	 * の何れかに該当するまで、DOM構造的な親を辿ります。
	 * 
	 * <p>[備 考] </p>
	 * 上記に該当したエレメントが、このアクションを発生させたエレメントが属する
	 * ウィンドウであると見做されます。
	 * 
	 * @param element
	 * @return Actionを発生させたソースエレメントが所属するウィンドウレベルエレメント
	 */
	protected Element getOwnWindow() {

		Element element = getParameterMapping().getEventSourceElement();
		List<String> windowDefs = getController().getClientConfig()
				.getWindowLevelElementDefinition();

		// カテゴライズに該当するエレメントに遭遇するまで親エレメントを辿る
		while (true) {
			if ("rootPane".equals(element.getLocalName())
					|| windowDefs.contains(element.getLocalName())) {
				// 上記に該当したら抜ける
				break;
			}
			element = element.getParent();
		}

		return element;
	}

	/**
	 * <p>[概 要] </p>
	 * ソースエレメントが属するウィンドウエレメントを削除します。
	 * 
	 * <p>[詳 細] </p>
	 * getOwnWindowメソッドで返却されたウィンドウエレメントをui DOMから削除します。
	 * 但し、ウィンドウエレメントが「rootPane」であった場合は処理しません。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	protected void closeOwnWindow() {
		Element ownWindowElement = getOwnWindow();
		if (!"rootPane".equals(ownWindowElement.getLocalName())) {
			ownWindowElement.getParent().removeChild(ownWindowElement);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * ソースエレメントが属するウィンドウの親ウィンドウエレメントを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * getOwnWindowメソッドが返却するウィンドウレベルエレメントから「parentId」
	 * 属性値を取得、その属性値を「id」として持つエレメントを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 「親」はDOM構造的な親では無く、「ウィンドウの発生元」を意味します。
	 * 
	 * @return ウィンドウ発生元ウィンドウエレメント
	 */
	protected Element getParentWindow() {
		Element parentWindow = null;

		String parentId = getOwnWindow().getAttribute("parentId");
		if (parentId != null) {
			parentWindow = getElementById(parentId);
		}

		return parentWindow;
	}

	/**
	 * <p>[概 要] </p>
	 * ソースエレメントが属するウィンドウの子ウィンドウエレメント群を返却します。
	 *  
	 * <p>[詳 細] </p>
	 * getOwnWindowメソッドが返却するウィンドウレベルエレメントから「id」
	 * 属性値を取得、その属性値を「parentId」として持つエレメント群を返却します。
	 * 
	 * <p>[備 考] </p>
	 * 「子」はDOM構造的な子では無く、「自ウィンドウから発生したウィンドウ」を意味します。
	 * 
	 * @return ウィンドウ発生先ウィンドウ群
	 */
	@SuppressWarnings("unchecked")
	protected Vector<Element> getChildWindows() {
		Document document = getController().getSession().getDocumentRegistry()
				.getUiDocument();
		String id = getOwnWindow().getAttribute("id");
		Vector<Element> childWindows = XPathFactory.createXPath(
				"//[@parentId='" + id + "']").evaluate(document);

		return childWindows;
	}

	/**
	 * <p>[概 要] </p>
	 * 子ウィンドウエレメント群を削除します。
	 * 
	 * <p>[詳 細] </p>
	 * getChildWindowsメソッドで返却されたウィンドウレベルエレメント群を、
	 * ui DOMからremoveします。結果、子ウィンドウ群は画面から削除されます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	protected void closeChildWindows() {
		Vector<Element> childWindows = getChildWindows();
		for (Element child : childWindows) {
			child.getParent().removeChild(child);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * ソースエレメントが属するウィンドウの兄弟ウィンドウエレメント群を返却します。
	 *  
	 * <p>[詳 細] </p>
	 * getOwnWindowメソッドが返却するウィンドウレベルエレメントから「parentId」
	 * 属性値を取得、その属性値を同じく「parentId」として持つエレメント群を返却します。
	 *  
	 * <p>[備 考] </p>
	 * 「兄弟」はDOM構造的な兄弟では無く、「同じウィンドウから発生したウィンドウ」を意味します。
	 * 
	 * @return 発生元が同じウィンドウ群
	 */
	@SuppressWarnings("unchecked")
	protected Vector<Element> getSiblingWindows() {
		Document document = getController().getSession().getDocumentRegistry()
				.getUiDocument();
		String parentId = getOwnWindow().getAttribute("parentId");
		Vector<Element> siblingWindows = XPathFactory.createXPath(
				"//[@parentId='" + parentId + "']").evaluate(document);

		return siblingWindows;
	}

	/**
	 * <p>[概 要] </p>
	 * 兄弟エレメント群を削除します。
	 * 
	 * <p>[詳 細] </p>
	 * getSiblingWindowsメソッドで返却されたウィンドウレベルエレメント群を、
	 * ui DOMからremoveします。結果、兄弟ウィンドウ群は画面から削除されます。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	protected void closeSiblingWindows() {
		Vector<Element> siblingWindows = getSiblingWindows();
		for (Element sibling : siblingWindows) {
			sibling.getParent().removeChild(sibling);
		}
	}

	/**
	 * <p>[概 要] </p>
	 * 画面上(ui DOM上)の全ウィンドウレベルエレメントを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * ui Dom上から全window、dialogエレメントを取得して返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	protected Vector<Element> getAllWindows() {
		Vector<Element> windows = new Vector<Element>();
		Document document = getController().getSession().getDocumentRegistry()
				.getUiDocument();

		List<String> windowDefs = getController().getClientConfig()
				.getWindowLevelElementDefinition();
		for (String windowDef : windowDefs) {
			Vector<Element> allWindows = XPathFactory.createXPath(
					"//" + windowDef).evaluate(document);
			windows.addAll(allWindows);
		}

		return windows;
	}

	/**
	 * <p>[概 要] </p>
	 * ui DOMの中から引数communicateIdをcommunicateId属性として持つ、
	 * ウィンドウレベルエレメントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * communicateIdに関しては
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.model.XalReturnPossibilityModel#addIdentifierToWindows}
	 * を参照して下さい。
	 * 
	 * @param communicateId 通信識別ID
	 * @return
	 */
	@SuppressWarnings("unchecked")
	protected Vector<Element> getWindowsByCommunicateId(String communicateId) {
		Document document = getController().getSession().getDocumentRegistry()
				.getUiDocument();
		Vector<Element> windows = XPathFactory.createXPath(
				"//[@communicateId='" + communicateId + "']")
				.evaluate(document);

		return windows;
	}

	/**
	 * <p>[概 要] </p>
	 * 自ウィンドウのcommunicateIdを返却します。
	 *   
	 * <p>[詳 細] </p>
	 * getOwnWindow()メソッドでこのアクションを起動したコンポーネントが属する
	 * ウィンドウレベルエレメントを求めた後、そのエレメントからcommunicateId属性値を
	 * 取得、返却します。
	 * 
	 * <p>[備 考] </p>
	 * communicateIdに関しては
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.model.XalReturnPossibilityModel#addIdentifierToWindows}
	 * を参照して下さい。
	 * 
	 * @return
	 */
	protected String getOwnWindowCommunicateId() {
		return getOwnWindow().getAttribute("communicateId");
	}

	/**
	 * <p>[概 要] </p>
	 * ウィンドウ位置を対象ウィンドウの位置まで移動させます。
	 *   
	 * <p>[詳 細] </p>
	 * moveToWindowPositionオーバーロードメソッドを、x_gap=0、y_gap=0でコールします。
	 * 
	 * <p>[備 考] </p>
	 * 移動後のウィンドウのx、y属性値は、baseエレメントのx、y属性値と同値です。
	 * 
	 * @param base 移動先ウィンドウ
	 * @param window 移動するウィンドウ
	 */
	protected void moveToWindowPosition(Element base, Element window) {
		moveToWindowPosition(base, window, 0, 0);
	}

	/**
	 * <p>[概 要] </p>
	 *  ウィンドウ位置を対象ウィンドウの位置まで移動させます。
	 *  
	 * <p>[詳 細] </p>
	 * baseとなるウィンドウのx座標、y座標にx_gap、y_gapをプラスした座標にwindowを移動します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param base 移動先ウィンドウ
	 * @param window 移動するウィンドウ
	 * @param x_gap 移動先ウィンドウとのx座標差異
	 * @param y_gap 移動先ウィンドウとのy座標差異
	 */
	protected void moveToWindowPosition(Element base, Element window,
			int x_gap, int y_gap) {
		Rectangle baseWindowRect = getWindowRect(base);

		int moveTo_x = baseWindowRect.x + x_gap;
		int moveTo_y = baseWindowRect.y + y_gap;

		window.setAttribute("x", String.valueOf(moveTo_x));
		window.setAttribute("y", String.valueOf(moveTo_y));
	}

	/**
	 * <p>[概 要] </p>
	 *  ウィンドウ位置を対象ウィンドウの位置（右）まで移動させます。
	 *  
	 * <p>[詳 細] </p>
	 * baseとなるウィンドウの右隣座標にwindowを移動します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param base 移動先ウィンドウ
	 * @param window 移動するウィンドウ
	 */
	protected void moveToWindowRightPosition(Element base, Element window) {
		Rectangle baseWindowRect = getWindowRect(base);

		window.setAttribute("x", String.valueOf(baseWindowRect.x
				+ baseWindowRect.width));
		window.setAttribute("y", String.valueOf(baseWindowRect.y));
	}

	/**
	 * <p>[概 要] </p>
	 *  ウィンドウ位置を対象ウィンドウの位置（下）まで移動させます。
	 *  
	 * <p>[詳 細] </p>
	 * baseとなるウィンドウの下隣座標にwindowを移動します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param base 移動先ウィンドウ
	 * @param window 移動するウィンドウ
	 */
	protected void moveToWindowBottomPosition(Element base, Element window) {
		Rectangle baseWindowRect = getWindowRect(base);

		window.setAttribute("x", String.valueOf(baseWindowRect.x));
		window.setAttribute("y", String.valueOf(baseWindowRect.y
				+ baseWindowRect.height));
	}

	/**
	 * <p>[概 要] </p>
	 * 引数指定されたwindow(dialog)の矩形サイズを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * 引数指定されたウィンドウエレメントの、
	 * 	<OL>
	 * 		<LI>x</LI>
	 * 		<LI>y</LI>
	 * 		<LI>width</LI>
	 * 		<LI>height</LI>
	 * 	</OL>
	 * 属性値を取り出し、「px」をトリムしてRectangleに格納、返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param window 矩形サイズを調べるウィンドウエレメント
	 * @return 引数windowの矩形サイズ
	 */
	protected Rectangle getWindowRect(Element window) {
		int x = Integer.valueOf(window.getAttribute("x").replaceAll("px", ""));
		int y = Integer.valueOf(window.getAttribute("y").replaceAll("px", ""));
		int width = Integer.valueOf(window.getAttribute("width").replaceAll(
				"px", ""));
		int height = Integer.valueOf(window.getAttribute("height").replaceAll(
				"px", ""));

		return new Rectangle(x, y, width, height);
	}

	/**
	 * <p>[概 要] </p>
	 * 全window(dialog)を階段状に整列します。
	 *  
	 * <p>[詳 細] </p>
	 * ui Document上の全てのウィンドウ、引数x_gap、y_gapを元に、
	 * cascadeWindowsオーバーロードメソッドをコールします。
	 * 
	 * <p>[備 考] </p>
	 * x_gap、y_gapには負数も指定出来ます。
	 * 
	 * @param x_gap 1ウィンドウ毎のx座標増加分量
	 * @param y_gap 1ウィンドウ毎のy座標増加分量
	 */
	protected void cascadeWindows(int x_gap, int y_gap) {
		cascadeWindows(getAllWindows(), x_gap, y_gap);
	}

	/**
	 * <p>[概 要] </p>
	 * 全window(dialog)を階段状に整列します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数windowsに対して、x、y座標（0始まり）を設定します。<br>
	 * Vector内の設定対象ウィンドウが変わる度、x_gap、y_gapの値を設定値にプラスします。<br>
	 * 
	 * <p>[備 考] </p>
	 * x_gap、y_gapには負数も指定出来ます。
	 *
	 * @param windows 階段状に整列させる、対象のウィンドウ群
	 * @param x_gap 1ウィンドウ毎のx座標増加分量
	 * @param y_gap 1ウィンドウ毎のy座標増加分量
	 */
	protected void cascadeWindows(Vector<Element> windows, int x_gap, int y_gap) {

		int x = 0;
		int y = 0;

		for (Element window : windows) {
			window.setAttribute("x", String.valueOf(x));
			window.setAttribute("y", String.valueOf(y));
			window.setAttribute("focused", "true");
			x += x_gap;
			y += y_gap;
		}
	}

	/**
	 * <p>[概 要] </p>
	 * 全window(dialog)をタイル状に整列します。
	 * 
	 * <p>[詳 細] </p>
	 * ui Document上の全てのウィンドウを元にtileWindowsオーバーロードメソッドを
	 * コールします。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	protected void tileWindows() {
		tileWindows(getAllWindows());
	}

	/**
	 * <p>[概 要] </p>
	 * 引数で指定されたwindow(dialog)群をタイル状に整列します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数指定されたウィンドウ数の平方根を元に、縦、横のウィンドウ数の粗値を求めた後、
	 * 平方根の少数値を判断して粗値を補正します。<br>
	 * 画面サイズを求め、縦、横ウィンドウで除算した結果を一ウィンドウのサイズにします。<br>
	 * 算出された縦横ウィンドウ表示数、ウィンドウサイズを元に引数windowsのx、y、width、height
	 * 属性値を設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param windows タイル状に整列させる対象のウィンドウ
	 */
	protected void tileWindows(Vector<Element> windows) {
		int windowNum = windows.size();

		// 縦、横ウィンドウ表示枠数を求める
		double sqrt = Math.sqrt(windowNum);
		int xWindowNum = (int) sqrt;
		int yWindowNum = (int) sqrt;

		// 誤差を元に縦、横ウィンドウ表示枠数を補正
		double margin = sqrt - (int) sqrt;
		if (margin > 0.0 && margin < 0.5) {
			xWindowNum++;
		} else if (margin >= 0.5 && margin < 1.0) {
			xWindowNum++;
			yWindowNum++;
		}

		// 描画エリアと表示ウィンドウ枠数を元に、一ウィンドウの幅、高さを求める
		Dimension area = null;
		if (getController().getSession().getApplet() != null) {
			// アプレット起動の場合
			area = getController().getSession().getApplet().getSize();
		} else {
			// Swing起動の場合
			Window[] ws = Window.getWindows();
			area = ws[0].getSize();
		}
		int windowHeight = area.height / yWindowNum;
		int windowWidth = area.width / xWindowNum;

		// 縦、横のウィンドウを表示
		for (int y = 0; y < yWindowNum; y++) {
			for (int x = 0; x < xWindowNum; x++) {
				int cnt = y * xWindowNum + x;
				if (cnt + 1 > windowNum)
					break;

				Element window = windows.get(cnt);
				window.setAttribute("x", String.valueOf(x * windowWidth));
				window.setAttribute("y", String.valueOf(y * windowHeight));
				window.setAttribute("width", String.valueOf(windowWidth));
				window.setAttribute("height", String.valueOf(windowHeight));
			}
		}
	}

	/**
	 * <p>[概 要]</p>
	 * エレメントを生成します。
	 * 
	 * <p>[詳 細] </p>
	 * 名前空間が「"http://openxal.org/ui/java"」であるNexawebエレメントを生成します。<br>
	 * 生成するエレメント種がウィンドウレベル定義されているものである場合
	 * （{@link ClientConfig#getWindowLevelElementDefinition()}に含まれるものである場合）、
	 * 「parentId」属性、「communicateId」属性が自動的に付与されます。<br>
	 * 
	 * <p>[備 考] </p>
	 * parentId、communicateIdに関しては
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.model.XalReturnPossibilityModel#addIdentifierToWindows}
	 * を参照して下さい。
	 * 
	 * @param localName 生成するエレメント名
	 * @return 生成されたエレメントオブジェクト
	 */
	protected Element createElement(String localName) {
		Document document = getController().getSession().getDocumentRegistry()
				.getUiDocument();
		Element element = document.createElement(localName,
				"http://openxal.org/ui/java");

		// ウィンドウレベル定義されているエレメントの場合はparentIdとcommunicateIdを付与
		List<String> windowDefs = getController().getClientConfig()
		.getWindowLevelElementDefinition();
		if(windowDefs.contains(element.getLocalName())){
			element.setAttribute("parentId", getOwnWindow().getAttribute("id"));
			element.setAttribute("communicateId", UUID.randomUUID().toString());
		}

		return element;
	}

	/**
	 * <p>[概 要] </p>
	 * ui Documentから、引数idをid属性値として持つエレメントを返却します。
	 *  
	 * <p>[詳 細] </p>
	 * Nexawebクライアントセッションからui Documentを取得し、
	 * UiDocument.getElementById(String)をコールします。
	 * 
	 * <p>[備 考] </p>
	 * XML idが重複するとParserはXMLをパースすることが出来ません。<br>
	 * idをエレメント取得ソースとすると、プログラム内から任意に識別可能なidを振る
	 * 必要が有りますが、同じxalを複数読み込むようなアプリケーションの場合、
	 * id conflictを回避する手法が必要になります。<p>
	 * 
	 * （Reffiフレームワークではidを任意に振らず（Nexawebに付与を委譲）、name属性値で
	 * 取得する方法を推奨しています。）
	 * 
	 * @param id 取得するエレメントID
	 * @return 引数idをid属性値として持つエレメント
	 */
	protected Element getElementById(String id) {
		Document uiDocument = getController().getSession()
				.getDocumentRegistry().getUiDocument();
		Element element = uiDocument.getElementById(id);

		return element;
	}

	/**
	 * <p>[概 要] </p>
	 * 引数nameをname属性値として持つエレメントの値を取得します。
	 *  
	 * <p>[詳 細] </p>
	 * 自ウィンドウレベルエレメント({@link #getOwnWindow()})から、
	 * 引数nameをname属性値として持つエレメントを取得、エレメントの保持する値を
	 * 汎用的な値格納オブジェクトに格納して返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param name 値を取得するエレメントのname属性値 
	 * @return エレメント値の汎用格納オブジェクト
	 * @throws Exception
	 */
	protected ElementValues getElementValueByNameFromOwnWindow(String name) throws Exception {
		ElementValues values = getElementValueByNameFromWindow(getOwnWindow(), name);

		return values;
	}

	/**
	 * <p>[概 要] </p>
	 * 引数nameをname属性値として持つエレメントの値を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 引数windowで指定されたウィンドウレベルエレメントから、
	 * 引数nameをname属性値として持つエレメントを取得、エレメントの保持する値を
	 * 汎用的な値格納オブジェクトに格納して返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param window 引数nameをname属性値として持つエレメントが存在するウィンドウレベルエレメント
	 * @param name 値を取得するエレメントのname属性値
	 * @return エレメント値の汎用格納オブジェクト
	 * @throws Exception
	 */
	protected ElementValues getElementValueByNameFromWindow(Element window, String name)
			throws Exception {
		Element element = getElementByNameFromWindow(window, name);
		if (element == null)
			return null;

		Parser parser = new ElementValueParser();
		ElementValues values = (ElementValues) parser.parse(element);

		return values;
	}

	/**
	 * <p>[概 要] </p>
	 * HttpResponseオブジェクトからシリアライズオブジェクトを取り出すメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * HttpResponse#getContent()の返却値を入力ストリームを使用してオブジェクト化します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param response HTTP応答オブジェクト
	 * @return HTTP応答オブジェクトから取り出されたシリアライズオブジェクト
	 * @throws IOException
	 * @throws ClassNotFoundException
	 */
	protected Object getSerializedHttpResponseContent(HttpResponse response)
			throws IOException, ClassNotFoundException {
		Object ret = null;

		ObjectInputStream objectInputStream = new ObjectInputStream(
				new ByteArrayInputStream(response.getContent()));
		ret = objectInputStream.readObject();

		return ret;
	}

	/**
	 * <p>[概 要] Element固有のデータソースをデータコンテナに生成します.</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考] </p>
	 * @param element 対象エレメント
	 * @return オブジェクトデータソース
	 * @throws Exception
	 */
	protected ObjectDataSource createElementObjectDataSource(Element element)
			throws Exception {

		//データソースインスタンス生成
		ObjectDataSource ods = new ObjectDataSource();

		//データソース名を準備
		String dataProviderId = element.getAttribute("id") + "_dataProvider";

		ods.initialize(dataProviderId, getController().getSession(), element);
		ods.setSource(Collections.EMPTY_LIST);
		getDataSourceContainer().addDataSource(dataProviderId, ods);

		element.setAttribute("dataProvider", dataProviderId);

		return ods;

	}

	/**
	 * <p>[概 要] Element固有のデータソースを取得します.</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考] </p>
	 * @param element 対象エレメント
	 * @return オブジェクトデータソース
	 */
	protected ObjectDataSource getElementObjectDataSource(Element element) {

		//データソース名を準備
		String dataProviderId = element.getAttribute("id") + "_dataProvider";

		ObjectDataSource dataSource = (ObjectDataSource) getDataSourceContainer()
				.getDataSource(dataProviderId);

		return dataSource;

	}

	/**
	 * <p>[概 要] Element固有のデータソースをデータコンテナーからRemoveします.</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考]</p>
	 * @param element 対象エレメント
	 */
	protected void removeElementObjectDataSource(Element element) {

		//データソース名を準備
		String dataProviderId = element.getAttribute("id") + "_dataProvider";

		getDataSourceContainer().removeDataSource(dataProviderId);
	}

	/**
	 * <p>[概 要] </p>
	 * xal内で定義されたObjectDataSourceを取得するメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param id
	 * @return
	 */
	protected ObjectDataSource getObjectDataSourceById(String id) {
	
		return (ObjectDataSource) getDataService().getDataSourceContainer()
				.getDataSource(id);
	}

	/**
	 * <p>[概 要] </p>
	 * xal内で定義されたObjectDataSourceを取得するメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param name
	 * @return
	 */
	protected ObjectDataSource getObjectDataSourceByName(String name) {
		ObjectDataSource ret = null;

		Element datasourceElement = getElementByNameFromOwnWindow(name);
		String id = datasourceElement.getAttribute("id");
		ret = getObjectDataSourceById(id);

		return ret;
	}

	/**
	 * <p>[概 要] データサービスを取得します。</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考] </p>
	 * @return データサービス
	 */
	private DataService getDataService() {
		return DataFramework.getDataService(getController().getSession());
	}

	/**
	 * <p>[概 要] データコンテナを取得します。</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考] </p>
	 * @return データコンテナ
	 */
	private DataSourceContainer getDataSourceContainer() {
		return getDataService().getDataSourceContainer();
	}

	/**
	 * <p>[概 要] </p>
	 * HttpResponseを画面にレンダリングするメソッドです。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * HttpResponseのコンテントはxalである必要が有ります。
	 * 
	 * @param response リモートサーバから取得したHttpResponseオブジェクト
	 */
	public void renderResponseToUI(HttpResponse response)
			throws ParserException {
		// UIに受信結果（XAL）をレンダリング
		getController().getSession().processXml(response);
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * コントローラに保持されているデータ保持領域Mapインスタンスを取得します。
	 * 
	 * <p>[備 考] </p>
	 * このデータ保持領域は、コントローラの初期化時に生成され、
	 * アプリケーション終了時まで保持されます。
	 * 
	 * @return アプリ起動～終了まで存在するデータ保存領域
	 */
	public Map<Object, Object> getPermanent() {
		return getController().getPermanent();
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域から引数keyに対応する値を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * コントローラに保持されているデータ保持領域Mapから、
	 * 引数keyをキーにして値を取得、返却します。
	 * 
	 * <p>[備 考] </p>
	 * このデータ保持領域は、コントローラの初期化時に生成され、
	 * アプリケーション終了時まで保持されます。
	 * 
	 * @param key データ保存領域Map内のキー
	 * @return 引数keyに対する値
	 */
	public Object getPermanent(Object key) {
		return getController().getPermanent().get(key);
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域にkey=valueの形式で値を追加します。
	 * 
	 * <p>[詳 細] </p>
	 * コントローラに保持されているデータ保持領域Mapを取得、
	 * 引数keyをキーにして引数valueを値として追加します。
	 *  
	 * <p>[備 考] </p>
	 * このデータ保持領域は、コントローラの初期化時に生成され、
	 * アプリケーション終了時まで保持されます。
	 * 
	 * @param key データ保存領域Map内のキー
	 * @param value 引数keyに対する値
	 */
	public void addPermanent(Object key, Object value) {
		getController().getPermanent().put(key, value);
	}

	/**
	 * <p>[概 要] </p>
	 * アプリ起動～終了まで存在するデータ保存領域から引数keyに対応する値を削除します。
	 * 
	 * <p>[詳 細] </p>
	 * コントローラに保持されているデータ保持領域Mapから、
	 * 引数keyをキーにして値を削除します。
	 * 
	 * <p>[備 考] </p>
	 * このデータ保持領域は、コントローラの初期化時に生成され、
	 * アプリケーション終了時まで保持されます。
	 * 
	 * @param key データ保存領域Map内のキー
	 */
	public void removePermanent(Object key) {
		getController().getPermanent().remove(key);
	}

	/**
	 * <p>[概 要] </p>
	 * 現在購読中のトピック名リストを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * トピック購読を管理しているSubscriberManagerシングルトンインスタンスから、
	 * getSubscribingTopicNames()メソッドの戻り値を受け取り、返却します。
	 * 
	 * <p>[備 考] </p>
	 * {@link SubscribeManager#getSubscribingTopicNames()}
	 * 
	 * @return 購読中のトピック名リスト
	 */
	protected List<String> getSubscribingTopicNames() {
		return SubscribeManager.instance.getSubscribingTopicNames();
	}

	/**
	 * <p>[概 要] </p>
	 * 引数で指定されたトピックを現在購読中かどうか調べます。
	 * 
	 * <p>[詳 細] </p>
	 * トピック購読を管理しているSubscriberManagerシングルトンインスタンスから、
	 * isSubscribing(topicName)メソッドの戻り値を受け取り、返却します。
	 *  
	 * <p>[備 考] </p>
	 * {@link SubscribeManager#isSubscribing(String)}
	 * 
	 * @param topicName トピック名
	 * @return true : 購読中、 false : 非購読中
	 */
	protected boolean isSubscribing(String topicName) {
		return SubscribeManager.instance.isSubscribing(topicName);
	}

	/**
	 * <p>[概 要] </p>
	 * クライアント側のログ発行オブジェクトを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * コントローラが保持するクライアントログ発行オブジェクトを取得して返却します。<br>
	 * 
	 * <p>[備 考] </p>
	 * 以下のように使用します。
	 * <pre class="samplecode">
	 * 	getClientLogger().log(Log.DEBUG, "データ登録処理終了");
	 * </pre>
	 * 
	 * ログ発行イベントをハンドルするLogConsumerインスタンスの種類は、
	 * ClientConfigオブジェクト内の以下の設定によって変更可能です。<p>
	 * 
	 * 【ClientConfig#isEnableLogConsumer()がtrue】<br>
	 * ClientConfig#getLogConsumerClass()で設定されているLogConsumer実装クラスがログイベントをハンドリングします。<p>
	 * 
	 * 【ClientConfig#isEnableLogConsumer()がfalse】<br>
	 * ClientConfig#getLogConsumerClass()は判断されず、標準出力に出力されます。<p>
	 * 
	 * デフォルトではClientConfig#isEnableLogConsumer()はfalseです。
	 * 
	 * @return クライアントログ発行オブジェクト
	 */
	public Log getClientLogger() {
		return getController().getClientLogger();
	}

	/**
	 * <p>[概 要] </p>
	 * 警告ダイアログ表示をします。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param title ダイアログタイトル
	 * @param message メッセージ
	 */
	protected void alertDialog(String title, String message) {
		getController().getSession().getDisplayService().modalAlert(message,
				title, DisplayService.ALERT_ERROR, new String[] { "OK" },
				new String[] { "OK" });
	}

}
