/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.controller;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.co.fujitsu.reffi.client.nexaweb.logconsumer.SystemOutLogConsumer;

import com.nexaweb.util.LogConsumer;

/**
 * <p>[概 要] </p>
 * フレームワーク挙動情報が格納されたクラスです。
 * 
 * <p>[詳 細] </p>
 * 
 * 
 * <p>[備 考] </p>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi 
 */
public class ClientConfig {

	// バリデーション関連

	/** フレームワークによるデフォルトエラー処理を行うかどうかのフラグです。 */
	private boolean enableValidationFaultProcessing = true;

	/** コンポーネント背景色の変更、ツールチップのエラー文言化を行うかどうかのフラグです。 */
	private boolean componentColorAndTipChangeOnValidationFault = true;

	/** エラー文言をダイアログ表示するかどうかのフラグです。 */
	private boolean displayDialogOnValidationFault = false;

	/** バリデーションエラー発生時の、コンポーネント背景色を指定します。 */
	private String componentColorOnValidationFault = "orange";

	/** 画面上に表示するバリデーションエラー数を指定します。 */
	private int maxDisplayCountOnValidationFault = 10;

	// ログコンシューマ関連

	/** ログイベントを受信するコンシューマクラスです。 */
	private Class<? extends LogConsumer> logConsumerClass = SystemOutLogConsumer.class;

	/** ログヘッダの日付、時間フォーマットです。 */
	private String logDateFormat = "HH:mm:ss.SSS";

	/** ログファイルを格納するディレクトリ名です。 */
	private String logFileDir = "reffi-client-log";

	/** ログファイル名の接頭文字です。 */
	private String logFileNamePrefix = "reffi-client-";

	/** ログファイル名の接尾文字です。 */
	private String logFileNameSuffix = ".log";

	/** ディレクトリ内ログファイルの格納数上限です。 */
	private int logFileMaxExist = 100;

	/** ログファイルサイズ上限です。(デフォルト：10485760（10MByte）) */
	private long logFileMaxSize = 10485760;

	/** ログファイルサイズローテーション時の最大バックアップファイル保持数です。 */
	private int logFileMaxBackupIndex = 10;

	/** ログファイルの文字エンコーディングです。 */
	private String logFileCharset = "UTF-8";

	//エラー処理

	/** 予期せぬ例外発生時に表示するダイアログタイトルです。 */
	private String unexpectedErrorDialogTitle = "error";

	/** 予期せぬ例外発生時に表示するダイアログメッセージです。 */
	private String unexpectedErrorDialogMessage = "unexpected exception occured.";

	// その他

	/** ui DOMにwindowエレメントが追加された時、focused="true"属性を自動で付与するかどうかのフラグです。 */
	private boolean windowElementDefaultFocus = true;

	/** フレームワークが「ウィンドウ」として認識するエレメント名のリストです。 */
	private List<String> windowLevelElementDefinition;

	/** 同一エレメントからのアクションを、同時に複数起動するかどうかのフラグです。 */
	private boolean duplicateActionInvoke = false;
	
	/** 多重起動許可エレメントとイベントの組み合わせを管理するマップです。 */
	private Map<String, List<String>> duplicateElementAndEventMap;

	/**
	 * <p>[概 要] </p>
	 * フレームワークによるデフォルトエラー処理を行うかどうかのフラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * enableValidationFaultProcessingフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return フレームワークによるデフォルトエラー処理を行うかどうかのフラグ
	 */
	public boolean isEnableValidationFaultProcessing() {
		return enableValidationFaultProcessing;
	}

	/**
	 * <p>[概 要] </p>
	 * フレームワークによるデフォルトエラー処理を行うかどうかのフラグを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * EnableValidationFaultProcessingフィールドを引数EnableValidationFaultProcessingで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param enableValidationFaultProcessing フレームワークによるデフォルトエラー処理を行うかどうかのフラグ
	 */
	public void setEnableValidationFaultProcessing(
			boolean enableValidationFaultProcessing) {
		this.enableValidationFaultProcessing = enableValidationFaultProcessing;
	}

	/**
	 * <p>[概 要] </p>
	 * コンポーネント背景色の変更、ツールチップのエラー文言化を行うかどうかのフラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * componentColorAndTipChangeOnValidationFaultフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return コンポーネント背景色の変更、ツールチップのエラー文言化を行うかどうかのフラグ
	 */
	public boolean isComponentColorAndTipChangeOnValidationFault() {
		return componentColorAndTipChangeOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * コンポーネント背景色の変更、ツールチップのエラー文言化を行うかどうかのフラグ
	 * 
	 * <p>[詳 細] </p>
	 * componentColorAndTipChangeOnValidationFaultフィールドを引数ComponentColorAndTipChangeOnValidationFaultで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param componentColorAndTipChangeOnValidationFault コンポーネント背景色の変更、ツールチップのエラー文言化を行うかどうかのフラグ
	 */
	public void setComponentColorAndTipChangeOnValidationFault(
			boolean componentColorAndTipChangeOnValidationFault) {
		this.componentColorAndTipChangeOnValidationFault = componentColorAndTipChangeOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * エラー文言をダイアログ表示するかどうかのフラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * displayDialogOnValidationFaultフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return エラー文言をダイアログ表示するかどうかのフラグ
	 */
	public boolean isDisplayDialogOnValidationFault() {
		return displayDialogOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * エラー文言をダイアログ表示するかどうかのフラグを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * displayDialogOnValidationFaultフィールドを引数displayDialogOnValidationFaultで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param displayDialogOnValidationFault エラー文言をダイアログ表示するかどうかのフラグ
	 */
	public void setDisplayDialogOnValidationFault(
			boolean displayDialogOnValidationFault) {
		this.displayDialogOnValidationFault = displayDialogOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * バリデーションエラー発生時の、コンポーネント背景色を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * componentColorOnValidationFaultフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return バリデーションエラー発生時の、コンポーネント背景色
	 */
	public String getComponentColorOnValidationFault() {
		return componentColorOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * バリデーションエラー発生時の、コンポーネント背景色を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * ComponentColorOnValidationFaultフィールドを引数ComponentColorOnValidationFaultで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param componentColorOnValidationFault バリデーションエラー発生時の、コンポーネント背景色
	 */
	public void setComponentColorOnValidationFault(
			String componentColorOnValidationFault) {
		this.componentColorOnValidationFault = componentColorOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * 画面上に表示するバリデーションエラー数を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * maxDisplayCountOnValidationFaultフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 画面上に表示するバリデーションエラー数
	 */
	public int getMaxDisplayCountOnValidationFault() {
		return maxDisplayCountOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * 画面上に表示するバリデーションエラー数を設定します。
	 * 
	 * <p>[詳 細] </p>
	 *  maxDisplayCountOnValidationFaulｔフィールドを引数 maxDisplayCountOnValidationFaulｔで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param  maxDisplayCountOnValidationFault 画面上に表示するバリデーションエラー数
	 */
	public void setMaxDisplayCountOnValidationFault(
			int maxDisplayCountOnValidationFault) {
		this.maxDisplayCountOnValidationFault = maxDisplayCountOnValidationFault;
	}

	/**
	 * <p>[概 要] </p>
	 * ログイベントを受信するコンシューマクラスを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logConsumerClassフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログイベントを受信するコンシューマクラス
	 */
	public Class<? extends LogConsumer> getLogConsumerClass() {
		return logConsumerClass;
	}

	/**
	 * <p>[概 要] </p>
	 * ログイベントを受信するコンシューマクラスを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logConsumerClassフィールドを引数logConsumerClassで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logConsumerClass ログイベントを受信するコンシューマクラスです。
	 */
	public void setLogConsumerClass(
			Class<? extends LogConsumer> logConsumerClass) {
		this.logConsumerClass = logConsumerClass;
	}

	/**
	 * <p>[概 要] </p>
	 * ログヘッダの日付、時間フォーマットを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logDateFormatフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログヘッダの日付、時間フォーマット
	 */
	public String getLogDateFormat() {
		return logDateFormat;
	}

	/**
	 * <p>[概 要] </p>
	 * ログヘッダの日付、時間フォーマットを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logDateFormatフィールドを引数logDateFormatで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logDateFormat ログヘッダの日付、時間フォーマット
	 */
	public void setLogDateFormat(String logDateFormat) {
		this.logDateFormat = logDateFormat;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルを格納するディレクトリ名を取得します。
	 *  
	 * <p>[詳 細] </p>
	 * logFileDirフィールドを返却します。
	 * 
	 * <p>[備 考] </p>

	 * @return ログファイルを格納するディレクトリ名
	 */
	public String getLogFileDir() {
		return logFileDir;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルを格納するディレクトリ名を設定します。
	 *  
	 * <p>[詳 細] </p>
	 * logFileDirフィールドを引数logFileDirで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFileDir ログファイルを格納するディレクトリ名
	 */
	public void setLogFileDir(String logFileDir) {
		this.logFileDir = logFileDir;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の接頭文字を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileNamePrefixフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイル名の接頭文字
	 */
	public String getLogFileNamePrefix() {
		return logFileNamePrefix;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の接頭文字を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileNamePrefixフィールドを引数logFileNamePrefixで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFileNamePrefix ログファイル名の接頭文字
	 */
	public void setLogFileNamePrefix(String logFileNamePrefix) {
		this.logFileNamePrefix = logFileNamePrefix;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の接尾文字を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileNameSuffixフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイル名の接尾文字
	 */
	public String getLogFileNameSuffix() {
		return logFileNameSuffix;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイル名の接尾文字を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileNameSuffixフィールドを引数logFileNameSuffixで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFileNameSuffix ログファイル名の接尾文字
	 */
	public void setLogFileNameSuffix(String logFileNameSuffix) {
		this.logFileNameSuffix = logFileNameSuffix;
	}

	/**
	 * <p>[概 要] </p>
	 * ディレクトリ内ログファイルの格納数上限を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileMaxExistフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ディレクトリ内ログファイルの格納数上限
	 */
	public int getLogFileMaxExist() {
		return logFileMaxExist;
	}

	/**
	 * <p>[概 要] </p>
	 * ディレクトリ内ログファイルの格納数上限を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileMaxExistフィールドを引数logFileMaxExistで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFileMaxExist ディレクトリ内ログファイルの格納数上限
	 */
	public void setLogFileMaxExist(int logFileMaxExist) {
		this.logFileMaxExist = logFileMaxExist;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルサイズ上限を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileMaxSizeフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイルサイズ上限
	 */
	public long getLogFileMaxSize() {
		return logFileMaxSize;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルサイズ上限を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileMaxSizeフィールドを引数logFileMaxSizeで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFileMaxSize ログファイルサイズ上限
	 */
	public void setLogFileMaxSize(long logFileMaxSize) {
		this.logFileMaxSize = logFileMaxSize;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルサイズローテーション時の最大バックアップファイル保持数を取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileMaxBackupIndexフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイルサイズローテーション時の最大バックアップファイル保持数
	 */
	public int getLogFileMaxBackupIndex() {
		return logFileMaxBackupIndex;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルサイズローテーション時の最大バックアップファイル保持数を設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileMaxBackupIndexフィールドを引数logFileMaxBackupIndexで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFileMaxBackupIndex ログファイルサイズローテーション時の最大バックアップファイル保持数
	 */
	public void setLogFileMaxBackupIndex(int logFileMaxBackupIndex) {
		this.logFileMaxBackupIndex = logFileMaxBackupIndex;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルの文字エンコーディングを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileCharsetフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ログファイルの文字エンコーディング
	 */
	public String getLogFileCharset() {
		return logFileCharset;
	}

	/**
	 * <p>[概 要] </p>
	 * ログファイルの文字エンコーディングを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * logFileCharsetフィールドを引数logFileCharsetで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param logFileCharset ログファイルの文字エンコーディング
	 */
	public void setLogFileCharset(String logFileCharset) {
		this.logFileCharset = logFileCharset;
	}

	/**
	 * <p>[概 要] </p>
	 * 予期せぬ例外発生時に表示するダイアログタイトルを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * unexpectedErrorDialogTitleフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 予期せぬ例外発生時に表示するダイアログタイトル
	 */
	public String getUnexpectedErrorDialogTitle() {
		return unexpectedErrorDialogTitle;
	}

	/**
	 * <p>[概 要] </p>
	 * 予期せぬ例外発生時に表示するダイアログタイトルを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * unexpectedErrorDialogTitleフィールドを引数unexpectedErrorDialogTitleで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param unexpectedErrorDialogTitle 予期せぬ例外発生時に表示するダイアログタイトル
	 */
	public void setUnexpectedErrorDialogTitle(String unexpectedErrorDialogTitle) {
		this.unexpectedErrorDialogTitle = unexpectedErrorDialogTitle;
	}

	/**
	 * <p>[概 要] </p>
	 * 予期せぬ例外発生時に表示するダイアログメッセージを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * unexpectedErrorDialogMessageフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 予期せぬ例外発生時に表示するダイアログメッセージ
	 */
	public String getUnexpectedErrorDialogMessage() {
		return unexpectedErrorDialogMessage;
	}

	/**
	 * <p>[概 要] </p>
	 * 予期せぬ例外発生時に表示するダイアログメッセージを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * unexpectedErrorDialogMessageフィールドを引数unexpectedErrorDialogMessageで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param unexpectedErrorDialogTitle 予期せぬ例外発生時に表示するダイアログメッセージ
	 */
	public void setUnexpectedErrorDialogMessage(
			String unexpectedErrorDialogMessage) {
		this.unexpectedErrorDialogMessage = unexpectedErrorDialogMessage;
	}

	/**
	 * <p>[概 要] </p>
	 * ui DOMにwindowエレメントが追加された時、focused="true"属性を自動で付与するかどうかのフラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * windowElementDefaultFocusフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return ui DOMにwindowエレメントが追加された時、focused="true"属性を自動で付与するかどうかのフラグ
	 */
	public boolean isWindowElementDefaultFocus() {
		return windowElementDefaultFocus;
	}

	/**
	 * <p>[概 要] </p>
	 * ui DOMにwindowエレメントが追加された時、focused="true"属性を自動で付与するかどうかのフラグを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * WindowElementDefaultFocusフィールドを引数WindowElementDefaultFocusで設定します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param windowElementDefaultFocus ui DOMにwindowエレメントが追加された時、focused="true"属性を自動で付与するかどうかのフラグ
	 */
	public void setWindowElementDefaultFocus(boolean windowElementDefaultFocus) {
		this.windowElementDefaultFocus = windowElementDefaultFocus;
	}

	/**
	 * <p>[概 要] </p>
	 * フレームワークが「ウィンドウ」として認識するエレメント名のリストを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * windowLevelElementDefinitionフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * フレームワークが「ウィンドウ」として認識するエレメントは、
	 * デフォルトでは「window」エレメントと「dialog」エレメントです。<p>
	 * 
	 * このリストに入っている名前をローカル名として持つエレメントには、
	 * parentId属性とcommunicateId属性が追加されます。<br>
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.model.XalReturnPossibilityModel#addIdentifierToWindows}<br>
	 * 又、{@link jp.co.fujitsu.reffi.client.nexaweb.action.BaseAction#getAllWindows()}等のウィンドウ
	 * 管理APIの対象エレメントになります。
	 * 
	 * @return フレームワークが「ウィンドウ」として認識するエレメント名のリスト
	 */
	public List<String> getWindowLevelElementDefinition() {
		return windowLevelElementDefinition;
	}

	/**
	 * <p>[概 要] </p>
	 * フレームワークが「ウィンドウ」として認識するエレメント名のリストを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * windowLevelElementDefinitionフィールドに引数windowLevelElementDefinitionを設定します。
	 * 
	 * <p>[備 考] </p>
	 * フレームワークが「ウィンドウ」として認識するエレメントは、
	 * デフォルトでは「window」エレメントと「dialog」エレメントです。<p>
	 * 
	 * このリストに入っている名前をローカル名として持つエレメントには、
	 * parentId属性とcommunicateId属性が追加されます。<br>
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.model.XalReturnPossibilityModel#addIdentifierToWindows}<br>
	 * 又、{@link jp.co.fujitsu.reffi.client.nexaweb.action.BaseAction#getAllWindows()}等のウィンドウ
	 * 管理APIの対象エレメントになります。
	 * 
	 * @param windowLevelElementDefinition フレームワークが「ウィンドウ」として認識するエレメント名のリスト
	 */
	public void setWindowLevelElementDefinition(
			List<String> windowLevelElementDefinition) {
		this.windowLevelElementDefinition = windowLevelElementDefinition;
	}

	/**
	 * <p>[概 要] </p>
	 * フレームワークが「ウィンドウ」として認識するエレメント名のリストに追加します。
	 * 
	 * <p>[詳 細] </p>
	 * フレームワークがウィンドウとして対処するAPIの対象エレメントローカル名を追加します。<br>
	 * window、dialogエレメントの他、ウィンドウとして認識させたいエレメント名を追加します。
	 * 
	 * <p>[備 考] </p>
	 * フレームワークが「ウィンドウ」として認識するエレメントは、
	 * デフォルトでは「window」エレメントと「dialog」エレメントです。<p>
	 * 
	 * このリストに入っている名前をローカル名として持つエレメントには、
	 * parentId属性とcommunicateId属性が追加されます。<br>
	 * {@link jp.co.fujitsu.reffi.client.nexaweb.model.XalReturnPossibilityModel#addIdentifierToWindows}<br>
	 * 又、{@link jp.co.fujitsu.reffi.client.nexaweb.action.BaseAction#getAllWindows()}等のウィンドウ
	 * 管理APIの対象エレメントになります。
	 * 
	 * @param elementName ウィンドウとして認識させるエレメント名
	 */
	public void addWindowLevelElementDefinition(String elementName) {
		getWindowLevelElementDefinition().add(elementName);
	}

	/**
	 * <p>[概 要] </p>
	 * 同一エレメントからのアクションを、同時に複数起動するかどうかのフラグを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * duplicateActionInvokeフィールドを返却します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 同一エレメントからのアクションを、同時に複数起動するかどうかのフラグ
	 */
	public boolean isDuplicateActionInvoke() {
		return duplicateActionInvoke;
	}

	/**
	 * <p>[概 要] </p>
	 * 同一エレメントからのアクションを、同時に複数起動するかどうかのフラグを設定します。
	 * 
	 * <p>[詳 細] </p>
	 * duplicateActionInvokeフィールドを引数duplicateActionInvokeで設定します。
	 * 
	 * <p>[備 考] </p>
	 * true : 同時複数起動可。<br>
	 * false : 不可。二度押し制御が可能になります。<br>
	 * デフォルトはfalseです。
	 * 
	 * @param duplicateActionInvoke 同一エレメントからのアクションを、同時に複数起動するかどうかのフラグ
	 */
	public void setDuplicateActionInvoke(boolean duplicateActionInvoke) {
		this.duplicateActionInvoke = duplicateActionInvoke;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 指定したエレメントとイベントの組み合わせでアクション多重起動を許可するように設定をします。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * イベント種別には「onCommand」等、コントローラで定義するイベント名と同一のものを指定してください。
	 * 
	 * @param elementName アクション多重起動を許可するエレメント名
	 * @param eventType アクション多重起動を許可するイベント種別
	 */
	public void addDuplicateElementAndEvent(String elementName, String eventType) {
		List<String> eventTypeList = null;
		if (this.duplicateElementAndEventMap.containsKey(elementName)) {
			eventTypeList = this.duplicateElementAndEventMap.get(elementName);
		} else {
			eventTypeList = new ArrayList<String>();
		}
		eventTypeList.add(eventType);
		this.duplicateElementAndEventMap.put(elementName, eventTypeList);
	}
	
	/**
	 * <p>[概 要] </p>
	 * 指定したエレメントとイベントの組み合わせでアクション多重起動が許可されているか取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @param elementName エレメント名
	 * @param eventType イベント種別
	 */
	public boolean isDuplicateElementAndEvent(String elementName, String eventType) {
		if (!this.duplicateElementAndEventMap.containsKey(elementName)) {
			return false;
		}
		return this.duplicateElementAndEventMap.get(elementName).contains(eventType);
	}
	
	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * this.windowLevelElementDefinitionインスタンスを生成して、
	 * "window"と"dialog"を追加します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public ClientConfig() {
		this.windowLevelElementDefinition = new ArrayList<String>();
		addWindowLevelElementDefinition("window");
		addWindowLevelElementDefinition("dialog");
		this.duplicateElementAndEventMap = new HashMap<String, List<String>>();
	}
}
