/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.logconsumer;

import java.net.SocketException;
import java.util.Properties;

import jp.co.fujitsu.reffi.client.nexaweb.util.DialogUtil;
import jp.co.fujitsu.reffi.common.nexaweb.dialog.CoreDialog;
import jp.co.fujitsu.reffi.common.nexaweb.dialog.CoreDialog.MessageDialogType;
import jp.co.fujitsu.reffi.common.util.ResourceUtil;

import com.nexaweb.client.ClientSession;
import com.nexaweb.client.netservice.NetServiceException;
import com.nexaweb.util.Log;
import com.nexaweb.util.LogConsumer;
import com.nexaweb.util.LogEvent;
import com.nexaweb.xml.Document;
import com.nexaweb.xml.Element;

/**
 * <p>[概 要]</p>
 * Nexaweb内部エラーを出力するコダイアログ出力クラス
 * <p>[詳 細]</p>
 * nexaweb-client.xmlの以下に使用するLogConsumerクラスを設定。
 * <p>[備 考]</p>
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * <b>使用例)</b>
 * <PRE style="border:1px; border-style:solid; margin:5px; padding:5px">
 *  &lt;log-configuration default-log-level="exception"&gt;
 *    &lt;consumers&gt;
 *      &lt;consumer class="jp.co.fujitsu.reffi.client.log.consumer.DialogLogConsumer"/&gt;
 *    &lt;/consumers&gt;
 *  &lt;/log-configuration&gt;
 * </PRE>
 * 
 * @author Project Reffi
 */
public class DialogLogConsumer implements LogConsumer {
	/** エラーメッセージ取得ソースです。 */
	private final String MESSAGE_RESOURCE = "jp.co.fujitsu.reffi.common.exception.corelogic_message";
	/** ダイアログエレメント */
	private Element _dialog;
	/** クライアントセッション */
	private ClientSession _session;

	/**
	 * <p>[概 要]</p>
	 * 初期化処理
	 * 
	 * <p>[詳 細]</p>
	 * ダイアログの初期化を行います。
	 * 本クラスはMCOとなるため、本クラスが「dialogLogConsumer」として登録されます。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param obj オブジェクト
	 * @see com.nexaweb.util.LogConsumer#init(java.lang.Object)
	 */
	@Override
	public void init(Object obj) {
		if (obj instanceof ClientSession) {
			_session = (ClientSession) obj;
			try {
				_session.getMcoContainer().addMco("dialogLogConsumer", this);
			} catch (Exception exception) {
			}
		}
	}

	/**
	 * <p>[概 要]</p>
	 * ログ出力処理
	 * 
	 * <p>[詳 細]</p>
	 * ログイベントからログを取得、出力します。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @param logevent　ログイベント
	 * @see com.nexaweb.util.LogConsumer#log(com.nexaweb.util.LogEvent)
	 */
	@Override
	public void log(LogEvent logevent) {
		logevent.getThrowable();
		Throwable e = logevent.getThrowable();
		if (logevent.getType() == Log.ERROR
				|| logevent.getType() == Log.EXCEPTION) {
			if (recursiveCheckConnectError(e)) {
				addMessage(logevent.toString());
			}
		}
	}

	/**
	 * <p>[概 要] 発生した例外が原因がサーバ接続エラーかを判定します.</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考] </p>
	 * @param e 発生例外
	 * @return 判定結果 true:サーバ接続エラー false:その他のエラー
	 */
	private boolean recursiveCheckConnectError(Throwable e) {

		boolean result = false;
		if (e instanceof SocketException) {
			result = true;
		} else if (e instanceof NetServiceException
				&& ((NetServiceException) e).getCausalThrowable() != null) {
			result = recursiveCheckConnectError(((NetServiceException) e)
					.getCausalThrowable());
		} else if (e.getCause() != null) {
			result = recursiveCheckConnectError(e.getCause());
		}
		return result;
	}

	/**
	 * <p>[概 要]</p>
	 * 終了処理
	 * 
	 * <p>[詳 細]</p>
	 * ログコンシューマの終了処理を行います。
	 * 
	 * <p>[備 考]</p>
	 * 
	 * @see com.nexaweb.util.LogConsumer#destroy()
	 */
	@Override
	public void destroy() {
	}

	/**
	 * <p>[概 要]</p>
	 * クローズ処理
	 * 
	 * <p>[詳 細]</p>
	 * ダイアログのクローズ処理を行います。
	 * クローズをした時点でコンシューマは破棄されません。
	 * 
	 * <p>[備 考]</p>
	 * 
	 */
	public void onClose() {
		synchronized (_session.getDocumentRegistry().getUiDocument()
				.getDomSynchronizationObject()) {
			_dialog.setAttribute("visible", "false");
		}
	}

	/**
	 * <p>[概 要] ダイアログにメッセージを設定します。</p>
	 * <p>[詳 細] </p>
	 * <p>[備 考] </p>
	 * @param message メッセージ
	 */
	private void addMessage(String message) {

		synchronized (_session.getDocumentRegistry().findDocument("ui")
				.getDomSynchronizationObject()) {

			Document uiDoc = _session.getDocumentRegistry().getUiDocument();
			Element rootPane = DialogUtil.getRootPane(uiDoc);

			Properties messages = ResourceUtil.instance
					.asProperties(MESSAGE_RESOURCE);

			String id = "EFC1009";
			String msg = messages.getProperty(id);
			CoreDialog cmd = new CoreDialog.Builder(id, msg).dialogType(
					MessageDialogType.ERROR.getType()).build();

			if (_dialog == null) {
				_dialog = DialogUtil.getElement(cmd);
			}

			rootPane.appendChild(_dialog);

		}
	}
}
