/* 
 * Copyright (c) 2008-2010, FUJITSU LIMITED
 * All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice, this
 *    list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation and/or
 *    other materials provided with the distribution.
 * 
 * 3. Redistributions with modification must carry prominent notices stating that you changed 
 *    the files and the date of any change.
 * 
 * 4. Neither the name of FUJITSU LIMITED nor the names of its contributors may be used
 *    to endorse or promote products derived from this software without specific prior
 *    written permission.
 * 
 * 5. All your rights under this license shall terminate automatically if you fail to
 *    comply  with any of this list of conditions. If your rights under this license terminate,
 *    you agree to cease use and distribution of this software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES;LOSS OF USE,DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package jp.co.fujitsu.reffi.client.nexaweb.util;

import java.util.HashMap;
import java.util.Map;

import com.nexaweb.xml.Element;

/**
 * <p>[概 要] </p>
 * エレメントのスナップショットを採取するクラスです。
 * 
 * <p>[詳 細] </p>
 * このクラスは「ある時点のエレメントの状態を取っておきたい」場合に使用します。<BR>
 * Reffi内部では、エラー状態から復帰したエレメントの、背景色復帰、tip文言復帰
 * を実行する為に使用されます。
 * 
 * <p>[備 考] </p>
 * 「スナップショット」はクローンされたエレメントオブジェクトの事を指します。<p>
 *
 * <b>使用例）</b><br>
 * <pre class="samplecode">
 *	ElementSnapshot snapshot = new ElementSnapshot();
 *	snapshot.snap(textFieldElement);
 *	// 各種処理 
 *	textFieldElement.setAttribute("text", "foo");
 *	textFieldElement.setAttribute("tip", "sample tip");
 *		:
 *	// 元データ
 *	Element textFieldElement_old = snapshot.fetch(textFieldElement);
 *	textFieldElement_old.getAttribute("text");	// not "foo"
 *	textFieldElement_old.getAttribute("tip");	// not "sample tip"
 * </pre>
 * 
 * <p>[環 境] JDK 6.0 Update 11</p>
 * <p>Copyright (c) 2008-2009 FUJITSU Japan All rights reserved.</p>
 * 
 * @author Project Reffi
 */
public class ElementSnapshot {
	
	/** 「element = element.clone(true)」の形式で記録するマップです。 */
	private Map<Element, Element> record;
	
	/**
	 * <p>[概 要] </p>
	 * 記録マップを返却します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 
	 * @return 「element = element.clone(true)」形式記録マップ
	 */
	public Map<Element, Element> getRecord(){
		return this.record;
	}
	
	/**
	 * <p>[概 要] </p>
	 * デフォルトコンストラクタです。
	 * 
	 * <p>[詳 細] </p>
	 * 記録マップオブジェクトを生成します。
	 * 
	 * <p>[備 考] </p>
	 * 
	 */
	public ElementSnapshot(){
		this.record = new HashMap<Element, Element>();
	}

	/**
	 * <p>[概 要] </p>
	 * スナップショットを採ります。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 既に記録されているエレメントはスナップショットが採られません。
	 * その際の戻り値はfalseになります。
	 * 
	 * @param element スナップショットを採るエレメント 
	 * @return true : スナップショット採取成功、 false : 既にスナップ済み
	 */
	public boolean snap(Element element){
		boolean ret = false;
		
		if(!this.record.containsKey(element)){
			this.record.put(element, element.clone(true));
			ret = true;
		}
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * スナップショットを採ります。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 既に記録されているエレメントであっても、上書きスナップショットが採られます。
	 * 
	 * @param element スナップショットを採るエレメント
	 */
	public void reSnap(Element element){
		this.record.put(element, element.clone(true));
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数elementで指定されたエレメントのスナップショットを取得します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 該当するエレメントが記録されていなかった場合はnullを返却します。
	 * 
	 * @param element 取得するスナップショットの現在のエレメント
	 * @return スナップショットエレメント
	 */
	public Element fetch(Element element){
		Element ret = null;
		
		if(element != null){
			ret = this.record.get(element);
		}
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 引数elementで指定されたエレメントのスナップショットを削除します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 * 該当するエレメントが記録されていなかった場合はfalseが返却されます。
	 * 
	 * @param element 削除するスナップショットの現在のエレメント
	 * @return true : 削除成功、 false : 削除失敗
	 */
	public boolean erase(Element element){
		boolean ret = false;
		
		if(this.record.containsKey(element)){
			this.record.remove(element);
			ret = true;
		}
		
		return ret;
	}
	
	/**
	 * <p>[概 要] </p>
	 * 記録されている全てのスナップショットを削除します。
	 * 
	 * <p>[詳 細] </p>
	 * 
	 * <p>[備 考] </p>
	 */
	public void erase(){
		this.record.clear();
	}
}
