unit UNsmPluginMain;

interface

uses
  Windows, SysUtils, UNsmConsts, UNsmTypes;

type
  TNsmPluginMain = class(TObject)
  private
    FInitInfo: TNsmPluginInitInfo;
    FInitialized: Boolean;
    FPluginInfo: array[NMPI_APIVER..NMPI_PLUGINVER] of String;
  protected
    procedure DoInitialize; virtual;
    procedure DoTerminate; virtual;
    procedure DoAllModulesLoaded; virtual;

    function GetPluginInfo(Idx: Integer): String; virtual;
    procedure SetPluginInfo(Idx: Integer; Value: String); virtual;
    function GetDllFileName: String; virtual;
    function GetWorkDir: String; virtual;
    function GetIniFileName: String; virtual;
  public
    constructor Create; virtual;
    destructor Destroy; override;
    procedure Initialize(AInitInfo: TNsmPluginInitInfo); virtual;
    procedure Terminate; virtual;

    function CreateService(ServiceName: String; ProcAddr: TNsmServiceProc): HNsmService;
    function GetService(ServiceName: String): HNsmService;
    function CallService(ServiceHandle: HNsmService; wParam, lParam: Cardinal): Integer; overload;
    function CallService(ServiceName: String; wParam, lParam: Cardinal): Integer; overload;
    function CreateEvent(EventName: String): HNsmEvent;
    function GetEvent(EventName: String): HNsmEvent;
    function NotifyEvent(EventHandle: HNsmEvent; wParam, lParam: Cardinal): Integer;
    function HookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;
    function UnhookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;

    property InitInfo: TNsmPluginInitInfo read FInitInfo;
    property Initialized: Boolean read FInitialized;
    property PluginInfo[Idx: Integer]: String read  GetPluginInfo write SetPluginInfo;
    property DllFileName: String read GetDllFileName;
    property WorkDir: String read GetWorkDir;
    property IniFileName: String read GetIniFileName;
  end;

var
  NsmPluginMain: TNsmPluginMain;

implementation

function OnAllModulesLoadeded(wParam, lParam: Cardinal): Integer; Stdcall; forward;

// TNsmPluginMain --------------------------------------------------------------

constructor TNsmPluginMain.Create;
begin
  if not Assigned(NsmPluginMain) then
    NsmPluginMain := Self;
  FInitialized := False;
  FPluginInfo[NMPI_APIVER] := NSM_API_VERSION;
end;

destructor TNsmPluginMain.Destroy;
begin
  if NsmPluginMain = Self then
    NsmPluginMain := nil;
  inherited;
end;

procedure TNsmPluginMain.Initialize(AInitInfo: TNsmPluginInitInfo);
begin
  if not FInitialized then
  begin
    FInitialized := True;
    FInitInfo := AInitInfo;
    DoInitialize;
  end;
end;

procedure TNsmPluginMain.Terminate;
begin
  if FInitialized then
  begin
    FInitialized := False;
    DoTerminate;
  end;
end;

procedure TNsmPluginMain.DoInitialize;
begin
  HookEvent(NME_SYSTEM_MODULESLOADED, OnAllModulesLoadeded);
end;

procedure TNsmPluginMain.DoTerminate;
begin
end;

procedure TNsmPluginMain.DoAllModulesLoaded;
begin
end;

function TNsmPluginMain.GetPluginInfo(Idx: Integer): String;
begin
  if Idx in [NMPI_APIVER..NMPI_PLUGINVER] then
    Result := FPluginInfo[Idx]
  else
    Result := '';
end;

procedure TNsmPluginMain.SetPluginInfo(Idx: Integer; Value: String);
begin
  if Idx in [NMPI_APIVER..NMPI_PLUGINVER] then
    FPluginInfo[Idx] := Value;
end;

function TNsmPluginMain.GetDllFileName: String;
var
  Buff: array[0..255] of Char;
begin
  GetModuleFileName(hInstance, Buff, SizeOf(Buff));
  Result := Buff;
end;

function TNsmPluginMain.GetWorkDir: String;
begin
  Result := ChangeFileExt(GetDllFileName, '') + '\';
end;

function TNsmPluginMain.GetIniFileName: String;
begin
  Result := GetWorkDir + ChangeFileExt(ExtractFileName(GetDllFileName), '.ini');
end;

function TNsmPluginMain.CreateService(ServiceName: String; ProcAddr: TNsmServiceProc): HNsmService;
begin
  Result := FInitInfo.CreateService(PChar(ServiceName), ProcAddr);
end;

function TNsmPluginMain.GetService(ServiceName: String): HNsmService;
begin
  Result := FInitInfo.GetService(PChar(ServiceName));
end;

function TNsmPluginMain.CallService(ServiceHandle: HNsmService; wParam, lParam: Cardinal): Integer;
begin
  Result := FInitInfo.CallService(ServiceHandle, wParam, lParam);
end;

function TNsmPluginMain.CallService(ServiceName: String; wParam, lParam: Cardinal): Integer;
var
  hService: HNsmService;
begin
  hService := GetService(ServiceName);
  if hService <> 0 then
    Result := FInitInfo.CallService(hService, wParam, lParam)
  else
    Result := 0;
end;

function TNsmPluginMain.CreateEvent(EventName: String): HNsmEvent;
begin
  Result := FInitInfo.CreateEvent(PChar(EventName));
end;

function TNsmPluginMain.GetEvent(EventName: String): HNsmEvent;
begin
  Result := FInitInfo.GetEvent(PChar(EventName));
end;

function TNsmPluginMain.NotifyEvent(EventHandle: HNsmEvent; wParam, lParam: Cardinal): Integer;
begin
  Result := FInitInfo.NotifyEvent(EventHandle, wParam, lParam);
end;

function TNsmPluginMain.HookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;
begin
  Result := FInitInfo.HookEvent(PChar(EventName), ProcAddr);
end;

function TNsmPluginMain.UnhookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;
begin
  Result := FInitInfo.UnhookEvent(PChar(EventName), ProcAddr);
end;

// -----------------------------------------------------------------------------
//  SẴW[[hꂽCxg
//  wParam  : 0
//  lParam  : 0
//  Return  : 0
function OnAllModulesLoadeded(wParam, lParam: Cardinal): Integer; Stdcall;
begin
  if Assigned(NsmPluginMain) then
    NsmPluginMain.DoAllModulesLoaded;
  Result := 0;
end;

end.
