unit ExRichEdit;
(* ExRichEdit *)

interface

uses
  Windows, Messages, SysUtils, Classes, Controls, ComCtrls, RichEdit, Menus,
  Clipbrd;

const
  ST_DEFAULT = 0;
  EM_SETTEXTEX = WM_USER + 97;

type
  TSetTextEx = record
    flags: Cardinal;
    codepage: Cardinal;
  end;

  TURLClickEvent = procedure (Sender: TObject; URL: WideString) of Object;
  TTextMode = (tmRichText, tmPlainText);
  TExRichEdit = class(TCustomRichEdit)
  private
    FLibHandle: THandle;
    FVer10: Boolean;
  	FOnURLClick: TURLClickEvent;
    FDefWndProcA: TFNWndProc;
    FTextMode: TTextMode;
    FAutoURLDetect: Boolean;

    FDefaultMenu: TPopupMenu;
    mnuUndo: TMenuItem;
    mnuCopy: TMenuItem;
    mnuCut: TMenuItem;
    mnuPaste: TMenuItem;
    mnuDelete: TMenuItem;
    mnuSelectAll: TMenuItem;
    mnuSepalator1: TMenuItem;
    mnuSepalator2: TMenuItem;

    procedure SetAutoURLDetect(Value: Boolean);
    procedure SetTextMode(Value: TTextMode);
    function GetWideText: WideString;
    procedure SetWideText(Value: WideString);

    procedure DefaultMenuPopup(Sender: TObject);
    procedure DefaultMenuClick(Sender: TObject);
  protected
    procedure CreateWnd; override;
    procedure CreateParams(var Params: TCreateParams);override;
    procedure ON_WM_NCDESTROY(var Message: TWMNCDestroy); message WM_NCDESTROY;
    procedure ON_CN_NOTIFY(var Message: TWMNotify); message CN_NOTIFY;
    procedure DoURLClick(URL: WideString);
    procedure DoContextPopup(MousePos: TPoint; var Handled: Boolean); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Ver10: Boolean read FVer10;
    property WideText: WideString read GetWideText write SetWideText;    
  published
	  property AutoURLDetect: Boolean read  FAutoURLDetect write  SetAutoURLDetect default False;
    property TextMode: TTextMode read FTextMode write SetTextMode default tmRichText;
    property OnURLClick: TURLClickEvent read  FOnURLClick write FOnURLClick;

    property Align;
    property Alignment;
    property Anchors;
    property BiDiMode;
    property BorderStyle;
    property BorderWidth;
    property BevelEdges;
		property BevelInner;
		property BevelKind;
		property BevelOuter;
		property BevelWidth;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property HideScrollBars;
    property ImeMode;
    property ImeName;
    property Constraints;
    property Lines;
    property MaxLength;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PlainText;
    property PopupMenu;
    property ReadOnly;
    property ScrollBars;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property WantTabs;
    property WantReturns;
    property WordWrap;
    property OnChange;
    property OnContextPopup;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnMouseWheel;
    property OnMouseWheelDown;
    property OnMouseWheelUp;
    property OnProtectChange;
    property OnResizeRequest;
    property OnSaveClipboard;
    property OnSelectionChange;
    property OnStartDock;
	  property OnStartDrag;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('NSM2', [TExRichEdit]);
end;

constructor TExRichEdit.Create(AOwner: TComponent);
begin
  inherited;
  FDefaultMenu := TPopupMenu.Create(Self);
  FDefaultMenu.OnPopup := DefaultMenuPopup;
  mnuUndo := TMenuItem.Create(Self);
  mnuCut := TMenuItem.Create(Self);
  mnuCopy := TMenuItem.Create(Self);
  mnuPaste := TMenuItem.Create(Self);
  mnuDelete := TMenuItem.Create(Self);
  mnuSelectAll := TMenuItem.Create(Self);
  mnuSepalator1 := TMenuItem.Create(Self);
  mnuSepalator2 := TMenuItem.Create(Self);
  mnuUndo.Caption := 'ɖ߂(&U)';
  mnuCut.Caption := '؂(&T)';
  mnuCopy.Caption := 'Rs[(&C)';
  mnuPaste.Caption := '\t(&P)';
  mnuDelete.Caption := '폜(&D)';
  mnuSelectAll.Caption := 'SđI(&A)';
  mnuSepalator1.Caption := '-';
  mnuSepalator2.Caption := '-';
  mnuUndo.OnClick := DefaultMenuClick;
  mnuCut.OnClick := DefaultMenuClick;
  mnuCopy.OnClick := DefaultMenuClick;
  mnuPaste.OnClick := DefaultMenuClick;
  mnuDelete.OnClick := DefaultMenuClick;
  mnuSelectAll.OnClick := DefaultMenuClick;
  FDefaultMenu.Items.Add(mnuUndo);
  FDefaultMenu.Items.Add(mnuSepalator1);
  FDefaultMenu.Items.Add(mnuCut);
  FDefaultMenu.Items.Add(mnuCopy);
  FDefaultMenu.Items.Add(mnuPaste);
  FDefaultMenu.Items.Add(mnuDelete);
  FDefaultMenu.Items.Add(mnuSepalator2);  
  FDefaultMenu.Items.Add(mnuSelectAll);
end;

destructor TExRichEdit.Destroy;
begin
  FDefaultMenu.Items.Clear;
  FDefaultMenu.Free;
  inherited;
end;

procedure TExRichEdit.CreateWnd;
begin
  inherited;
	Perform(EM_SETEVENTMASK, 0,
  	  ENM_CHANGE or ENM_SELCHANGE or ENM_REQUESTRESIZE or
	    ENM_PROTECTED or ENM_LINK);
  SetTextMode(FTextMode);
  SetAutoURLDetect(FAutoURLDetect);
end;

procedure TExRichEdit.CreateParams(var Params: TCreateParams);
const
  RichEditModuleName = 'RICHED20.DLL';
  ControlClassName = 'RICHEDIT20A';  // Ansi
//  ControlClassName = 'RICHEDIT20W';  // Unicode
  CS_OFF = CS_OWNDC or CS_CLASSDC or CS_PARENTDC or CS_GLOBALCLASS;
  CS_ON = CS_VREDRAW or CS_HREDRAW;
var
  OldError: Longint;
  WCA: TWndClassA;
begin
  OldError := SetErrorMode(SEM_NOOPENFILEERRORBOX);
  FLibHandle := LoadLibrary(RichEditModuleName);
  FVer10:= False;
  if (FLibHandle > 0) and (FLibHandle < HINSTANCE_ERROR) then
    FLibHandle := 0;
  if FLibHandle=0 then
  begin
    FVer10:= True;
    inherited CreateParams(Params);
    Exit;
  end;
  SetErrorMode(OldError);
  GetClassInfoA(HInstance, ControlClassName, WCA);
  FDefWndProcA:= WCA.lpfnWndProc;
  inherited CreateParams(Params);
  Params.Style:= Params.Style or WS_CHILD or WS_CLIPCHILDREN or WS_CLIPSIBLINGS or ES_SAVESEL;
  CreateSubClass(Params, ControlClassName);
end;

procedure TExRichEdit.ON_WM_NCDESTROY(var Message: TWMNCDestroy);
begin
  inherited;
  if FLibHandle <> 0 then
    FreeLibrary(FLibHandle);
end;

procedure TExRichEdit.SetAutoURLDetect(Value: Boolean);
begin
  FAutoURLDetect := Value;
  if FAutoURLDetect then
  	SendMessage(Handle, EM_AUTOURLDETECT, 1, 0)
  else
  	SendMessage(Handle, EM_AUTOURLDETECT, 0, 0);
end;

procedure TExRichEdit.SetTextMode(Value: TTextMode);
var
  Flags: Integer;
begin
  FTextMode := Value;
  Flags := TM_SINGLELEVELUNDO or TM_MULTICODEPAGE;
  case FTextMode of
  tmPlainText:
    SendMessage(Handle, EM_SETTEXTMODE, Flags or TM_PLAINTEXT, 0);
  tmRichText:
    SendMessage(Handle, EM_SETTEXTMODE, Flags or TM_RICHTEXT, 0);
  end;
end;

function TExRichEdit.GetWideText: WideString;
var
  GTL: TGetTextLengthEx;
  GT: TGetTextEx;
  L: Integer;
begin
  GTL.flags := GTL_DEFAULT;
  GTL.codepage := 1200;
  L := Perform(EM_GETTEXTLENGTHEX, Integer(@GTL), 0);
  SetLength(Result, L + 1);
  GT.cb := L * 2 + 2;
  GT.flags := GT_DEFAULT;
  GT.codepage := 1200;
  GT.lpDefaultChar := nil;
  GT.lpUsedDefChar := nil;
  Perform(EM_GETTEXTEX, Integer(@GT), Integer(@Result[1]));
end;

procedure TExRichEdit.SetWideText(Value: WideString);
var
  ST: TSetTextEx;
begin
  ST.flags := ST_DEFAULT;
  ST.codepage := 1200;
  Perform(EM_SETTEXTEX, Integer(@ST), Integer(PWideChar(Value)));
end;

procedure TExRichEdit.ON_CN_NOTIFY(var Message: TWMNotify);
type
  PENLink = ^TENLink;
var
  URL: WideString;
  AENLink: TENLink;
begin
	case (Message.NMHdr^.code) of
  EN_LINK:
  begin
    AENLink := PENLink(Pointer(Message.NMHdr))^;
    URL := GetWideText;
    URL := Copy(URL, AENLink.chrg.cpMin + 1, AENLink.chrg.cpMax - AENLink.chrg.cpMin);
    case AENLink.msg of
    WM_LBUTTONUP:
      if (Length(URL) > 1) then
        DoURLClick(URL);
    end;
  end else
    inherited;
	end;
end;

procedure TExRichEdit.DoURLClick(URL: WideString);
begin
  if Assigned(FOnURLClick) then
    FOnURLClick(Self, URL);
end;

procedure TExRichEdit.DoContextPopup(MousePos: TPoint; var Handled: Boolean);
var
  P: TPoint;
begin
  inherited;
  if not Assigned(PopupMenu) then
  begin
    P := ClientToScreen(MousePos);
    FDefaultMenu.Popup(P.X, P.Y);
  end;  
end;

procedure TExRichEdit.DefaultMenuClick(Sender: TObject);
begin
  inherited;
  if Sender = mnuUndo then
    Undo
  else if Sender  = mnuCut then
    CutToClipboard
  else if Sender  = mnuCopy then
    CopyToClipboard
  else if Sender  = mnuPaste then
    PasteFromClipboard
  else if Sender  = mnuDelete then
    ClearSelection
  else if Sender  = mnuSelectAll then
    SelectAll;
end;

procedure TExRichEdit.DefaultMenuPopup(Sender: TObject);
begin
  mnuUndo.Enabled := Focused and CanUndo;
  mnuCut.Enabled := Focused and (SelLength > 0) and (not ReadOnly);
  mnuCopy.Enabled := Focused and (SelLength > 0);
  mnuPaste.Enabled := Focused and (Clipboard.HasFormat(CF_TEXT)) and (not ReadOnly);
  mnuDelete.Enabled := Focused and (SelLength > 0) and (not ReadOnly);
end;

end.
