unit UBmpMngr;

interface

uses
  Graphics, SysUtils, IniFiles;

type
  TBitmapRec = class(TObject)
    Bitmap: TBitmap;
    RefCount: Integer;

    constructor Create(ABitmap: TBitmap);
    destructor Destroy; override;
    function IncRefCount: Integer;
    function DecRefCount: Integer;
  end;

  TBitmapManager = class(TObject)
  private
    FEmptyBitmap: TBitmap;
    FItems: THashedStringList;

    function GetCount: Integer;
    function GetItem(Idx: Integer): TBitmapRec;
    function Add(FileName: String; Bmp: TBitmap): Integer;
    function LoadAndAdd(FileName: String): Integer;
    procedure Delete(Idx: Integer);
    function IndexOf(FileName: String): Integer; overload;
    function IndexOf(Bmp: TBitmap): Integer; overload;
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    procedure LoadBitmap(var Dest: TBitmap; const FileName: String);
    procedure CopyBitmap(Src: TBitmap; var Dest: TBitmap);
    procedure ReleaseBitmap(Bmp: TBitmap);

    property Count: Integer read GetCount;
  end;

var
  BitmapManager: TBitmapManager;

implementation

// TBitmapRec ------------------------------------------------------------------

constructor TBitmapRec.Create(ABitmap: TBitmap);
begin
  Bitmap := ABitmap;
  RefCount := 0;
end;

destructor TBitmapRec.Destroy;
begin
  inherited;
end;

function TBitmapRec.IncRefCount: Integer;
begin
  Inc(RefCount);
  Result := RefCount;
end;

function TBitmapRec.DecRefCount: Integer;
begin
  Dec(RefCount);
  Result := RefCount;
end;

// TBitmapManager --------------------------------------------------------------

constructor TBitmapManager.Create;
begin
  FEmptyBitmap := TBitmap.Create;
  FItems := THashedStringList.Create;
  FItems.Sorted := True;
  FItems.CaseSensitive := False;
end;

destructor TBitmapManager.Destroy;
begin
  Clear;
  FItems.Free;
  FEmptyBitmap.Free;
  inherited;
end;

function TBitmapManager.GetCount: Integer;
begin
  Result := FItems.Count;
end;

function TBitmapManager.GetItem(Idx: Integer): TBitmapRec;
begin
  Result := TBitmapRec(FItems.Objects[Idx]);
end;

function TBitmapManager.Add(FileName: String; Bmp: TBitmap): Integer;
begin
  Result := FItems.AddObject(FileName, TBitmapRec.Create(Bmp));
end;

function TBitmapManager.LoadAndAdd(FileName: String): Integer;
var
  Bmp: TBitmap;
begin
  Result := IndexOf(FileName);
  if Result = -1 then
  begin
    if FileExists(FileName) then
    begin
      Bmp := TBitmap.Create;
      Bmp.LoadFromFile(FileName);
      Result := Add(FileName, Bmp);
    end else
      Result := -1;
  end;
end;

procedure TBitmapManager.Delete(Idx: Integer);
begin
  if GetItem(Idx).Bitmap <> FEmptyBitmap then
    GetItem(Idx).Bitmap.Free;
  GetItem(Idx).Free;
  FItems.Delete(Idx);
end;

procedure TBitmapManager.Clear;
begin
  while FItems.Count > 0 do
    Delete(0);
end;

function TBitmapManager.IndexOf(FileName: String): Integer;
begin
  Result := FItems.IndexOf(FileName);
end;

function TBitmapManager.IndexOf(Bmp: TBitmap): Integer;
var
  I: Integer;
begin
  Result := -1;
  for I := 0 to FItems.Count - 1 do
    if GetItem(I).Bitmap = Bmp then
    begin
      Result := I;
      Break;
    end;
end;

procedure TBitmapManager.LoadBitmap(var Dest: TBitmap; const FileName: String);
var
  Idx: Integer;
begin
  ReleaseBitmap(Dest);

  Idx := IndexOf(FileName);
  if Idx = -1 then
    Idx := LoadAndAdd(FileName);

  if Idx > -1 then
  begin
    Dest := GetItem(Idx).Bitmap;
    GetItem(Idx).IncRefCount;
  end else
    Dest := FEmptyBitmap;
end;

procedure TBitmapManager.CopyBitmap(Src: TBitmap; var Dest: TBitmap);
var
  Idx: Integer;
begin
  if Src = Dest then Exit;
  ReleaseBitmap(Dest);
  Dest := Src;
  Idx := IndexOf(Src);
  if (Idx > -1) then
    GetItem(Idx).IncRefCount;
end;

procedure TBitmapManager.ReleaseBitmap(Bmp: TBitmap);
var
  Idx: Integer;
begin
  if Bmp <> FEmptyBitmap then
  begin
    Idx := IndexOf(Bmp);
    if (Idx > -1) and (GetItem(Idx).DecRefCount <= 0) then
      Delete(Idx);
  end;
end;


initialization
  BitmapManager := TBitmapManager.Create;

finalization
  BitmapManager.Free;

end.
