unit USkinUtils;

interface

uses
  Windows, SysUtils, Types, Graphics, Math, USkinStyles, UWideGraphics;

const
  MAXSTYLELEVEL = 3;

function GetStyleClientRect(R: TRect; Style: TSkinStyle): TRect;
procedure AssignStyleToFont(AFont: TFont; Style: TSkinStyle);

procedure FillStyleBackground(Canvas: TCanvas; R: TRect; Style: TSkinStyle);
procedure DrawStyleBorder(Canvas: TCanvas; R: TRect; Style: TSkinStyle;
  Transparent: Boolean = False);
procedure DrawStyleText(Canvas: TCanvas; R: TRect; Style: TSkinStyle;
  Text: WideString);

function TraceBackStyleName(Element, ClassName, Status: String;
  Level: Integer): String;

implementation

const
  HighlightColor = clBtnHighlight;
  ShadowColor = clBtnShadow;

function GetStyleClientRect(R: TRect; Style: TSkinStyle): TRect;
begin
  Result := R;
  if Style.BorderStyle = sbImage then
  begin
    Inc(Result.Top, Style.TopImage.Height);
    Inc(Result.Left, Style.LeftImage.Width);
    Dec(Result.Right, Style.RightImage.Width);
    Dec(Result.Bottom, Style.BottomImage.Height);
  end else if Style.BorderStyle <> sbNone then
    InflateRect(Result, -Style.BorderWidth, -Style.BorderWidth);
end;

procedure AssignStyleToFont(AFont: TFont; Style: TSkinStyle);
begin
  if Style.FontFamily <> '' then
    AFont.Name := Style.FontFamily;
  if Style.FontColor <> clNone then
    AFont.Color := Style.FontColor;
  if Style.FontSize >= 0 then
    AFont.Size := Style.FontSize;
  AFont.Style := Style.FontStyle;
end;

procedure TileBitmap(Canvas: TCanvas; R: TRect; Bmp: TBitmap;
  Mode: TSkinBackgroundMode);
var
  X, Y, W, H, W2, H2: Integer;
begin
  W := Bmp.Width;
  H := Bmp.Height;

  Y := R.Top;
  while Y < R.Bottom do
  begin
    X := R.Left;
    if Y + H > R.Bottom then
      H2 := R.Bottom - Y
    else
      H2 := H;
    while X < R.Right do
    begin
      if X + W > R.Right then
        W2 := R.Right - X
      else
        W2 := W;
      BitBlt(Canvas.Handle, X, Y, W2, H2, Bmp.Canvas.Handle, 0, 0, SRCCOPY);
      Inc(X, W);
      if Mode = bmRepeatY then Break;
    end;
    Inc(Y, H);
    if Mode = bmRepeatX then Break;
  end;
end;

procedure FillStyleBackground(Canvas: TCanvas; R: TRect; Style: TSkinStyle);
  procedure FillBackgroundColor;
  begin
    Canvas.Brush.Color := Style.BackgroundColor;
    Canvas.Brush.Style := bsSolid;
    Canvas.FillRect(R);
  end;
var
  W, H: Integer;
begin
  if not Style.BackgroundImage.Empty then
  begin
    case Style.BackgroundMode of
      bmRepeat:
        TileBitmap(Canvas, R, Style.BackgroundImage, Style.BackgroundMode);
      bmRepeatX, bmRepeatY:
      begin
        FillBackgroundColor;
        TileBitmap(Canvas, R, Style.BackgroundImage, Style.BackgroundMode);
      end;
      bmNoRepeat:
      begin
        FillBackgroundColor;
        if R.Left + Style.BackgroundImage.Width > R.Right then
          W := R.Right - R.Left
        else
          W := Style.BackgroundImage.Width;
        if R.Top + Style.BackgroundImage.Height > R.Bottom then
          H := R.Bottom - R.Top
        else
          H := Style.BackgroundImage.Height;
        BitBlt(Canvas.Handle, R.Left, R.Top, W, H,
            Style.BackgroundImage.Canvas.Handle, 0, 0, SRCCOPY);
      end;
      bmStretch:
        Canvas.StretchDraw(R, Style.BackgroundImage);
    end; //case
  end else
    FillBackgroundColor;
end;

procedure DrawSimpleBorder(Canvas: TCanvas; R: TRect; Style: TSkinStyle);
var
  B: Integer;
begin
  if Style.BorderWidth mod 2 = 0 then
  begin
    Inc(R.Right);
    Inc(R.Bottom);
  end;
  with Canvas do
  begin
    Pen.Color := Style.BorderColor;
    Brush.Style := bsClear;
    if Style.BorderStyle = sbDouble then
    begin
      B := Style.BorderWidth div 3;
      Pen.Width := B;
      Pen.Style := psSolid;
      InflateRect(R, - B div 2, - B div 2);
      Rectangle(R);
      InflateRect(R, -B * 2, -B * 2);
      Rectangle(R);
    end else
    begin
      B := Style.BorderWidth;
      Pen.Width := B;
      case Style.BorderStyle of
      sbSolid  : Pen.Style := psSolid;
      sbDashed : Pen.Style := psDash;
      sbDotted : Pen.Style := psDot;
      end;
      InflateRect(R, - B div 2, - B div 2);
      Rectangle(R);
    end;
  end;
end;

procedure Draw3DBorder(Canvas: TCanvas; R: TRect; Style: TSkinStyle);
  procedure Draw3DBorderSub(R: TRect; Depth: Integer);
  var
    Points1, Points2: array[0..5] of TPoint;
    D: Integer;
  begin
    D := Abs(Depth);
    Points1[0] := Point(R.Left, R.Top);
    Points1[1] := Point(R.Left, R.Bottom);
    Points1[2] := Point(R.Left + D, R.Bottom - D);
    Points1[3] := Point(R.Left + D, R.Top + D);
    Points1[4] := Point(R.Right - D, R.Top + D);
    Points1[5] := Point(R.Right, R.Top);

    Points2[0] := Point(R.Left, R.Bottom);
    Points2[1] := Point(R.Right, R.Bottom);
    Points2[2] := Point(R.Right, R.Top);
    Points2[3] := Point(R.Right - D, R.Top + D);
    Points2[4] := Point(R.Right - D, R.Bottom - D);
    Points2[5] := Point(R.Left + D, R.Bottom - D);
    with Canvas do
    begin
      Pen.Style := psClear;
      if Depth > 0 then
        Brush.Color := HighlightColor
      else
        Brush.Color := ShadowColor;
      Polygon(Points1);
      if Depth < 0 then
        Brush.Color := HighlightColor
      else
        Brush.Color := ShadowColor;
      Polygon(Points2);
    end;
  end;
var
  B: Integer;
begin
  case Style.BorderStyle of
    sbOutset :
      Draw3DBorderSub(R, Style.BorderWidth);
    sbInset :
      Draw3DBorderSub(R, -Style.BorderWidth);
    sbGroove :
    begin
      B := Style.BorderWidth div 2;
      Draw3DBorderSub(R, -B);
      InflateRect(R, -B, -B);
      Draw3DBorderSub(R, B);
    end;
    sbRidge :
    begin
      B := Style.BorderWidth div 2;
      Draw3DBorderSub(R, B);
      InflateRect(R, -B, -B);
      Draw3DBorderSub(R, -B);
    end;
  end;
end;

procedure DrawImageBorder(Canvas: TCanvas; R: TRect; Style: TSkinStyle; Transparent: Boolean);
  procedure SerialDraw(Bmp: TBitmap; X, Y, Max: Integer; Vertical: Boolean = False);
  begin
    if not Vertical then
      while X < Max do
      begin
        Canvas.Draw(X, Y, Bmp);
        Inc(X, Bmp.Width);
      end
    else
      while Y < Max do
      begin
        Canvas.Draw(X, Y, Bmp);
        Inc(Y, Bmp.Height);
      end;
  end;
var
  TL, TM, TR, SL, SR, BL, BM, BR: TBitmap;
  hClipRgn: HRGN;
begin
  hClipRgn := CreateRectRgnIndirect(R);
  SelectClipRgn(Canvas.Handle, hClipRgn);
  try
    with Canvas do
    begin
      TL := Style.TopLeftImage;
      TM := Style.TopImage;
      TR := Style.TopRightImage;
      SL := Style.LeftImage;
      SR := Style.RightImage;
      BL := Style.BottomLeftImage;
      BM := Style.BottomImage;
      BR := Style.BottomRightImage;

      if Transparent and (Style.TransparentColor <> clNone) then
      begin
        TL.TransparentColor := Style.TransparentColor;
        TM.TransparentColor := Style.TransparentColor;
        TR.TransparentColor := Style.TransparentColor;
        SL.TransparentColor := Style.TransparentColor;
        SR.TransparentColor := Style.TransparentColor;
        BL.TransparentColor := Style.TransparentColor;
        BM.TransparentColor := Style.TransparentColor;
        BR.TransparentColor := Style.TransparentColor;
        TL.Transparent := True;
        TM.Transparent := True;
        TR.Transparent := True;
        SL.Transparent := True;
        SR.Transparent := True;
        BL.Transparent := True;
        BM.Transparent := True;
        BR.Transparent := True;
      end;

      if not SL.Empty then
        SerialDraw(SL, R.Left, R.Top + TL.Height, R.Bottom - BL.Height, True);
      if not SR.Empty then
        SerialDraw(SR, R.Right - SR.Width, R.Top + TR.Height, R.Bottom - BR.Height, True);

      if not BL.Empty then
        Draw(R.Left, R.Bottom - BL.Height, BL);
      if not BM.Empty then
        SerialDraw(BM, R.Left + BL.Width, R.Bottom - BM.Height, R.Right - BR.Width);
      if not BR.Empty then
        Draw(R.Right - BR.Width, R.Bottom - BR.Height, BR);

      if not TL.Empty then
        Draw(R.Left, R.Top, TL);
      if not TM.Empty then
        SerialDraw(TM, R.Left + TL.Width, R.Top, R.Right - TR.Width);
      if not TR.Empty then
        Draw(R.Right - TR.Width, R.Top, TR);
    end;
  finally
    SelectClipRgn(Canvas.Handle, 0);
    DeleteObject(hClipRgn);
  end;
end;

procedure DrawStyleBorder(Canvas: TCanvas; R: TRect; Style: TSkinStyle; Transparent: Boolean = False);
begin
  case Style.BorderStyle of
  sbSolid, sbDashed, sbDotted, sbDouble :
    DrawSimpleBorder(Canvas, R, Style);
  sbInset, sbOutset, sbGroove, sbRidge :
    Draw3DBorder(Canvas, R, Style);
  sbImage :
    DrawImageBorder(Canvas, R, Style, Transparent);
  end;
end;

procedure DrawStyleText(Canvas: TCanvas; R: TRect; Style: TSkinStyle; Text: WideString);
var
  Flags: Cardinal;
  Extent: TSize;
begin
  AssignStyleToFont(Canvas.Font, Style);
  Extent := WideTextExtent(Canvas, Text);
  Flags := DT_NOPREFIX or DT_SINGLELINE or DT_VCENTER;

  Inc(R.Top, Style.PaddingTop);
  Inc(R.Left, Style.PaddingLeft);
  Dec(R.Bottom, Style.PaddingBottom);
  Dec(R.Right, Style.PaddingRight);
  case Style.TextAlign of
  taLeft:
    Flags := Flags or DT_LEFT;
  taRight:
    Flags := Flags or DT_RIGHT;
  taCenter:
    Flags := Flags or DT_CENTER;
  end;

  SetBkMode(Canvas.Handle, Windows.TRANSPARENT);
  WideDrawText(Canvas, Text, R, Flags);
end;

function TraceBackStyleName(Element, ClassName, Status: String;
  Level: Integer): String;
begin
  case Level of
  0 : Result := Format('%s.%s:%s', [Element, ClassName, Status]);
  1 : Result := Format('%s.%s', [Element, ClassName]);
  2 : Result := Format('%s:%s', [Element, Status]);
  3 : Result := Element;
  else
    Result := '';
  end;
end;

end.
