unit USimpleUICore;

interface

uses
  Windows, SysUtils, Forms, UNsmPlugin, UNsmTypes, UNsmConsts, UUserProfile,
  UNsmUIPluginMainEx, USkinResource, Graphics, UConfig, ActiveX, ShellAPI,
  UNotify, StrUtils, UNsmUtils, Classes, ExtCtrls;

type
  TSimpleUICore = class(TNsmUIPluginMainEx)
  private
    FNotifyItems: TEventNotifyList;
    FUserProfiles: TUserProfiles;
    FModules: TStringList;
    FActionTimer: TTimer;
    FGlobalStatus: Integer;
    procedure ConnectStartupProfiles;
    procedure CloseAllWindow;
    procedure OnActionUpdate(Sender: TObject);
    procedure CheckReverceMember(Connection: TNsmUIConnection; Member: TNsmUIMember);

  protected
    procedure DoInitialize; override;
    procedure DoTerminate; override;
    procedure DoAllModulesLoaded; override;

    procedure DoCreateConnection(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoDeleteConnection(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoConnectionChange; override;
    procedure DoConnectionInfoChange(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoAddMember(AHandle: HNsmConnection; AMInfo: TAddMemberInfo); override;
    procedure DoMemberChange(AHandle: HNsmConnection); override;
    procedure DoMemberInfoChange(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoMemberInfoChanging(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoGroupChange(AHandle: HNsmConnection); override;
    procedure DoGroupInfoChange(AHandle: HNsmConnection; lParam: Cardinal); override;
    procedure DoCreateSession(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoDeleteSession(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoSessionInfoChange(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoSessionMemberChange(AHandle: HNsmSession); override;
    procedure DoSessionMemberInfoChange(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoSessionSendMessage(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoSessionReceiveMessage(AHandle: HNsmSession; lParam: Cardinal); override;
    procedure DoNotify(NInfo: TNotifyInfo; var Result: Integer); override;
    procedure DoMessageBox(MBInfo: TMessageBoxInfo; var Result: Integer); override;
    procedure DoShowOptionDialog(hWnd: HWND; var Result: Integer); override;
  public
    constructor Create; override;
    destructor Destroy; override;

    procedure SetGlobalStatus(Status: Integer);
    function GetGlobalStatus: Integer;
    function FindListWindow(CHandle: HNsmConnection): TForm;
    function FindSessionWindow(SHandle: HNsmSession): TForm;
    procedure SortMemberList(AConnection: TNsmUIConnection);
    procedure SortGroupList(AConnection: TNsmUIConnection);
    procedure OpenURL(URL: String);
    procedure GetModuleList(Strings: TStrings);
    procedure DebugPrint(S: String);

    property NotifyItems: TEventNotifyList read FNotifyItems;
    property UserProfiles: TUserProfiles read FUserProfiles;
    property Modules: TStringList read FModules;
  end;

  // SimpleLog vOCƂ̘AgɎgp\
  TGetLogFileNameInfo = record
    lpProtocol: LPCTSTR;        // vgR
    lpUserAccount: LPCTSTR;     // [UAJEg
    lpAccount: LPCTSTR;         // ΏۃoAJEg
    lpLogName: LPTSTR;          // Ot@Ci[obt@
    nBuffSize: Integer;         // obt@TCY
  end;
  PGetLogFileNameInfo = ^TGetLogFileNameInfo;

const
  NMS_SIMPLELOG_GETLOGFILENAME = NMM_ADDIN + '/SimpleLog/GetLogFileName';

  NMP_INFO_APIVER       = NSM_API_VERSION;
  NMP_INFO_MODULENAME   = NMM_UI;
  NMP_INFO_TITLE        = 'Simple UI';
  NMP_INFO_DESCRIPTION  = 'Default UI for Regnessem';
  NMP_INFO_AUTHOR       = 'Yamaneko';
  NMP_INFO_COPYRIGHT    = 'Copyright (c) 2001-2003 Yamaneko';
  NMP_INFO_PLUGINVER    = '0.1.8';

var
  SimpleUICore: TSimpleUICore;

implementation

uses
  UFrmBase, UFrmMain, UFrmList, UFrmChat, UFrmSession, UFrmNotify,
  UStrResource, UFrmAddedNotify;

// TSimpleUICore ---------------------------------------------------------------

constructor TSimpleUICore.Create;
begin
  inherited;
  FNotifyItems := TEventNotifyList.Create;
  FUserProfiles := TUserProfiles.Create;
  FModules := TStringList.Create;
  FActionTimer := TTimer.Create(nil);
  FActionTimer.Interval := 500;
  FActionTimer.OnTimer := OnActionUpdate;
  FActionTimer.Enabled := True;

  // vOCݒ
  PluginInfo[NMPI_APIVER]       := NMP_INFO_APIVER;
  PluginInfo[NMPI_MODULENAME]   := NMP_INFO_MODULENAME;
  PluginInfo[NMPI_TITLE]        := NMP_INFO_TITLE;
  PluginInfo[NMPI_DESCRIPTION]  := NMP_INFO_DESCRIPTION;
  PluginInfo[NMPI_AUTHOR]       := NMP_INFO_AUTHOR;
  PluginInfo[NMPI_COPYRIGHT]    := NMP_INFO_COPYRIGHT;
  PluginInfo[NMPI_PLUGINVER]    := NMP_INFO_PLUGINVER;
end;

destructor TSimpleUICore.Destroy;
begin
  FModules.Free;
  FNotifyItems.Free;
  FUserProfiles.Free;
  FActionTimer.Free;
  inherited;
end;

procedure TSimpleUICore.SetGlobalStatus(Status: Integer);
var
  I: Integer;
begin
  FGlobalStatus := Status;
  for I := 0 to Connections.Count - 1 do
    Connections[I].ChangeUserStatus(Status);
end;

function TSimpleUICore.GetGlobalStatus: Integer;
var
  I: Integer;
begin
  if Connections.Count = 0 then
    Result := NMST_OFFLINE
  else if Connections.Count = 1 then
    Result := Connections[0].UserStatus
  else
  begin
    Result := Connections[0].UserStatus;
    for I := 1 to Connections.Count - 1 do
      if Connections[I].UserStatus <> Result then
      begin
        Result := FGlobalStatus;
        Break;
      end;
  end;
  FGlobalStatus := Result;
end;

procedure TSimpleUICore.OpenURL(URL: String);
  function OpenWith2chBrowser: Boolean;
  begin
    if (Is2chUrl(URL)) and (Length(Config.Path2chBrowser) > 0) then
    begin
      ShellExecute(0, nil, PChar(Config.Path2chBrowser), PChar(URL),
              nil, SW_SHOW);
      Result := True;
    end else
      Result := False;
  end;
  function OpenWithDefaultBrowser: Boolean;
  begin
    if (IsUrl(URL)) and (Length(Config.PathDefaultBrowser) > 0) then
    begin
      ShellExecute(0, nil, PChar(Config.PathDefaultBrowser), PChar(URL),
              nil, SW_SHOW);
      Result := True;
    end else
      Result := False;
  end;
begin
  if not OpenWith2chBrowser and not OpenWithDefaultBrowser then
    ShellExecute(0, nil, PChar(URL), nil, nil, SW_SHOW);
end;

function EnumModuleProc(lpModuleName: LPTSTR; nData: Integer): Integer; StdCall;
begin
  TStringList(nData).Add(lpModuleName);
  Result := 0;
end;

procedure TSimpleUICore.GetModuleList(Strings: TStrings);
begin
  CallService(NMS_SYSTEM_ENUMMODULES,
      Cardinal(@EnumModuleProc), Cardinal(Strings));
end;

function TSimpleUICore.FindListWindow(CHandle: HNsmConnection): TForm;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to Screen.FormCount - 1 do
    if (Screen.Forms[I] is TFrmList) and
       Assigned(TFrmList(Screen.Forms[I]).Connection) and
       (TFrmList(Screen.Forms[I]).Connection.Handle = CHandle) then
    begin
      Result := Screen.Forms[I];
      Break;
    end;
end;

function TSimpleUICore.FindSessionWindow(SHandle: HNsmSession): TForm;
var
  I: Integer;
begin
  Result := nil;
  for I := 0 to Screen.FormCount - 1 do
    if (Screen.Forms[I] is TFrmSession) and
       Assigned(TFrmSession(Screen.Forms[I]).Session) and
       (TFrmSession(Screen.Forms[I]).Session.Handle = SHandle) then
    begin
      Result := Screen.Forms[I];
      Break;
    end;
end;

procedure TSimpleUICore.SortMemberList(AConnection: TNsmUIConnection);
begin
  case Config.SortKey of
  skName    : AConnection.Members.Sort(skName, skAccount, skAccount);
  skAccount : AConnection.Members.Sort(skAccount, skAccount, skAccount);
  skStatus  : AConnection.Members.Sort(skBusyness, skStatus, skAccount);
  end;
end;

procedure TSimpleUICore.SortGroupList(AConnection: TNsmUIConnection);
begin
  AConnection.Groups.Sort;
end;

procedure TSimpleUICore.ConnectStartupProfiles;
var
  I: Integer;
begin
  for I := 0 to UserProfiles.Count - 1 do
    if UserProfiles[I].AutoConnect then
      LogIn(UserProfiles[I].Protocol,
            UserProfiles[I].Account,
            UserProfiles[I].Password,
            UserProfiles[I].ProfileName,
            UserProfiles[I].Status);
end;

procedure TSimpleUICore.CloseAllWindow;
var
  I: Integer;
begin
  for I := Screen.FormCount - 1 downto 0 do
  begin
    if Screen.Forms[I] is TFrmBase then
      TFrmBase(Screen.Forms[I]).UpdateConfig;
    if Screen.Forms[I] <> FrmMain then
      Screen.Forms[I].Free;
  end;
end;

procedure TSimpleUICore.OnActionUpdate(Sender: TObject);
var
  I: Integer;
begin
  for I := 0 to Screen.FormCount - 1 do
    if (Screen.Forms[I] is TFrmBase) and (Screen.Forms[I].Visible) then
      TFrmBase(Screen.Forms[I]).UpdateActions;
end;

procedure TSimpleUICore.CheckReverceMember(Connection: TNsmUIConnection; Member: TNsmUIMember);
begin
  // ɃXgɓĂꍇ͂ȂɂȂ
  if Connection.Members.Contains(Member.Account) or
     Connection.BlockMembers.Contains(Member.Account) or
     Connection.AllowMembers.Contains(Member.Account) then
    Exit;
{
  // ʒmAIɋXgɒǉ
  if not MsnConnection.GTC then
  begin
    MsnConnection.AllowMember(Member.Account);
    Exit;
  end;
}
  // ʒm\
  with TfrmAddedNotify.CreateAsDialog(FrmMain) do
  begin
    Prompt := Member.Name + ' ( ' + Member.Account +
                         ' ) ̃oXgɒǉ܂B';
    if ShowModal = IDOK then
    begin
      // E֎~Xgɒǉ
      if rbAllow.Checked then
        Connection.AddMember(Member.Account, NMLK_ALLOWLIST)
      else
        Connection.AddMember(Member.Account, NMLK_BLOCKLIST);

      // R^NgXgɒǉ
      if cbAddMemberList.Checked then
        Connection.AddMember(Member.Account, NMLK_FORWARDLIST);
    end;
    Free;
  end;
end;

procedure TSimpleUICore.DebugPrint(S: String);
begin
  CallService(NMS_SYSTEM_DEBUG_PRINT, Cardinal(PChar(S)), 0);
end;

// -----------------------------------------------------------------------------
// p protected \bh

procedure TSimpleUICore.DoInitialize;
begin
  inherited;
  Config.LoadFromFile(GetIniFileName);
  UserProfiles.LoadFromFile(GetIniFileName);
  FNotifyItems.LoadFromFile(GetIniFileName);
  FrmMain := TFrmMain.Create(nil);
end;

procedure TSimpleUICore.DoTerminate;
begin
  inherited;
  CloseAllWindow;
  FrmMain.TaskTray.Active := False;
  FrmMain.Free;
  FrmMain := nil;
  Config.SaveToFile(GetIniFileName);
  UserProfiles.SaveToFile(GetIniFileName);
  FNotifyItems.SaveToFile(GetIniFileName);
end;

procedure TSimpleUICore.DoAllModulesLoaded;
begin
  inherited;
  GetModuleList(FModules);
  FrmMain.Initialize;
  ConnectStartupProfiles;  
end;

procedure TSimpleUICore.DoCreateConnection(AHandle: HNsmConnection; lParam: Cardinal);
var
  Form: TFrmList;
begin
  inherited;
  if Assigned(FrmMain) then
  begin
    Form := TFrmList.Create(FrmMain);
    Form.Connection := Connections.Find(AHandle);
    FrmMain.DockWindow(Form);
  end;
end;

procedure TSimpleUICore.DoDeleteConnection(AHandle: HNsmConnection; lParam: Cardinal);
var
  Form: TFrmList;
begin
  inherited;
  Form := TFrmList(FindListWindow(AHandle));
  if Assigned(Form) and Assigned(FrmMain) then
  begin
    FrmMain.RemoveTab(Form);
    Form.Connection := nil;
    Form.Release;
  end;
end;

procedure TSimpleUICore.DoConnectionChange;
begin
  inherited;
  if Assigned(FrmMain) then
    FrmMain.UpdateTaskTray;
end;

procedure TSimpleUICore.DoConnectionInfoChange(AHandle: HNsmConnection; lParam: Cardinal);
var
  Form: TFrmList;
  I: Integer;
begin
  inherited;
  Form := TFrmList(FindListWindow(AHandle));
  if Assigned(Form) then
  begin
    if (PNsmConnectionInfo(lParam).nInfoKey = NMCI_STATUS) and
       (Form.Connection.Status = NMCS_CONNECTED) then
    begin
      SortMemberList(Form.Connection);
      SortGroupList(Form.Connection);
      Form.UpdateMemberList;
      for I := 0 to Form.Connection.ReverseMembers.Count - 1 do
        CheckReverceMember(Form.Connection, Form.Connection.ReverseMembers[I]);
    end;
    Form.UpdateUserStatus;
  end;
  if Assigned(FrmMain) then
    FrmMain.UpdateTaskTray;
end;

procedure TSimpleUICore.DoAddMember(AHandle: HNsmConnection; AMInfo: TAddMemberInfo);
var
  Form: TFrmList;
  Idx: Integer;
begin
  inherited;
  if AMInfo.nListKind = NMLK_REVERSELIST then
  begin
    Form := TFrmList(FindListWindow(AHandle));
    if Assigned(Form) and (Form.Connection.Status = NMCS_CONNECTED) then
    begin
      Idx := Form.Connection.ReverseMembers.IndexOfAccount(AMInfo.lpAccount);
      if Idx > -1 then
        CheckReverceMember(Form.Connection, Form.Connection.ReverseMembers[Idx]);
    end;
  end;
end;

procedure TSimpleUICore.DoMemberChange(AHandle: HNsmConnection);
var
  Form: TFrmList;
begin
  inherited;
  Form := TFrmList(FindListWindow(AHandle));
  if Assigned(Form) and (Form.Connection.Status = NMCS_CONNECTED) then
  begin
    SortMemberList(Form.Connection);
    Form.UpdateMemberList;
  end;
end;

procedure TSimpleUICore.DoMemberInfoChange(AHandle: HNsmConnection; lParam: Cardinal);
var
  Form: TFrmList;
begin
  inherited;
  if not (PNsmMemberInfo(lParam).nListKind = NMLK_FORWARDLIST) then Exit;
  Form := TFrmList(FindListWindow(AHandle));
  if Assigned(Form) and (Form.Connection.Status = NMCS_CONNECTED) then
  begin
    SortMemberList(Form.Connection);
    Form.UpdateMemberList;
  end;
end;

procedure TSimpleUICore.DoMemberInfoChanging(AHandle: HNsmConnection; lParam: Cardinal);
var
  Form: TFrmList;
  Idx, NewStatus, OldStatus: Integer;
  Member: TNsmUIMember;
begin
  inherited;
  Form := TFrmList(FindListWindow(AHandle));
  if Assigned(Form) then
  begin
    if (Form.Connection.Status = NMCS_CONNECTED) and
       (PNsmMemberInfo(lParam).nInfoKey = NMMI_STATUS) and
       ((PNsmMemberInfo(lParam).nFlags and NMIF_INITLIST) = 0) then
    begin
      Idx := Form.Connection.Members.IndexOfAccount(PNsmMemberInfo(lParam).lpAccount);
      if Idx > -1 then
      begin
        Member := Form.Connection.Members[Idx];
        OldStatus := Member.Status;
        NewStatus := PInteger(PNsmMemberInfo(lParam).lpInfo.lpBuffer)^;
        if (OldStatus in [NMST_OFFLINE, NMST_HIDDEN]) and
            not (NewStatus in [NMST_OFFLINE, NMST_HIDDEN]) then
          FNotifyItems[ntMemberOnline].ShowDefaultMsg([Member.Name])
        else if not (OldStatus in [NMST_OFFLINE, NMST_HIDDEN]) and
            (NewStatus in [NMST_OFFLINE, NMST_HIDDEN]) then
          FNotifyItems[ntMemberOffline].ShowDefaultMsg([Member.Name]);
      end;
    end;
  end;
end;

procedure TSimpleUICore.DoGroupChange(AHandle: HNsmConnection);
var
  Form: TFrmList;
begin
  inherited;
  Form := TFrmList(FindListWindow(AHandle));
  if Assigned(Form) and (Form.Connection.Status = NMCS_CONNECTED) then
  begin
    SortGroupList(Form.Connection);
    Form.UpdateMemberList;
  end;
end;

procedure TSimpleUICore.DoGroupInfoChange(AHandle: HNsmConnection; lParam: Cardinal);
var
  Form: TFrmList;
begin
  inherited;
  Form := TFrmList(FindListWindow(AHandle));
  if Assigned(Form) and (Form.Connection.Status = NMCS_CONNECTED) then
  begin
    SortGroupList(Form.Connection);
    Form.UpdateMemberList;
  end;
end;

procedure TSimpleUICore.DoCreateSession(AHandle: HNsmSession; lParam: Cardinal);
var
  Form: TFrmSession;
begin
  inherited;
  if not Assigned(FrmChat) then
    FrmChat := TFrmChat.Create(FrmMain);
  Form := TFrmSession.Create(FrmMain);
  Form.Session := Sessions.Find(AHandle);
  FrmChat.DockWindow(Form);
  FrmChat.Show;
end;

procedure TSimpleUICore.DoDeleteSession(AHandle: HNsmSession; lParam: Cardinal);
var
  Form: TFrmSession;
begin
  inherited;
  Form := TFrmSession(FindSessionWindow(AHandle));
  if Assigned(Form) then
  begin
    if Assigned(FrmChat) then
      FrmChat.RemoveTab(Form);
    Form.Release;
  end;
end;

procedure TSimpleUICore.DoSessionInfoChange(AHandle: HNsmSession; lParam: Cardinal);
var
  Form: TFrmSession;
begin
  inherited;
  Form := TFrmSession(FindSessionWindow(AHandle));
  if Assigned(Form) then
    Form.UpdateSessionInfo;
end;

procedure TSimpleUICore.DoSessionMemberChange(AHandle: HNsmSession);
var
  Form: TFrmSession;
begin
  inherited;
  Form := TFrmSession(FindSessionWindow(AHandle));
  if Assigned(Form) then
    Form.UpdateMemberList;
end;

procedure TSimpleUICore.DoSessionMemberInfoChange(AHandle: HNsmSession; lParam: Cardinal);
var
  Form: TFrmSession;
begin
  inherited;
  Form := TFrmSession(FindSessionWindow(AHandle));
  if Assigned(Form) then
    Form.UpdateMemberList;
end;

procedure TSimpleUICore.DoSessionSendMessage(AHandle: HNsmSession; lParam: Cardinal);
var
  Form: TFrmSession;
begin
  inherited;
  Form := TFrmSession(FindSessionWindow(AHandle));
  if Assigned(Form) then
  begin
    Form.DoSendMessage(PMessageInfo(lParam)^);
    FNotifyItems[ntSendMessage].ShowDefaultMsg([
        Form.Session.Connection.UserName,
        WideString(PMessageInfo(lParam)^.lpBody)]);
  end;
end;

procedure TSimpleUICore.DoSessionReceiveMessage(AHandle: HNsmSession; lParam: Cardinal);
var
  Form: TFrmSession;
  Idx: Integer;
begin
  inherited;
  Form := TFrmSession(FindSessionWindow(AHandle));
  if Assigned(Form) then
  begin
    Form.DoReceiveMessage(PMessageInfo(lParam)^);
    Idx := Form.Session.Members.IndexOfAccount(PMessageInfo(lParam)^.lpFrom);
    if (not Form.ParentFormActive) or (not Form.Visible) or
       (Form.WindowState = wsMinimized) then
    begin
      if Idx > -1 then
        FNotifyItems[ntReceiveMessage].ShowDefaultMsg([
            Form.Session.Members[Idx].Name,
            WideString(PMessageInfo(lParam)^.lpBody)])
      else
        FNotifyItems[ntReceiveMessage].ShowDefaultMsg([
            String(PMessageInfo(lParam)^.lpFrom),
            WideString(PMessageInfo(lParam)^.lpBody)]);
    end else
      FNotifyItems[ntReceiveMessage].PlaySound;
  end;
end;

procedure TSimpleUICore.DoNotify(NInfo: TNotifyInfo; var Result: Integer);
var
  ENT: TEventNotifyType;
begin
  inherited;
  case NInfo.nIcon of
  NMNI_EXCLAMATION  : ENT := ntExclamation;
  NMNI_STOP         : ENT := ntStop;
  NMNI_INFORMATION  : ENT := ntInformation;
  NMNI_QUESTION     : ENT := ntQuestion;
  NMNI_MESSAGE      : ENT := ntReceiveMessage;
  NMNI_MAIL         : ENT := ntReceiveMail;
  NMNI_FILE         : ENT := ntReceiveFile;
  NMNI_CONNECT      : ENT := ntConnect;
  NMNI_DISCONNECT   : ENT := ntDisconnect;
  NMNI_ONLINE       : ENT := ntMemberOnline;
  NMNI_OFFLINE      : ENT := ntMemberOffline;
  NMNI_BUSY         : ENT := ntMemberOnline;
  else
    ENT := ntExclamation;
  end;
  FNotifyItems[ENT].ShowNotify(String(NInfo.lpText), NInfo.nTimeToLive, False,
      NInfo.lpLinkTo, NInfo.lpOnClick, NInfo.nData);
  FNotifyItems[ENT].PlaySound;
end;

procedure TSimpleUICore.DoMessageBox(MBInfo: TMessageBoxInfo; var Result: Integer);
var
  Handle: HWND;
begin
  inherited;
  if MBInfo.hWnd <> 0 then
    Handle := MBInfo.hWnd
  else if Assigned(FrmMain) then
    Handle := FrmMain.Handle
  else
    Handle := 0;
  Result := MessageBoxW(Handle, MBInfo.lpText, MBInfo.lpCaption, MBINfo.uType);
end;

procedure TSimpleUICore.DoShowOptionDialog(hWnd: HWND; var Result: Integer);
begin
  inherited;
  FrmMain.actUIOption.Execute;
  Result := 0;
end;

// -----------------------------------------------------------------------------

initialization
  SimpleUICore := TSimpleUICore.Create;
  OleInitialize(nil);

finalization
  SimpleUICore.Free;
  OleUninitialize;
end.
