unit USkinResource;

interface

uses
  SysUtils, Windows, Classes, Graphics, IniFiles, Controls, ImgList, USkinStyles;

const
  MENUICON_CUT        = 0;
  MENUICON_COPY       = 1;
  MENUICON_PASTE      = 2;
  MENUICON_DELETE     = 3;
  MENUICON_UNDO       = 4;
  MENUICON_PROPERTY   = 5;
  MENUICON_OPTION     = 6;
  MENUICON_SOUND      = 7;
  MENUICON_NOTIFY     = 8;
  MENUICON_PLUGIN     = 9;
type
  TSkinElement = (seWindow, seCaption, seClientArea, seButton, seMenuBar,
      seTabSet, seTabButton, seScrollBar, seScrollThumb, seScrollUpButton,
      seScrollDownButton, seResizeGrip, seNone);
  TSkinStatus = (ssNone, ssActive, ssInactive, ssUp, ssDown, ssHot);      
  TWindowKind = (wkDefault, wkMain, wkChat, wkNotify);

  TWindowIconArray = array[wkDefault..wkNotify] of TIcon;

  TSkinResources = class(TObject)
  private
    FSkinFolder: String;
    FCommonImages: TImageList;
    FStateImages: TImageList;
    FMenuImages: TImageList;
    FWindowIcons: TWindowIconArray;

    FSkinStyles: TSkinStyles;
    procedure LoadImagesFromResource(ImageList: TImageList; Ident: String);
    procedure LoadIconFromResource(Icon: TIcon; Ident: String);
    procedure LoadImagesFromFile(ImageList: TImageList; Ident: String);
    procedure LoadIconFromFile(Icon: TIcon; Ident: String);
    procedure SetSkinFolder(Value: String);
  public
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    procedure Init;
    function GetStyle(Element: TSkinElement; ClassName: String = '';
        Status: TSkinStatus = ssNone): TSkinStyle;
    function GetWindowStyle(WindowKind: TWindowKind; Status: TSkinStatus): TSkinStyle;
    function GetWindowElementStyle(WindowKind: TWindowKind; WindowStatus: TSkinStatus;
        Element: TSkinElement; ClassName: String = ''; Status: TSkinStatus = ssNone): TSkinStyle;
    function ReadSkinString(const Section, Ident, Default: String): String;

    property CommonImages: TImageList read FCommonImages;
    property StateImages: TImageList read FStateImages;
    property MenuImages: TImageList read FMenuImages;
    property WindowIcons: TWindowIconArray read FWindowIcons;
    property SkinFolder: String read FSkinFolder write SetSkinFolder;
    property Styles: TSkinStyles read FSkinStyles;
  end;

function WindowKindToStr(WindowKind: TWindowKind): String;
function SkinElementToStr(SkinElement: TSkinElement): String;
function SkinStatusToStr(Status: TSkinStatus): String;

var
  SkinResources: TSkinResources;

implementation

const
  SkinIniName = 'Skin.ini';

// -----------------------------------------------------------------------------

constructor TSkinResources.Create;
var
  I: TWindowKind;
begin
  FSkinStyles := TSkinStyles.Create;
  FStateImages := TImageList.Create(nil);
  FCommonImages := TImageList.Create(nil);
  FMenuImages := TImageList.Create(nil);

  for I := Low(FWindowIcons) to High(FWindowIcons) do
    FWindowIcons[I] := TIcon.Create;
end;

destructor TSkinResources.Destroy;
var
  I: TWindowKind;
begin
  Clear;
  FSkinStyles.Free;
  FStateImages.Free;
  FCommonImages.Free;
  FMenuImages.Free;

  for I := Low(FWindowIcons) to High(FWindowIcons) do
    FWindowIcons[I].Free;
  inherited;
end;

procedure TSkinResources.SetSkinFolder(Value: String);
var
  I: TWindowKind;
begin
  if DirectoryExists(Value) then
  begin
    FSkinFolder := IncludeTrailingPathDelimiter(Value);
    Clear;
    Init;
    FSkinStyles.AppendFromFile(FSkinFolder + SkinIniName);
    LoadImagesFromFile(FStateImages, 'ListIcons');
    LoadImagesFromFile(FCommonImages, 'CommonIcons');
    LoadImagesFromFile(FMenuImages, 'MenuIcons');

    for I := Low(FWindowIcons) to High(FWindowIcons) do
      LoadIconFromFile(FWindowIcons[I], WindowKindToStr(I) + 'WindowIcon');
  end else
  begin
    FSkinFolder := '';
    Clear;
    Init;
  end;
end;

procedure TSkinResources.Clear;
var
  I: TWindowKind;
begin
  FSkinStyles.Clear;
  FStateImages.Clear;
  FCommonImages.Clear;
  FMenuImages.Clear;
  for I := Low(FWindowIcons) to High(FWindowIcons) do
    FWindowIcons[I].Assign(nil);
end;

procedure TSkinResources.Init;
var
  I: TWindowKind;
begin
  LoadImagesFromResource(FStateImages, 'STATEIMAGES16');
  LoadImagesFromResource(FCommonImages, 'COMMONIMAGES16');
  LoadImagesFromResource(FMenuImages, 'MENUIMAGES16');
  for I := Low(FWindowIcons) to High(FWindowIcons) do
    LoadIconFromResource(FWindowIcons[I], 'DEFAULTICON');

  with FSkinStyles.DefaultStyle do
  begin
    // tHg
    FontColor         := clBtnText;
    FontFamily        := 'MS UI Gothic';
    FontStyle         := [];
    FontSize          := 9;
    // eLXg
    TextAlign         := taLeft;
    // {bNX
    BorderStyle       := sbNone;
    BorderWidth       := 1;
    BorderColor       := clBlack;
    // wi
    BackgroundColor   := clBtnFace;
    TransparentColor  := clNone;
  end;
end;

function TSkinResources.GetStyle(Element: TSkinElement; ClassName: String = '';
    Status: TSkinStatus = ssNone): TSkinStyle;
begin
  Result := FSkinStyles.GetStyle(
      SkinElementToStr(Element),
      ClassName,
      SkinStatusToStr(Status));
end;

function TSkinResources.GetWindowStyle(WindowKind: TWindowKind;
    Status: TSkinStatus): TSkinStyle;
begin
  Result := FSkinStyles.GetStyle(
      SkinElementToStr(seWindow),
      WindowKindToStr(WindowKind),
      SkinStatusToStr(Status));
end;

function TSkinResources.GetWindowElementStyle(WindowKind: TWindowKind;
    WindowStatus: TSkinStatus; Element: TSKinElement; ClassName: String = '';
    Status: TSkinStatus = ssNone): TSkinStyle;
begin
  Result := FSkinStyles.GetChildStyle(
      SkinElementToStr(seWindow),
      WindowKindToStr(WindowKind),
      SkinStatusToStr(WindowStatus),
      SkinElementToStr(Element),
      ClassName,
      SkinStatusToStr(Status));
end;

procedure TSkinResources.LoadImagesFromResource(ImageList: TImageList; Ident: String);
var
  Bmp: TBitmap;
begin
  Bmp := TBitmap.Create;
  try
    Bmp.LoadFromResourceName(HInstance, Ident);
    ImageList.Height := Bmp.Height;
    ImageList.Width := Bmp.Height;
    ImageList.AddMasked(Bmp, clFuchsia);
  finally
    Bmp.Free;
  end;
end;

procedure TSkinResources.LoadIconFromResource(Icon: TIcon; Ident: String);
var
  HI: HICON;
begin
  HI := LoadIcon(HInstance, PChar(Ident));
  if HI <> 0 then
    Icon.Handle := HI;
end;

procedure TSkinResources.LoadImagesFromFile(ImageList: TImageList; Ident: String);
var
  Ini: TSkinIniFile;
  AFileName: String;
  Bmp: TBitmap;
begin
  Ini := FSkinStyles.IniFile;
  if not Assigned(Ini) then Exit;

  AFileName := FSkinFolder + Ini.ReadString('Images', Ident, '');
  if FileExists(AFileName) then
  begin
    Bmp := TBitmap.Create;
    try
      Bmp.LoadFromFile(AFileName);
      ImageList.Clear;
      ImageList.Height := Bmp.Height;
      ImageList.Width := Bmp.Height;
      ImageList.AddMasked(Bmp, clFuchsia);
    finally
      Bmp.Free;
    end;
  end;
end;

procedure TSkinResources.LoadIconFromFile(Icon: TIcon; Ident: String);
var
  Ini: TSkinIniFile;
  AFileName: String;
begin
  Ini := FSkinStyles.IniFile;
  if not Assigned(Ini) then Exit;

  AFileName := FSkinFolder + Ini.ReadString('Icons', Ident, '');
  if FileExists(AFileName) then
    Icon.LoadFromFile(AFileName);
end;

function TSkinResources.ReadSkinString(const Section, Ident, Default: String): String;
var
  Ini: TSkinIniFile;
begin
  Result := Default;
  Ini := FSkinStyles.IniFile;
  if Assigned(Ini) then
    Result := Ini.ReadString(Section, Ident, Default);
end;

// -----------------------------------------------------------------------------

function WindowKindToStr(WindowKind: TWindowKind): String;
begin
  case WindowKind of
    wkMain    : Result := 'Main';
    wkChat    : Result := 'Chat';
    wkNotify  : Result := 'Notify';
    wkDefault : Result := '';
  end;
end;

function SkinElementToStr(SkinElement: TSkinElement): String;
begin
  case SkinElement of
    seWindow            : Result := 'Window';
    seCaption           : Result := 'Caption';
    seClientArea        : Result := 'ClientArea';
    seButton            : Result := 'Button';
    seMenuBar           : Result := 'MenuBar';
    seTabSet            : Result := 'TabSet';
    seTabButton         : Result := 'TabButton';
    seScrollBar         : Result := 'ScrollBar';
    seScrollThumb       : Result := 'ScrollThumb';
    seScrollUpButton    : Result := 'ScrollUpButton';
    seScrollDownButton  : Result := 'ScrollDownButton';
    seResizeGrip        : Result := 'ResizeGrip';
    seNone              : Result := '';
  end;
end;

function SkinStatusToStr(Status: TSkinStatus): String;
begin
  case Status of
    ssNone        : Result := '';
    ssActive      : Result := 'Active';
    ssInactive    : Result := 'Inactive';
    ssUp          : Result := 'Up';
    ssDown        : Result := 'Down';
    ssHot         : Result := 'Hot';
  end;
end;

initialization
begin
  SkinResources := TSkinResources.Create;
  SkinResources.Init;
end;

finalization
  SkinResources.Free;
  
end.
