unit UNsmCom;
(* NSM W[ԒʐMjbg *)

interface

uses
  SysUtils, Windows, UNsmService, UNsmEvent, UNsmPlugin, UNsmTypes, Classes;

type
  TLogEvent = procedure(LogStr: String; User: Boolean) of Object;
  TNsmCom = class(TObject)
  private
    FServiceList: TNsmThreadServiceList;
    FEventList: TNsmThreadEventList;
    FMainThreadID: Cardinal;
    FMainThreadHandle: THandle;
    FDebugMode: Boolean;
    FOnLog: TLogEvent;
    FOnServiceListChange: TNotifyEvent;
    FOnEventListChange: TNotifyEvent;
  protected
    procedure DoLog(LogStr: String; User: Boolean = False);
    procedure DoServiceListChange;
    procedure DoEventListChange;
  public
    constructor Create;
    destructor Destroy; override;
    function CreateService(ServiceName: String; ProcAddr: TNsmServiceProc): HNsmService;
    function GetService(ServiceName: String): HNsmService;
    function CallService(ServiceHandle: HNsmService; wParam, lParam: Cardinal): Integer;

    function CreateEvent(EventName: String): HNsmEvent;
    function GetEvent(EventName: String): HNsmEvent;
    function NotifyEvent(EventHandle: HNsmEvent; wParam, lParam: Cardinal): Integer;
    function HookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;
    function UnhookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;

    property ServiceList: TNsmThreadServiceList read FServiceList;
    property EventList: TNsmThreadEventList read FEventList;
    property MainThreadID: Cardinal read FMainThreadID;
    property MainThreadHandle: THandle read FMainThreadHandle;
    property DebugMode: Boolean read FDebugMode write FDebugMode;
    property OnLog: TLogEvent read FOnLog write FOnLog;
    property OnServiceListChange: TNotifyEvent read FOnServiceListChange write FOnServiceListChange;
    property OnEventListChange: TNotifyEvent read FOnEventListChange write FOnEventListChange;
  end;

function GetPluginInitInfo: TNsmPluginInitInfo;

var
  NsmCom: TNsmCom;

implementation

// -----------------------------------------------------------------------------

constructor TNsmCom.Create;
begin
  inherited;
  FDebugMode := False;
  FServiceList := TNsmThreadServiceList.Create;
  FEventList := TNsmThreadEventList.Create;
  FMainThreadID := GetCurrentThreadId;
  DuplicateHandle(GetCurrentProcess(), GetCurrentThread(), GetCurrentProcess(),
    @FMainThreadHandle, 0, FALSE, DUPLICATE_SAME_ACCESS);
end;

destructor TNsmCom.Destroy;
begin
  FServiceList.Free;
  FEventList.Free;
  CloseHandle(FMainThreadHandle);
  inherited;
end;

procedure TNsmCom.DoLog(LogStr: String; User: Boolean = False);
begin
  if Assigned(FOnLog) and FDebugMode then
    FOnLog(LogStr, User);
end;

procedure TNsmCom.DoServiceListChange;
begin
  if Assigned(FOnServiceListChange) and FDebugMode then
    FOnServiceListChange(Self);
end;

procedure TNsmCom.DoEventListChange;
begin
  if Assigned(FOnEventListChange) and FDebugMode then
    FOnEventListChange(Self);
end;

function TNsmCom.CreateService(ServiceName: String; ProcAddr: TNsmServiceProc): HNsmService;
begin
  if FDebugMode then
    DoLog(Format('CreateService'#9'"%s", %d', [ServiceName, Integer(@ProcAddr)]));
  with FServiceList.LockList do
    try
      Result := HNsmService(Add(ServiceName, ProcAddr));
    finally
      FServiceList.UnlockList;
    end;
  DoServiceListChange;
end;

function TNsmCom.GetService(ServiceName: String): HNsmService;
var
  Svc: TNsmService;
begin
  if FDebugMode then
    DoLog(Format('GetService'#9'"%s"', [ServiceName]));
  with FServiceList.LockList do
    try
      Svc := Find(ServiceName);
      if Assigned(Svc) then
        Result := HNsmService(Svc)
      else
        Result := 0;
    finally
      FServiceList.UnlockList;
    end;
end;

function TNsmCom.CallService(ServiceHandle: HNsmService; wParam, lParam: Cardinal): Integer;
var
  Svc: TNsmService;
begin
  Result := 0;
  Svc := TNsmService(ServiceHandle);
  if Assigned(Svc) then
  begin
    if FDebugMode then
      DoLog(Format('CallService'#9'"%s", %d, %d', [Svc.Name, wParam, lParam]));
    if Assigned(Svc.ServiceProc) then
      Result := Svc.ServiceProc(wParam, lParam)
    else
      Result := 0;
  end;
end;

// -----------------------------------------------------------------------------

function TNsmCom.CreateEvent(EventName: String): HNsmEvent;
begin
  if FDebugMode then
    DoLog(Format('CreateEvent'#9'"%s"', [EventName]));
  with FEventList.LockList do
    try
      Result := HNsmEvent(Add(EventName));
    finally
      FEventList.UnlockList;
    end;
  DoEventListChange;
end;

function TNsmCom.GetEvent(EventName: String): HNsmEvent;
var
  Evt: TNsmEvent;
begin
  if FDebugMode then
    DoLog(Format('EventExists'#9'"%s"', [EventName]));
  with FEventList.LockList do
    try
      Evt := Find(EventName);
      if Assigned(Evt) then
        Result := HNsmEvent(Evt)
      else
        Result := 0;
    finally
      FEventList.UnlockList;
    end;
end;

function TNsmCom.NotifyEvent(EventHandle: HNsmEvent; wParam, lParam: Cardinal): Integer;
var
  Evt: TNsmEvent;
  I: Integer;
begin
  Result := 0;
  Evt := TNsmEvent(EventHandle);
  if Assigned(Evt) then
  begin
    if FDebugMode then
      DoLog(Format('NotifyEvent'#9'"%s", %d, %d', [Evt.Name, wParam, lParam]));
    for I := 0 to Evt.HookProcs.Count - 1 do
      Result := TNsmEventProc(Evt.HookProcs[I])(wParam, lParam);
  end;
end;

function TNsmCom.HookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;
var
  Evt: TNsmEvent;
begin
  if FDebugMode then
    DoLog(Format('HookEvent'#9'"%s", %d', [EventName, Integer(@ProcAddr)]));
  Result := 0;
  with FEventList.LockList do
    try
      Evt := Find(EventName);
      if Evt <> nil then
        Result := Evt.HookProcs.Add(@ProcAddr);
    finally
      FEventList.UnlockList;
    end;

  DoEventListChange;
end;

function TNsmCom.UnhookEvent(EventName: String; ProcAddr: TNsmEventProc): Integer;
var
  Evt: TNsmEvent;
begin
  if FDebugMode then
    DoLog(Format('UnhookEvent'#9'"%s, %d', [EventName, Integer(@ProcAddr)]));
  Result := 0;
  with FEventList.LockList do
    try
      Evt := Find(EventName);
      if Evt <> nil then
        Result := Evt.HookProcs.Remove(@ProcAddr);
    finally
      FEventList.UnlockList;
    end;

  DoEventListChange;
end;

// -----------------------------------------------------------------------------

type
  TAPCCallServiceInfo = record
    ServiceHandel: HNsmService;
    wParam: Cardinal;
    lParam: Cardinal;
    DoneEvent: Cardinal;
    Result: Integer;
  end;
  PAPCCallServiceInfo = ^TAPCCallServiceInfo;

function _CreateService(ServiceName: LPTSTR; ProcAddr: TNsmServiceProc): HNsmService; stdcall;
begin
  Result := NsmCom.CreateService(ServiceName, ProcAddr);
end;

function _GetService(ServiceName: LPTSTR): HNsmService; stdcall;
begin
  Result := NsmCom.GetService(ServiceName);
end;

procedure _APCCallService(dwParam: Cardinal); stdcall;
var
  CSInfo: PAPCCallServiceInfo;
begin
  CSInfo := PAPCCallServiceInfo(Pointer(dwParam));
  CSinfo.Result := NsmCom.CallService(
      CSInfo.ServiceHandel,
      CSInfo.wParam,
      CSInfo.lParam);
  SetEvent(CSInfo.DoneEvent);
end;

function _CallService(ServiceHandle: HNsmService; wParam,
  lParam: Cardinal): Integer; stdcall;
var
  CSInfo: TAPCCallServiceInfo;
begin
  if GetCurrentThreadID <> NsmCom.MainThreadID then
  begin
    CSInfo.ServiceHandel := ServiceHandle;
    CSInfo.wParam := wParam;
    CSInfo.lParam := lParam;
    CSInfo.DoneEvent := CreateEvent(nil, false, False, '');
    QueueUserAPC(@_APCCallService, NsmCom.MainThreadHandle, Cardinal(@CSInfo));
		WaitForSingleObject(CSInfo.DoneEvent, INFINITE);
		CloseHandle(CSInfo.DoneEvent);
    Result := CSInfo.Result;
  end else
    Result := NsmCom.CallService(ServiceHandle, wParam, lParam);
end;

function _CreateEvent(EventName: LPTSTR): HNsmEvent; stdcall;
begin
  Result := NsmCom.CreateEvent(EventName);
end;

function _GetEvent(EventName: LPTSTR): HNsmEvent; stdcall;
begin
  Result := NsmCom.GetEvent(EventName);
end;

function _NotifyEvent(EventHandle: HNsmEvent; wParam, lParam: Cardinal): Integer; stdcall;
begin
  Result := NsmCom.NotifyEvent(EventHandle, wParam, lParam);
end;

function _HookEvent(EventName: LPTSTR; ProcAddr: TNsmEventProc): Integer; stdcall;
begin
  Result := NsmCom.HookEvent(EventName, ProcAddr);
end;

function _UnhookEvent(EventName: LPTSTR; ProcAddr: TNsmEventProc): Integer; stdcall;
begin
  Result := NsmCom.UnhookEvent(EventName, ProcAddr);
end;

function GetPluginInitInfo: TNsmPluginInitInfo;
begin
  with Result do
  begin
    cbSize := SizeOf(TNsmPluginInitInfo);
    CreateService := _CreateService;
    GetService := _GetService;
    CallService := _CallService;
    CreateEvent := _CreateEvent;
    GetEvent := _GetEvent;
    NotifyEvent := _NotifyEvent;
    HookEvent := _HookEvent;
    UnhookEvent := _UnhookEvent;
  end;
end;

// -----------------------------------------------------------------------------

initialization
  NsmCom := TNsmCom.Create;

finalization
  NsmCom.Free;

end.
 