unit UNsmSystem;
(* NSM SYSTEM W[ *)

interface

uses
  Classes, Windows, Messages, MinApp, UNsmPlugin, UNsmCom, UNsmConnection,
  UNsmConsts, UNsmTypes, UNsmUtils, UNsmMemberList, UNsmSession, SysUtils,
  UVersionInfo;

type
  TNsmSystem = class(TObject)
  private
    FPluginList: TNsmPluginList;
    FConnections: TNsmThreadConnectionList;
    FSessions: TNsmThreadSessionList;
    FOnConnectionChange: TNotifyEvent;
    FOnSessionChange: TNotifyEvent;

    HModulesLoaded: HNsmEvent;
    HConnectionConnect: HNsmEvent;
    HConnectionDisconnect: HNsmEvent;
    HConnectionInfoChange: HNsmEvent;
    HConnectionMembersAdd: HNsmEvent;
    HConnectionMembersRemove: HNsmEvent;
    HConnectionMembersInfoChange: HNsmEvent;
    HConnectionMembersInfoChanging: HNsmEvent;
    HConnectionGroupsAdd: HNsmEvent;
    HConnectionGroupsRemove: HNsmEvent;
    HConnectionGroupsInfoChange: HNsmEvent;
    HSessionOpen: HNsmEvent;
    HSessionClose: HNsmEvent;
    HSessionInfoChange: HNsmEvent;
    HSessionMembersAdd: HNsmEvent;
    HSessionMembersRemove: HNsmEvent;
    HSessionMembersInfoChange: HNsmEvent;
    HSessionSendMessage: HNsmEvent;
    HSessionReceiveMessage: HNsmEvent;
  protected
    procedure DoConnectionChange;
    procedure DoSessionChange;
  public
    constructor Create; virtual;
    destructor Destroy; override;
    function Initialize: Integer;
    function Terminate: Integer;
    procedure CloseRelatedSessions(AConnection: TNsmConnection);
    procedure CloseAllSession;
    procedure DisconnectAll;

    property Connections: TNsmThreadConnectionList read FConnections;
    property Sessions: TNsmThreadSessionList read FSessions;
    property Plugins: TNsmPluginList read FPluginList write FPluginList;
    property OnConnectionChange: TNotifyEvent read FOnConnectionChange write FOnConnectionChange;
    property OnSessionChange: TNotifyEvent read FOnSessionChange write FOnSessionChange;
  end;

var
  NsmSystem: TNsmSystem;

implementation

function DoExit(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoDebugPrint(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEnumProtocols(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEnumModules(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoGetPluginInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;

function DoCreateConnection(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoDeleteConnection(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoGetConnectionInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoSetConnectionInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEnumConnections(wParam, lParam: Cardinal): Integer; StdCall; forward;

function DoAddMember(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoRemoveMember(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoGetMemberInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoSetMemberInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoBeginUpdate(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEndUpdate(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEnumMembers(wParam, lParam: Cardinal): Integer; StdCall; forward;

function DoAddGroup(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoRemoveGroup(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoGetGroupInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoSetGroupInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoBeginGroupsUpdate(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEndGroupsUpdate(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEnumGroups(wParam, lParam: Cardinal): Integer; StdCall; forward;

function DoCreateSession(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoDeleteSession(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoGetSessionInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoSetSessionInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEnumSessions(wParam, lParam: Cardinal): Integer; StdCall; forward;

function DoAddSessionMember(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoRemoveSessionMember(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoGetSessionMemberInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoSetSessionMemberInfo(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoBeginSessionMembersUpdate(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEndSessionMembersUpdate(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoEnumSessionMembers(wParam, lParam: Cardinal): Integer; StdCall; forward;

function DoSendMessage(wParam, lParam: Cardinal): Integer; StdCall; forward;
function DoReceiveMessage(wParam, lParam: Cardinal): Integer; StdCall; forward;

function StrToNsmInfo(NsmInfo: PNsmInfo; S: String): Integer; forward;
function WideStrToNsmInfo(NsmInfo: PNsmInfo; S: WideString): Integer; forward;
function IntToNsmInfo(NsmInfo: PNsmInfo; Value: Integer): Integer; forward;
function BoolToNsmInfo(NsmInfo: PNsmInfo; Value: Boolean): Integer; forward;
function NsmInfoToStr(NsmInfo: PNsmInfo): String; forward;
function NsmInfoToWideStr(NsmInfo: PNsmInfo): WideString; forward;
function NsmInfoToInt(NsmInfo: PNsmInfo): Integer; forward;
function NsmInfoToBool(NsmInfo: PNsmInfo): Boolean; forward;

// -----------------------------------------------------------------------------

constructor TNsmSystem.Create;
begin
  inherited;
  FConnections := TNsmThreadConnectionList.Create;
  FSessions := TNsmThreadSessionList.Create;
  FPluginList := TNsmPluginList.Create;
end;

destructor TNsmSystem.Destroy;
begin
  FConnections.Free;
  FSessions.Free;
  FPluginList.Free;
  inherited;
end;

function TNsmSystem.Initialize: Integer;
var
  I: Integer;
  InitInfo: TNsmPluginInitInfo;
begin
  // Cxg̍쐬
  HModulesLoaded := NsmCom.CreateEvent(NME_SYSTEM_MODULESLOADED);
  HConnectionConnect := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_CONNECT);
  HConnectionDisconnect := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_DISCONNECT);
  HConnectionInfoChange := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_INFOCHANGE);
  HConnectionMembersAdd := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_MEMBERS_ADD);
  HConnectionMembersRemove := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_MEMBERS_REMOVE);
  HConnectionMembersInfoChange := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_MEMBERS_INFOCHANGE);
  HConnectionMembersInfoChanging := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_MEMBERS_INFOCHANGING);
  HConnectionGroupsAdd := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_GROUPS_ADD);
  HConnectionGroupsRemove := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_GROUPS_REMOVE);
  HConnectionGroupsInfoChange := NsmCom.CreateEvent(NME_SYSTEM_CONNECTION_GROUPS_INFOCHANGE);

  HSessionOpen := NsmCom.CreateEvent(NME_SYSTEM_SESSION_OPEN);
  HSessionClose := NsmCom.CreateEvent(NME_SYSTEM_SESSION_CLOSE);
  HSessionInfoChange := NsmCom.CreateEvent(NME_SYSTEM_SESSION_INFOCHANGE);
  HSessionMembersAdd := NsmCom.CreateEvent(NME_SYSTEM_SESSION_MEMBERS_ADD);
  HSessionMembersRemove := NsmCom.CreateEvent(NME_SYSTEM_SESSION_MEMBERS_REMOVE);
  HSessionMembersInfoChange := NsmCom.CreateEvent(NME_SYSTEM_SESSION_MEMBERS_INFOCHANGE);

  HSessionSendMessage := NsmCom.CreateEvent(NME_SYSTEM_SESSION_SENDMESSAGE);
  HSessionReceiveMessage := NsmCom.CreateEvent(NME_SYSTEM_SESSION_RECEIVEMESSAGE);

  // T[rX̍쐬
  NsmCom.CreateService(NMS_SYSTEM_EXIT, DoExit);
  NsmCom.CreateService(NMS_SYSTEM_DEBUG_PRINT, DoDebugPrint);
  NsmCom.CreateService(NMS_SYSTEM_ENUMPROTOCOLS, DoEnumProtocols);
  NsmCom.CreateService(NMS_SYSTEM_ENUMMODULES, DoEnumModules);
  NsmCom.CreateService(NMS_SYSTEM_GETPLUGININFO, DoGetPluginInfo);

  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_CREATE, DoCreateConnection);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_DELETE, DoDeleteConnection);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GETINFO, DoGetConnectionInfo);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_SETINFO, DoSetConnectionInfo);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_ENUM, DoEnumConnections);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_MEMBERS_ADD, DoAddMember);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_MEMBERS_REMOVE, DoRemoveMember);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_MEMBERS_GETINFO, DoGetMemberInfo);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_MEMBERS_SETINFO, DoSetMemberInfo);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_MEMBERS_BEGINUPDATE, DoBeginUpdate);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_MEMBERS_ENDUPDATE, DoEndUpdate);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_MEMBERS_ENUM, DoEnumMembers);

  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GROUPS_ADD, DoAddGroup);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GROUPS_REMOVE, DoRemoveGroup);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GROUPS_GETINFO, DoGetGroupInfo);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GROUPS_SETINFO, DoSetGroupInfo);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GROUPS_BEGINUPDATE, DoBeginGroupsUpdate);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GROUPS_ENDUPDATE, DoEndGroupsUpdate);
  NsmCom.CreateService(NMS_SYSTEM_CONNECTION_GROUPS_ENUM, DoEnumGroups);

  NsmCom.CreateService(NMS_SYSTEM_SESSION_CREATE, DoCreateSession);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_DELETE, DoDeleteSession);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_GETINFO, DoGetSessionInfo);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_SETINFO, DoSetSessionInfo);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_ENUM, DoEnumSessions);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_MEMBERS_ADD, DoAddSessionMember);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_MEMBERS_REMOVE, DoRemoveSessionMember);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_MEMBERS_GETINFO, DoGetSessionMemberInfo);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_MEMBERS_SETINFO, DoSetSessionMemberInfo);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_MEMBERS_BEGINUPDATE, DoBeginSessionMembersUpdate);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_MEMBERS_ENDUPDATE, DoEndSessionMembersUpdate);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_MEMBERS_ENUM, DoEnumSessionMembers);

  NsmCom.CreateService(NMS_SYSTEM_SESSION_SENDMESSAGE, DoSendMessage);
  NsmCom.CreateService(NMS_SYSTEM_SESSION_RECEIVEMESSAGE, DoReceiveMessage);

  InitInfo := GetPluginInitInfo;
  // W[̏
  for I := 0 to FPluginList.Count - 1 do
    FPluginList[I].Initialize(@InitInfo);
  // SW[ǍICxg
  NsmCom.NotifyEvent(NsmCom.GetEvent(NME_SYSTEM_MODULESLOADED), 0, 0);

  Result := 0;
end;

function TNsmSystem.Terminate: Integer;
var
  I: Integer;
begin
  CloseAllSession;
  DisconnectAll;
  for I := 0 to FPluginList.Count - 1 do
    FPluginList[I].Terminate;
  Result := 0;
end;

// AConnection Ɋ֘AZbVSĕ
procedure TNsmSystem.CloseRelatedSessions(AConnection: TNsmConnection);
var
  I: Integer;
begin
  with FSessions.LockList do
    try
      for I := Count -1 downto 0 do
      begin
        if Session[I].Connection = AConnection then
          NsmCom.CallService(NsmCom.GetService(
              Format(NMS_PROTOCOL_SESSION_CLOSE, [AConnection.Protocol])),
              Cardinal(Session[I]), 0);
      end;
    finally
      FSessions.UnlockList;
    end;
end;

// SẴZbV
procedure TNsmSystem.CloseAllSession;
var
  I: Integer;
begin
  with FSessions.LockList do
    try
      for I := Count -1 downto 0 do
        NsmCom.CallService(NsmCom.GetService(
            Format(NMS_PROTOCOL_SESSION_CLOSE, [Session[I].Protocol])),
            Cardinal(Session[I]), 0);
    finally
      FSessions.UnlockList;
    end;
end;

// SẴRlNV폜
procedure TNsmSystem.DisconnectAll;
var
  I: Integer;
begin
  with FConnections.LockList do
    try
      for I := Count -1 downto 0 do
        NsmCom.CallService(NsmCom.GetService(
            Format(NMS_PROTOCOL_CONNECTION_DISCONNECT, [Connection[I].Protocol])),
            Cardinal(Connection[I]), 0);
    finally
      FConnections.UnlockList;
    end;
end;

procedure TNsmSystem.DoConnectionChange;
begin
  if Assigned(FOnConnectionChange) then
    FOnConnectionChange(Self);
end;

procedure TNsmSystem.DoSessionChange;
begin
  if Assigned(FOnSessionChange) then
    FOnSessionChange(Self);
end;

// -----------------------------------------------------------------------------

//  wParam  : 0
//  lParam  : 0
//  Return  : 0
function DoExit(wParam, lParam: Cardinal): Integer; StdCall;
begin
  Application.Terminate;
  Result := 0;
end;

//  wParam  : PChar
//  lParam  : 0
//  Return  : 0
function DoDebugPrint(wParam, lParam: Cardinal): Integer; StdCall;
begin
  if Assigned(NsmCom.OnLog) then
    NsmCom.OnLog(PChar(wParam), True);
  Result := 0;
end;

//  wParam  : TEnumProtocolCallback
//  lParam  : CӃf[^
//  Return  : 0
function DoEnumProtocols(wParam, lParam: Cardinal): Integer; StdCall;
var
  I: Integer;
  ModuleName, ProtcolName: String;
begin
  for I := 0 to NsmSystem.Plugins.Count - 1 do
  begin
    ModuleName := ExtractModuleType(NsmSystem.Plugins[I].PluginInfo.ModuleName);
    if ModuleName = NMM_PROTOCOL then
    begin
      ProtcolName := ExtractProtocolName(NsmSystem.Plugins[I].PluginInfo.ModuleName);
      TEnumProtocolCallback(wParam)(PChar(ProtcolName), lParam);
    end;
  end;
  Result := 0;
end;

//  wParam  : TEnumModuleCallback
//  lParam  : CӃf[^
//  Return  : 0
function DoEnumModules(wParam, lParam: Cardinal): Integer; StdCall;
var
  I: Integer;
begin
  for I := 0 to NsmSystem.Plugins.Count - 1 do
    TEnumModuleCallback(wParam)
        (PChar(NsmSystem.Plugins[I].PluginInfo.ModuleName), lParam);
  Result := 0;
end;

//  wParam  : LPPLUGININFO
//  lParam  : 0
//  Return  : ۂ lpBuffer ɏ񂾕
function DoGetPluginInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
  InfoStr: String;
begin
  Result := 0;
  Idx := NsmSystem.Plugins.IndexOfModuleName(PNsmPluginInfo(wParam)^.lpModuleName);
  if Idx > -1 then
  begin
    Result := NsmSystem.Plugins[Idx].GetPluginInfo(
        PNsmPluginInfo(wParam)^.nInfoNo, PNsmPluginInfo(wParam)^.lpBuffer,
        PNsmPluginInfo(wParam)^.nBufferSize);
  end else
  if PNsmPluginInfo(wParam)^.lpModuleName = NMM_SYSTEM then
  begin
    case PNsmPluginInfo(wParam)^.nInfoNo of
    NMPI_APIVER       : InfoStr := NSM_API_VERSION;
    NMPI_MODULENAME   : InfoStr := NMM_SYSTEM;
    NMPI_TITLE        : InfoStr := Application.Title;
    NMPI_DESCRIPTION  : InfoStr := GetVersionInfo(vrFileDescription);
    NMPI_AUTHOR       : InfoStr := GetVersionInfo(vrCompanyName);
    NMPI_COPYRIGHT    : InfoStr := GetVersionInfo(vrLegalCopyright);
    NMPI_PLUGINVER    : InfoStr := GetVersionInfo(vrFileVersion);
    end;
    StrLCopy(PNsmPluginInfo(wParam)^.lpBuffer, PChar(InfoStr),
      PNsmPluginInfo(wParam)^.nBufferSize);
    Result := StrLen(PNsmPluginInfo(wParam)^.lpBuffer) + 1;
  end;
end;

// Connection ------------------------------------------------------------------

//  wParam  : PChar vgR
//  lParam  : PChar 쐬RlNṼLvV
//  Return  :  HNsmConnection s 0 Ԃ
function DoCreateConnection(wParam, lParam: Cardinal): Integer; StdCall;
begin
  with NsmSystem.Connections.LockList do
    try
      Result := Integer(Add(PChar(wParam), PChar(wParam)));
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Assigned(Pointer(Result)) then
  begin
    NsmCom.NotifyEvent(NsmSystem.HConnectionConnect, Result, 0);
    NsmSystem.DoConnectionChange;
  end;
end;

//  wParam  : Ώ HNsmConnection
//  lParam  : 0
//  Return  :  -1 ȊOԂ
function DoDeleteConnection(wParam, lParam: Cardinal): Integer; StdCall;
begin
  with NsmSystem.Connections.LockList do
    try
      Result := IndexOf(TNsmConnection(wParam));
      if Result > -1 then
      begin
        NsmSystem.CloseRelatedSessions(Connection[Result]);
        Delete(Result);
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Result > -1 then
  begin
    NsmCom.NotifyEvent(NsmSystem.HConnectionDisconnect, wParam, 0);
    NsmSystem.DoConnectionChange;
  end;
end;

//  wParam  : Ώ HNsmConnection
//  lParam  : PNsmConnectionInfo
//  Return  : ۂɏ񂾃obt@TCY
function DoGetConnectionInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
  AConnection: TNsmConnection;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      Idx := IndexOf(TNsmConnection(wParam));
      if Idx > -1 then
      begin
        AConnection := TNsmConnection(wParam);
        case PNsmConnectionInfo(lParam)^.nInfoKey of
        NMCI_PROTOCOL:
          Result := StrToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.Protocol);
        NMCI_CAPTION:
          Result := WideStrToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.Caption);
        NMCI_STATUS:
          Result := IntToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.Status);
        NMCI_USER_NAME:
          Result := WideStrToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.User.Name);
        NMCI_USER_ACCOUNT:
          Result := StrToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.User.Account);
        NMCI_USER_STATUS:
          Result := IntToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.User.Status);
        NMCI_USER_STATUSSTRING:
          Result := WideStrToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.User.StatusStr);
        NMCI_USER_BUSYNESS:
          Result := IntToNsmInfo(PNsmConnectionInfo(lParam)^.lpInfo, AConnection.User.Busyness);
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
end;

//  wParam  : Ώ HNsmConnection
//  lParam  : PNsmConnectionInfo
//  Return  :  0 ȊOԂ
function DoSetConnectionInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
  AConnection: TNsmConnection;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      Idx := IndexOf(TNsmConnection(wParam));
      if Idx > -1 then
      begin
        AConnection := TNsmConnection(wParam);
        case PNsmConnectionInfo(lParam)^.nInfoKey of
        NMCI_PROTOCOL:
            AConnection.Protocol := NsmInfoToStr(PNsmConnectionInfo(lParam)^.lpInfo);
        NMCI_CAPTION :
            AConnection.Caption := NsmInfoToWideStr(PNsmConnectionInfo(lParam)^.lpInfo);
        NMCI_STATUS:
            AConnection.Status := NsmInfoToInt(PNsmConnectionInfo(lParam)^.lpInfo);
        NMCI_USER_ACCOUNT:
            AConnection.User.Account := NsmInfoToStr(PNsmConnectionInfo(lParam)^.lpInfo);
        NMCI_USER_NAME:
            AConnection.User.Name := NsmInfoToWideStr(PNsmConnectionInfo(lParam)^.lpInfo);
        NMCI_USER_STATUS:
            AConnection.User.Status := NsmInfoToInt(PNsmConnectionInfo(lParam)^.lpInfo);
        NMCI_USER_STATUSSTRING:
            AConnection.User.StatusStr := NsmInfoToWideStr(PNsmConnectionInfo(lParam)^.lpInfo);
        NMCI_USER_BUSYNESS:
            AConnection.User.Busyness := NsmInfoToInt(PNsmConnectionInfo(lParam)^.lpInfo);
        end;
        Result := 1;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Result = 1 then
  begin
    NsmCom.NotifyEvent(NsmSystem.HConnectionInfoChange, wParam, lParam);
    NsmSystem.DoConnectionChange;
  end;
end;

//  wParam  : TEnumConnectionCallback R[obN֐ւ̃|C^
//  lParam  : R[obN֐ɓnCӃf[^
//  Return  : 0
function DoEnumConnections(wParam, lParam: Cardinal): Integer; StdCall;
var
  I: Integer;
begin
  with NsmSystem.Connections.LockList do
    try
      for I := 0 to Count - 1 do
        TEnumConnectionCallback(wParam)(Integer(Connection[I]), lParam);
    finally
      NsmSystem.Connections.UnlockList;
    end;
  Result := 0;
end;

// -----------------------------------------------------------------------------
//  wParam  : Ώ HNsmConnection
//  lParam  : PAddMemberInfo
//  Return  :  0 ȊOԂ
function DoAddMember(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
  List: TNsmMemberList;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      Idx := IndexOf(TNsmConnection(wParam));
      if (Idx > -1) then
      begin
        case PAddMemberInfo(lParam).nListKind of
          NMLK_FORWARDLIST : List := Connection[Idx].Members;
          NMLK_REVERSELIST : List := Connection[Idx].ReverseMembers;
          NMLK_ALLOWLIST   : List := Connection[Idx].AllowMembers;
          NMLK_BLOCKLIST   : List := Connection[Idx].BlockMembers;
        else
          List := Connection[Idx].Members;
        end;
        if List.IndexOfAccount(PAddMemberInfo(lParam).lpAccount) = -1 then
        begin
          with List.Add(PAddMemberInfo(lParam).lpAccount) do
          begin
            Name := PAddMemberInfo(lParam).lpAccount;
            Status := NMST_OFFLINE;
          end;
          Result := 1;
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Result <> 0 then
    NsmCom.NotifyEvent(NsmSystem.HConnectionMembersAdd, wParam, lParam);
end;

//  wParam  : Ώ HNsmConnection
//  lParam  : PRemoveMemberInfo
//  Return  :  0 ȊOԂ
function DoRemoveMember(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, mIdx: Integer;
  List: TNsmMemberList;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
      begin
        case PRemoveMemberInfo(lParam).nListKind of
          NMLK_FORWARDLIST : List := Connection[cIdx].Members;
          NMLK_REVERSELIST : List := Connection[cIdx].ReverseMembers;
          NMLK_ALLOWLIST   : List := Connection[cIdx].AllowMembers;
          NMLK_BLOCKLIST   : List := Connection[cIdx].BlockMembers;
        else
          List := Connection[cIdx].Members;
        end;
        mIdx := List.IndexOfAccount(PRemoveMemberInfo(lParam).lpAccount);
        if mIdx > -1 then
        begin
          List.Delete(mIdx);
          Result := 1;
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Result = 1 then
    NsmCom.NotifyEvent(NsmSystem.HConnectionMembersRemove, wParam, lParam);
end;

//  wParam  : Ώ HNsmConnection
//  lParam  : PNsmMemberInfo
//  Return  :  0 ȊOԂ
function DoSetMemberInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, mIdx: Integer;
  Member: TNsmMember;
  MemberInfo: TNsmMemberInfo;
  List: TNsmMemberList;
begin
  Result := 0;
  MemberInfo := PNsmMemberInfo(lParam)^;
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
      begin
        case MemberInfo.nListKind of
          NMLK_FORWARDLIST : List := Connection[cIdx].Members;
          NMLK_REVERSELIST : List := Connection[cIdx].ReverseMembers;
          NMLK_ALLOWLIST   : List := Connection[cIdx].AllowMembers;
          NMLK_BLOCKLIST   : List := Connection[cIdx].BlockMembers;
        else
          List := Connection[cIdx].Members;
        end;
        mIdx := List.IndexOfAccount(MemberInfo.lpAccount);
        if mIdx > -1 then
        begin
          NsmCom.NotifyEvent(NsmSystem.HConnectionMembersInfoChanging, wParam, lParam);
          Member := List[mIdx];
          case MemberInfo.nInfoKey of
//          NMMI_ACCOUNT:   Member.Account  := NsmInfoToStr(MemberInfo.lpInfo);
          NMMI_NAME:      Member.Name     := NsmInfoToWideStr(MemberInfo.lpInfo);
          NMMI_STATUS:    Member.Status   := NsmInfoToInt(MemberInfo.lpInfo);
          NMMI_STATUSSTRING:
                          Member.StatusStr := NsmInfoToWideStr(MemberInfo.lpInfo);
          NMMI_BUSYNESS:  Member.Busyness  := NsmInfoToInt(MemberInfo.lpInfo);
          NMMI_GROUPID:   Member.Group := NsmInfoToInt(MemberInfo.lpInfo);
          end;
          Result := 1;
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if (Result = 1) and  (MemberInfo.nFlags and NMIF_NOCHANGEEVENT = 0) then
    NsmCom.NotifyEvent(NsmSystem.HConnectionMembersInfoChange, wParam, lParam);
end;

//  wParam  : Ώ HNsmConnection
//  lParam  : PNsmMemberInfo
//  Return  : ۂ lpBuffer ɏ񂾃TCY
function DoGetMemberInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, mIdx: Integer;
  Member: TNsmMember;
  List: TNsmMemberList;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
      begin
        case PNsmMemberInfo(lParam).nListKind of
          NMLK_FORWARDLIST : List := Connection[cIdx].Members;
          NMLK_REVERSELIST : List := Connection[cIdx].ReverseMembers;
          NMLK_ALLOWLIST   : List := Connection[cIdx].AllowMembers;
          NMLK_BLOCKLIST   : List := Connection[cIdx].BlockMembers;
        else
          List := Connection[cIdx].Members;
        end;
        mIdx := List.IndexOfAccount(PNsmMemberInfo(lParam).lpAccount);
        if mIdx > -1 then
        begin
          Member := List[mIdx];
          case PNsmMemberInfo(lParam)^.nInfoKey of
          NMMI_ACCOUNT:
              Result := StrToNsmInfo(PNsmMemberInfo(lParam)^.lpInfo, Member.Account);
          NMMI_NAME:
              Result := WideStrToNsmInfo(PNsmMemberInfo(lParam)^.lpInfo, Member.Name);
          NMMI_STATUS:
              Result := IntToNsmInfo(PNsmMemberInfo(lParam)^.lpInfo, Member.Status);
          NMMI_STATUSSTRING:
              Result := WideStrToNsmInfo(PNsmMemberInfo(lParam)^.lpInfo, Member.StatusStr);
          NMMI_BUSYNESS:
              Result := IntToNsmInfo(PNsmMemberInfo(lParam)^.lpInfo, Member.Busyness);
          NMMI_GROUPID:
              Result := IntToNsmInfo(PNsmMemberInfo(lParam)^.lpInfo, Member.Group);
          end;
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
end;

//  wParam  : Ώ HNsmConnection
//  lParam  : PEnumMemberInfo
//  Return  : 0
function DoEnumMembers(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, I: Integer;
  Member: TNsmMember;
  List: TNsmMemberList;
begin
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
      begin
        case PEnumMemberInfo(lParam).nListKind of
          NMLK_FORWARDLIST : List := Connection[cIdx].Members;
          NMLK_REVERSELIST : List := Connection[cIdx].ReverseMembers;
          NMLK_ALLOWLIST   : List := Connection[cIdx].AllowMembers;
          NMLK_BLOCKLIST   : List := Connection[cIdx].BlockMembers;
        else
          List := Connection[cIdx].Members;
        end;
        for I := 0 to List.Count - 1 do
        begin
          Member := List[I];
          TEnumMemberCallback(PEnumMemberInfo(lParam).lpCallBackProc)(PChar(Member.Account), PEnumMemberInfo(lParam).nData);
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  Result := 0;
end;

function DoBeginUpdate(wParam, lParam: Cardinal): Integer; StdCall;
begin
  Result := 0;
end;

function DoEndUpdate(wParam, lParam: Cardinal): Integer; StdCall;
begin
  Result := 0;
end;

// Group -----------------------------------------------------------------------

//  wParam  : ConnectionID
//  lParam  : GroupID
//  Return  :  0 ȊOԂ
function DoAddGroup(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      Idx := IndexOf(TNsmConnection(wParam));
      if (Idx > -1) and
         (Connection[Idx].Groups.IndexOf(lParam) = -1) then
      begin
        with Connection[Idx].Groups.Add do
        begin
          Id := lParam;
          Name := IntToStr(lParam);
        end;
        Result := 1;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Result <> 0 then
    NsmCom.NotifyEvent(NsmSystem.HConnectionGroupsAdd, wParam, lParam);
end;

//  wParam  : ConnectionID
//  lParam  : GroupID
//  Return  : s 0  1
function DoRemoveGroup(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, gIdx: Integer;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
      begin
        gIdx := Connection[cIdx].Groups.IndexOf(lParam);
        if gIdx > -1 then
        begin
          Connection[cIdx].Groups.Delete(gIdx);
          Result := 1;
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Result = 1 then
    NsmCom.NotifyEvent(NsmSystem.HConnectionGroupsRemove, wParam, lParam);
end;

//  wParam  : HConnection
//  lParam  : PNsmGroupInfo
//  Return  : s 0  1
function DoSetGroupInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, gIdx: Integer;
  Group: TNsmGroup;
  GroupInfo: TNsmGroupInfo;
begin
  Result := 0;
  GroupInfo := PNsmGroupInfo(lParam)^;
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
      begin
        gIdx := Connection[cIdx].Groups.IndexOf(GroupInfo.nGroupId);
        if gIdx > -1 then
        begin
          Group := Connection[cIdx].Groups[gIdx];
          case PNsmGroupInfo(lParam)^.nInfoKey of
          NMGI_ID:        Group.Id  := NsmInfoToInt(GroupInfo.lpInfo);
          NMGI_NAME:      Group.Name := NsmInfoToWideStr(GroupInfo.lpInfo);
          NMGI_EXPANDED:  Group.Expanded := NsmInfoToBool(GroupInfo.lpInfo);
          end;
          Result := 1;
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if (Result = 1) and (GroupInfo.nFlags and NMIF_NOCHANGEEVENT = 0) then
    NsmCom.NotifyEvent(NsmSystem.HConnectionGroupsInfoChange, wParam, lParam);
end;

//  wParam  : ConnectionID
//  lParam  : PNsmGroupInfo
//  Return  : ݒ肵oCg
function DoGetGroupInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, gIdx: Integer;
  Group: TNsmGroup;
begin
  Result := 0;
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
      begin
        gIdx := Connection[cIdx].Groups.IndexOf(PNsmGroupInfo(lParam).nGroupId);
        if gIdx > -1 then
        begin
          Group := Connection[cIdx].Groups[gIdx];
          case PNsmGroupInfo(lParam)^.nInfoKey of
          NMGI_ID:
              Result := IntToNsmInfo(PNsmGroupInfo(lParam)^.lpInfo, Group.Id);
          NMGI_NAME:
              Result := WideStrToNsmInfo(PNsmGroupInfo(lParam)^.lpInfo, Group.Name);
          NMGI_EXPANDED:
              Result := BoolToNsmInfo(PNsmGroupInfo(lParam)^.lpInfo, Group.Expanded);
          end;
        end;
      end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
end;

//  wParam  : HNsmConnection
//  lParam  : PEnumGroupInfo
//  Return  : 0
function DoEnumGroups(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx, I: Integer;
  Group: TNsmGroup;
begin
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
        for I := 0 to Connection[cIdx].Groups.Count - 1 do
        begin
          Group := Connection[cIdx].Groups[I];
          TEnumGroupCallback(PEnumGroupInfo(lParam).lpCallBackProc)(Group.Id, PEnumGroupInfo(lParam).nData);
        end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  Result := 0;
end;

//  wParam  : eHNsmConnection
//  lParam  : 0
//  Return  : 0
function DoBeginGroupsUpdate(wParam, lParam: Cardinal): Integer; StdCall;
begin
  Result := 0;
end;

//  wParam  : eHNsmConnection
//  lParam  : 0
//  Return  : 0
function DoEndGroupsUpdate(wParam, lParam: Cardinal): Integer; StdCall;
begin
  Result := 0;
end;

// Session ---------------------------------------------------------------------

//  wParam  : eHNsmConnection
//  lParam  : 0
//  Return  :  HNsmSession As 0
function DoCreateSession(wParam, lParam: Cardinal): Integer; StdCall;
var
  cIdx: Integer;
begin
  Result := Integer(nil);
  with NsmSystem.Connections.LockList do
    try
      cIdx := IndexOf(TNsmConnection(wParam));
      if cIdx > -1 then
        with NsmSystem.Sessions.LockList do
          try
            Result := Integer(Add(TNsmConnection(wParam)));
          finally
            NsmSystem.Sessions.UnlockList;
          end;
    finally
      NsmSystem.Connections.UnlockList;
    end;
  if Assigned(Pointer(Result)) then
  begin
    NsmCom.NotifyEvent(NsmSystem.HSessionOpen, Result, 0);
    NsmSystem.DoSessionChange;
  end;
end;

//  wParam  : 폜 HNsmSession
//  lParam  : 0
//  Return  :  -1 ȊO̐l
function DoDeleteSession(wParam, lParam: Cardinal): Integer; StdCall;
begin
  with NsmSystem.Sessions.LockList do
    try
      Result := IndexOf(TNsmSession(wParam));
      if Result > -1 then
        Delete(Result);
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  if Result > -1 then
  begin
    NsmCom.NotifyEvent(NsmSystem.HSessionClose, wParam, 0);
    NsmSystem.DoSessionChange;
  end;
end;

//  wParam  : Ώ HNsmSession
//  lParam  : PSessionInfo
//  Return  : ۂ lpBuffer Ɋi[̃TCY
function DoGetSessionInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
  ASession: TNsmSession;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      Idx := IndexOf(TNsmSession(wParam));
      if Idx > -1 then
      begin
        ASession := TNsmSession(wParam);
        case PNsmSessionInfo(lParam)^.nInfoKey of
        NMSI_CAPTION:
          Result := WideStrToNsmInfo(PNsmSessionInfo(lParam)^.lpInfo, ASession.Caption);
        NMSI_STATUS:
          Result := IntToNsmInfo(PNsmSessionInfo(lParam)^.lpInfo, ASession.Status);
        NMSI_CONNECTION:
          Result := IntToNsmInfo(PNsmSessionInfo(lParam)^.lpInfo, Integer(ASession.Connection));
        NMSI_PROTOCOL:
          Result := StrToNsmInfo(PNsmSessionInfo(lParam)^.lpInfo, ASession.Protocol);
        end;
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
end;

//  wParam : Ώ HNsmSession
//  lParam : PSessionInfo
//  Return :  0 ȊOԂ
function DoSetSessionInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
  ASession: TNsmSession;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      Idx := IndexOf(TNsmSession(wParam));
      if Idx > -1 then
      begin
        ASession := TNsmSession(wParam);
        case PNsmSessionInfo(lParam)^.nInfoKey of
        NMSI_CAPTION :
            ASession.Caption := NsmInfoToWideStr(PNsmSessionInfo(lParam)^.lpInfo);
        NMSI_STATUS:
            ASession.Status := NsmInfoToInt(PNsmSessionInfo(lParam)^.lpInfo);
        NMSI_CONNECTION:
          ASession.Connection := TNsmConnection(NsmInfoToInt(PNsmSessionInfo(lParam)^.lpInfo));
        NMSI_PROTOCOL:
          ASession.Protocol := NsmInfoToStr(PNsmSessionInfo(lParam)^.lpInfo);
        end;
        Result := 1;
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  if Result = 1 then
  begin
    NsmCom.NotifyEvent(NsmSystem.HSessionInfoChange, wParam, lParam);
    NsmSystem.DoSessionChange;
  end;
end;

//  wParam  : R[obN֐ւ̃|C^
//  lParam  : R[obN֐ɓnCӃf[^
//  Return  : 0
function DoEnumSessions(wParam, lParam: Cardinal): Integer; StdCall;
var
  I: Integer;
begin
  with NsmSystem.Sessions.LockList do
    try
      for I := 0 to Count - 1 do
        TEnumSessionCallback(wParam)(Integer(Session[I]), lParam);
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  Result := 0;
end;

//  wParam  : ZbVʎq
//  lParam  : PChar õAJEg
//  Return  :  0 ȊOԂ
function DoAddSessionMember(wParam, lParam: Cardinal): Integer; StdCall;
var
  Idx: Integer;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      Idx := IndexOf(TNsmSession(wParam));
      if (Idx > -1) and
         (Session[Idx].Members.IndexOfAccount(PChar(lParam)) = -1) then
      begin
        with Session[Idx].Members.Add(PChar(lParam)) do
        begin
          Name := PChar(lParam);
          Status := NMST_OFFLINE;
        end;
        Result := 1;
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  if Result <> 0 then
    NsmCom.NotifyEvent(NsmSystem.HSessionMembersAdd, wParam, lParam);
end;

//  wParam  : ZbVʎq
//  lParam  : PChar õAJEg
//  Return  :  0 ȊOԂ
function DoRemoveSessionMember(wParam, lParam: Cardinal): Integer; StdCall;
var
  sIdx, mIdx: Integer;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      sIdx := IndexOf(TNsmSession(wParam));
      if sIdx > -1 then
      begin
        mIdx := Session[sIdx].Members.IndexOfAccount(PChar(lParam));
        if mIdx > -1 then
        begin
          Session[sIdx].Members.Delete(mIdx);
          Result := 1;
        end;
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  if Result = 1 then
    NsmCom.NotifyEvent(NsmSystem.HSessionMembersRemove, wParam, lParam);
end;

//  wParam  : SessionID
//  lParam  : PMemberInfo
//  Return  :  0 ȊOԂ
function DoSetSessionMemberInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  sIdx, mIdx: Integer;
  Member: TNsmMember;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      sIdx := IndexOf(TNsmSession(wParam));
      if sIdx > -1 then
      begin
        mIdx := Session[sIdx].Members.IndexOfAccount(PNsmMemberInfo(lParam).lpAccount);
        if mIdx > -1 then
        begin
          Member := Session[sIdx].Members[mIdx];
          case PNsmMemberInfo(lParam)^.nInfoKey of
//          NMMI_ACCOUNT:   Member.Account  := NsmInfoToStr(PNsmMemberInfo(lParam).lpInfo);
          NMMI_NAME:      Member.Name     := NsmInfoToWideStr(PNsmMemberInfo(lParam).lpInfo);
          NMMI_STATUS:    Member.Status   := NsmInfoToInt(PNsmMemberInfo(lParam).lpInfo);
          NMMI_STATUSSTRING:
                          Member.StatusStr := NsmInfoToWideStr(PNsmMemberInfo(lParam).lpInfo);
          NMMI_BUSYNESS:  Member.Busyness  := NsmInfoToInt(PNsmMemberInfo(lParam).lpInfo);
          end;
          Result := 1;
        end;
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  if Result = 1 then
    NsmCom.NotifyEvent(NsmSystem.HSessionMembersInfoChange, wParam, lParam);
end;

//  wParam  : SessionID
//  lParam  : PMemberInfo
//  Return  :  0 ȊOԂ
function DoGetSessionMemberInfo(wParam, lParam: Cardinal): Integer; StdCall;
var
  sIdx, mIdx: Integer;
  Member: TNsmMember;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      sIdx := IndexOf(TNsmSession(wParam));
      if sIdx > -1 then
      begin
        mIdx := Session[sIdx].Members.IndexOfAccount(PNsmMemberInfo(lParam).lpAccount);
        if mIdx > -1 then
        begin
          Member := Session[sIdx].Members[mIdx];
          case PNsmMemberInfo(lParam)^.nInfoKey of
          NMMI_ACCOUNT:
              Result := StrToNsmInfo(PNsmMemberInfo(lParam).lpInfo, Member.Account);
          NMMI_NAME:
              Result := WideStrToNsmInfo(PNsmMemberInfo(lParam).lpInfo, Member.Name);
          NMMI_STATUS:
              Result := IntToNsmInfo(PNsmMemberInfo(lParam).lpInfo, Member.Status);
          NMMI_STATUSSTRING:
              Result := WideStrToNsmInfo(PNsmMemberInfo(lParam).lpInfo, Member.StatusStr);
          NMMI_BUSYNESS:
              Result := IntToNsmInfo(PNsmMemberInfo(lParam).lpInfo, Member.Busyness);
          end;
        end;
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
end;

function DoBeginSessionMembersUpdate(wParam, lParam: Cardinal): Integer; StdCall;
begin
  Result := 0;
end;

function DoEndSessionMembersUpdate(wParam, lParam: Cardinal): Integer; StdCall;
begin
  Result := 0;
end;

function DoEnumSessionMembers(wParam, lParam: Cardinal): Integer; StdCall;
var
  sIdx, I: Integer;
  Member: TNsmMember;
begin
  with NsmSystem.Sessions.LockList do
    try
      sIdx := IndexOf(TNsmSession(wParam));
      if sIdx > -1 then
        for I := 0 to Session[sIdx].Members.Count - 1 do
        begin
          Member := Session[sIdx].Members[I];
          TEnumMemberCallback(PEnumMemberInfo(lParam)^.lpCallBackProc)
            (PChar(Member.Account), PEnumMemberInfo(lParam)^.nData);
        end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  Result := 0;
end;

//  wParam  : SessionID
//  lParam  : LPMESSAGEINFO
//  Return  :  0 ȊOԂ
function DoSendMessage(wParam, lParam: Cardinal): Integer; StdCall;
var
  sIdx: Integer;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      sIdx := IndexOf(TNsmSession(wParam));
      if sIdx > -1 then
      begin
        NsmCom.CallService(NsmCom.GetService(
            Format(NMS_PROTOCOL_SESSION_SENDMESSAGE, [Session[sIdx].Protocol])),
            wParam, lParam);
        Result := 1;
      end;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  if Result <> 0 then
    NsmCom.NotifyEvent(NsmSystem.HSessionSendMessage, wParam, lParam);
end;

//  wParam  : SessionID
//  lParam  : LPMESSAGEINFO
//  Return  :  0 ȊOԂ
function DoReceiveMessage(wParam, lParam: Cardinal): Integer; StdCall;
var
  sIdx: Integer;
begin
  Result := 0;
  with NsmSystem.Sessions.LockList do
    try
      sIdx := IndexOf(TNsmSession(wParam));
      if sIdx > -1 then
        Result := 1;
    finally
      NsmSystem.Sessions.UnlockList;
    end;
  if Result <> 0 then
    NsmCom.NotifyEvent(NsmSystem.HSessionReceiveMessage, wParam, lParam);
end;

// -----------------------------------------------------------------------------

// String -> TNsmInfo
function StrToNsmInfo(NsmInfo: PNsmInfo; S: String): Integer;
var
  L: Integer;
begin
  with NsmInfo^ do
  begin
    if nBufferSize - 1 < Length(S) then
      L := nBufferSize - 1
    else
      L := Length(S);
    nType := NMIT_STRING;
    CopyMemory(lpBuffer, PChar(S), L);
    PChar(lpBuffer)[L] := #0;
    Result := L + 1;
  end;
end;

// WideString -> TNsmInfo
function WideStrToNsmInfo(NsmInfo: PNsmInfo; S: WideString): Integer;
var
  L: Integer;
begin
  with NsmInfo^ do
  begin
    if nBufferSize - 2 < Length(S) * 2 then
      L := nBufferSize - 2
    else
      L := Length(S) * 2;
    nType := NMIT_WIDESTRING;
    CopyMemory(lpBuffer, PWideChar(S), L);
    CopyMemory(@PChar(lpBuffer)[L], PChar(#0#0), 2);
    Result := L + 2;
  end;
end;

// Integer -> TNsmInfo
function IntToNsmInfo(NsmInfo: PNsmInfo; Value: Integer): Integer;
begin
  Result := 0;
  if NsmInfo^.nBufferSize < SizeOf(Integer) then
    Exit;
  with NsmInfo^ do
  begin
    nType := NMIT_INTEGER;
    CopyMemory(lpBuffer, @Value, SizeOf(Integer));
    Result := SizeOf(Integer);
  end;
end;

// Boolean -> TNsmInfo
function BoolToNsmInfo(NsmInfo: PNsmInfo; Value: Boolean): Integer;
begin
  Result := 0;
  if NsmInfo^.nBufferSize < SizeOf(Boolean) then
    Exit;
  with NsmInfo^ do
  begin
    nType := NMIT_BOOL;
    CopyMemory(lpBuffer, @Value, SizeOf(Boolean));
    Result := SizeOf(Boolean);
  end;
end;

// TNsmInfo -> String
function NsmInfoToStr(NsmInfo: PNsmInfo): String;
begin
  Result := '';
  if NsmInfo^.nType = NMIT_STRING then
  begin
    Result := PChar(NsmInfo^.lpBuffer);
  end else
  if NsmInfo^.nType = NMIT_WIDESTRING then
  begin
    Result := String(PWideChar(NsmInfo^.lpBuffer));
  end;
end;

// TNsmInfo -> WideString
function NsmInfoToWideStr(NsmInfo: PNsmInfo): WideString;
begin
  Result := '';
  if NsmInfo^.nType = NMIT_WIDESTRING then
  begin
    Result := PWideChar(NsmInfo^.lpBuffer);
  end else
  if NsmInfo^.nType = NMIT_STRING then
  begin
    Result := WideString(PChar(NsmInfo^.lpBuffer));
  end;
end;

// TNsmInfo -> Integer
function NsmInfoToInt(NsmInfo: PNsmInfo): Integer;
begin
  Result := 0;
  if NsmInfo^.nType = NMIT_INTEGER then
  begin
    Result := PInteger(NsmInfo^.lpBuffer)^;
  end;
end;

// TNsmInfo -> Boolean
function NsmInfoToBool(NsmInfo: PNsmInfo): Boolean;
begin
  Result := False;
  if NsmInfo^.nType = NMIT_BOOL then
  begin
    Result := PBoolean(NsmInfo^.lpBuffer)^;
  end;
end;
// -----------------------------------------------------------------------------

initialization
  NsmSystem := TNsmSystem.Create;

finalization
  NsmSystem.Free;

end.
