/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.db.expr;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;

import net.morilib.db.delay.Delay;
import net.morilib.db.engine.SqlEngine;
import net.morilib.db.relations.RelationAggregate;
import net.morilib.db.relations.RelationTuple;
import net.morilib.db.schema.SqlSchema;

public class RelationBinaryOperate extends RelationExpression {

	//
	private RelationBinaryOperator op;
	private RelationExpression pa, pb;

	public RelationBinaryOperate(RelationBinaryOperator o,
			RelationExpression a, RelationExpression b) {
		op = o;
		pa = a;  pb = b;
	}

	@Override
	public Object eval(SqlEngine v, SqlSchema f,
			RelationTuple tuple, RelationAggregate m,
			List<String> group,
			List<Object> h) throws IOException, SQLException {
		final Object a, b;

		a = pa.eval(v, f, tuple, m, group, h);
		b = pb.eval(v, f, tuple, m, group, h);
		if(a instanceof Delay) {
			return a;
		} else if(b instanceof Delay) {
			return b;
		} else {
			return op.op(a, b);
		}
	}

	@Override
	public boolean isAggregate() throws SQLException {
		return pa.isAggregate() || pb.isAggregate();
	}

	@Override
	public Object init(SqlEngine v, SqlSchema f) throws SQLException {
		final Object a, b;
		Delay d;

		a = pa.init(v, f);
		b = pb.init(v, f);
		if(a instanceof Delay && b instanceof Delay) {
			d = new Delay() {

				@Override
				public Object force() throws SQLException {
					return op.op(((Delay)a).force(), ((Delay)b).force());
				}

				@Override
				public void add(Object... args) throws SQLException {
					((Delay)a).add(args);
					((Delay)b).add(args);
				}

			};
			return d;
		} else if(a instanceof Delay) {
			d = new Delay() {

				@Override
				public Object force() throws SQLException {
					return op.op(((Delay)a).force(), b);
				}

				@Override
				public void add(Object... args) throws SQLException {
					((Delay)a).add(args);
				}

			};
			return d;
		} else if(b instanceof Delay) {
			d = new Delay() {

				@Override
				public Object force() throws SQLException {
					return op.op(a, ((Delay)b).force());
				}

				@Override
				public void add(Object... args) throws SQLException {
					((Delay)b).add(args);
				}

			};
			return d;
		} else {
			return op.op(a, b);
		}
	}

}
