/*
 * Header    :
 * File      : RescueObjectPool.cxx
 * Auther    : Kosuke Shinoda
 * Since     : 2001/12/26
 * Time-stamp: <2002-02-03 18:21:56 kshinoda>
 * Comment   : based source in RoboCupRescue Kernel source (C)koto
 * End       :
 */
/*
 * Copyright (C) 2001,2002 SHINODA, Kosuke. Jaist,Japan & CARC, AIST, JAPAN
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 */

#include "../librescue/librescue.hxx"
#include "RescueObjectPool.hxx"

namespace Rescue {

  /////////////////////////////////////////////////////////////////////////
  // RescueObjectPool
  RescueObjectPool::RescueObjectPool(){
    m_meshXBase = 0;
    m_meshYBase = 0;
    m_meshXSize = 100;
    m_meshYSize = 100;
    m_meshXCount = 1;
    m_meshYCount = 1;
    m_mesh.resize(m_meshYCount);
    for(int i=0; i<m_meshYCount; i++)
      m_mesh[i].resize(m_meshXCount);
  }
  void RescueObjectPool::add(Object* newObject){
    newObject->setOwner(this);
    ObjectPool::add(newObject);
    if(dynamic_cast<RealObject*>(newObject) != 0)
      m_movingObjects.push_back(newObject);
  }

  RescueObjectPool::~RescueObjectPool() {
  }

  void RescueObjectPool::addFiredObject(Object* object) {
    m_firedObjects.push_back(object);
  }
  void RescueObjectPool::clearFiredObjects() {
    m_firedObjects.clear();
  }
    
  const Objects& RescueObjectPool::firedObjects() const {
    return m_firedObjects;
  }

  //------------------------------------------------------------
  // Class : RescueObjectPool::Enumerator 
  //--------------------------------------------------
  // RescueObjectPool::Enumerator Constructor
  RescueObjectPool::Enumerator::Enumerator(RescueObjectPool* owner, S32 x, S32 y, S32 r){
    m_owner = owner;
    m_centerX = x;
    m_centerY = y;
    m_squaredRadius = (double)r * r;
        
    S32 tmp = x - r;
    m_xMin = (tmp - owner->m_meshXBase) / owner->m_meshXSize;
    if(m_xMin >= owner->m_meshXCount)   m_xMin = owner->m_meshXCount - 1;
    if(m_xMin < 0)                      m_xMin = 0;
    
    tmp = x + r;
    m_xMaz = (tmp - owner->m_meshXBase) / owner->m_meshXSize + 1;
    if(m_xMaz >= owner->m_meshXCount)       m_xMaz = owner->m_meshXCount - 1;
    if(m_xMaz < 0)                      m_xMaz = 0;
    
    tmp = y - r;
    m_yMin = (tmp - owner->m_meshYBase) / owner->m_meshYSize;
    if(m_yMin >= owner->m_meshYCount)       m_yMin = owner->m_meshYCount - 1;
    if(m_yMin < 0)                      m_yMin = 0;
    
    tmp = y + r;
    m_yMaz = (tmp - owner->m_meshYBase) / owner->m_meshYSize + 1;
    if(m_yMaz >= owner->m_meshYCount)       m_yMaz = owner->m_meshYCount - 1;
    if(m_yMaz < 0)                      m_yMaz = 0;
        
    m_x = m_xMin;
    m_y = m_yMin;
    m_it = m_owner->m_mesh[m_y][m_x].begin();
    
    //printf("(%d, %d) - (%d, %d)\n", m_xMin, m_yMin, m_xMaz, m_yMaz);
  }
  
  //--------------------------------------------------
  // RescueObjectPool::Enumerator nextCore()
  Object* RescueObjectPool::Enumerator::nextCore() {
    for(;;) {
      if(m_x < 0) {
	if(m_y >= 0) {
	  if(m_it != m_owner->m_movingObjects.end())
	    return *m_it++;
	  m_y = -1;
	}
	return 0;
      }
      if(m_it != m_owner->m_mesh[m_y][m_x].end())
	return *m_it++;
      if(++m_x >= m_xMaz) {
	m_x = m_xMin;
	if(++m_y >= m_yMaz) {
	  m_x = -1;
	  m_y = 0;
	  m_it = m_owner->m_movingObjects.begin();
	  continue;
	}
      }
      m_it = m_owner->m_mesh[m_y][m_x].begin();
    }
  }
  
  //--------------------------------------------------
  // RescueObjectPool::Enumerator next()
  Object* RescueObjectPool::Enumerator::next() {
    Object* result;
    while(result = nextCore(), result != 0) {
      ASSERT(dynamic_cast<RealObject*>(result) != 0);
      RealObject* ro = (RealObject*)result;
      double x = ro->x() - m_centerX;
      double y = ro->y() - m_centerY;
      if(x*x + y*y <= m_squaredRadius)
	break;
    }
    return result;
  }
  
  //--------------------------------------------------
  // RescueObjectPool::Enumerator objectsInRound
  RescueObjectPool::Enumerator RescueObjectPool::objectsInRound(S32 x, S32 y, S32 r) {
    return Enumerator(this, x, y, r);
  }

    
  /////////////////////////////////////////////////////////////////////////
  // RescueObjectPool

  void RescueObjectPool::setUpMesh(int meshX, int meshY) {
    m_meshXCount = meshX;
    m_meshYCount = meshY;
    
    m_movingObjects.clear();
    m_mesh.clear();
    m_mesh.resize(m_meshYCount);
    for(int i=0; i<m_meshYCount; i++)
      m_mesh[i].resize(m_meshXCount);
    
    S32 xMin = S32_MAX;
    S32 yMin = S32_MAX;
    S32 xMax = S32_MIN;
    S32 yMax = S32_MIN;
    Objects::const_iterator it = objects().begin();
    for(; it != objects().end(); it++) {
      RealObject* ro = dynamic_cast<RealObject*>(*it);
      if(ro != 0) {
	S32 x = ro->x();
	if(x < xMin)
	  xMin = x;
	if(x > xMax)
	  xMax = x;
	S32 y = ro->y();
	if(y < yMin)
	  yMin = y;
	if(y > yMax)
	  yMax = y;
      }
    }

    m_meshXBase = xMin;
    m_meshYBase = yMin;
    m_meshXSize = (xMax - xMin) / m_meshXCount + 1;
    m_meshYSize = (yMax - yMin) / m_meshYCount + 1;
    it = objects().begin();
    for(; it != objects().end(); it++) {
      Object* o = *it;
      if(dynamic_cast<RealObject*>(o) != 0) {
	MotionlessObject* mlo = dynamic_cast<MotionlessObject*>(o);
	if(mlo == 0) {
	  m_movingObjects.push_back(o);
	} else {
	  int x = (mlo->x() - m_meshXBase) / m_meshXSize;
	  if(x >= m_meshXCount)   x = m_meshXCount - 1;
	  if(x < 0)               x = 0;
	  int y = (mlo->y() - m_meshYBase) / m_meshYSize;
	  if(y >= m_meshYCount)   y = m_meshYCount - 1;
	  if(y < 0)               y = 0;
	  m_mesh[y][x].push_back(mlo);
	}
      }
    }
  }
}
