// -*- Mode: c++ -*-
//Header:
//File: Buffer.h
//Author: NODA, Itsuki
//Date: 1999/05/18
//

//ModifyHistory:
// 1999/05/18: Start to create this file
// 1999/11/07: introduce Itk namespace
// 1999/11/07: Divide each class to separate file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_Buffer_h_
#define _itk_Buffer_h_
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <strstream>

#include "itk/btype.h"
#include "itk/SString.h"

namespace Itk {
   
   //======================================================================
   // Buffer
   //

   /*--------------------*/
   /**
    * String Buffer for repeated I/O purpose.
    *
    * This class consists of original buffer area (SubString original),
    * reader-head (char* head) and writer-head (char* tail).
    * The pair of reader-/writer-head (head and tail) is also a SubString.
    */

   class Buffer : public SubString {

	 /*--------------------*/
	 /**
	  * original buffer area.
	  */
      public:
	 SString original ;

	 //--------------------------------------------------
	 // constructor
      public:

	 Buffer(const UInt size = 0) { alloc(size) ; } ;
	 Buffer(SubString& org) { set(org) ; } ;

	 //--------------------------------------------------
	 // alloc
	 /**
	  * allocate specifiled size l.
	  *
	  * If forcep = True, then it allocate even if the more size
	  * has been allocated for this buffer.
	  */
      public:
	 Bool alloc(const UInt l, Bool forcep = True) ;

	 //--------------------------------------------------
	 // rewind
	 /**
	  * clear and rewind the buffer.
	  * This reset the read and write pointer to the top of the
	  * buffer.
	  */
      public:
	 void clear() {
	    SubString::set(original.head, original.head) ;
	 } ;

	 /*--------------------*/
	 /**
	  * rewind only read pointer.
	  */
      public:
	 void rewindRead() { head = original.head ; } ;

	 /*--------------------*/
	 /**
	  * rewind only write pointer.
	  */
      public:
	 void rewindWrite() { clear() ; } ;

	 //--------------------------------------------------
	 // size
	 /**
	  * get the size of the allocated area.
	  */
      public:
	 UInt size() const { return original.size() ; } ;

	 //--------------------------------------------------
	 // operator=
	 /**
	  * assign the value from the source. 
	  * (see set(const Buffer&))
	  */
      public:
	 Buffer& operator=(const Buffer& src) {  return set(src) ; } ;

	 //--------------------------------------------------
	 // set
	 /**
	  * set the value from the src.
	  */
      public:
	 Buffer& set(const Buffer& src) ;

	 /*--------------------*/
	 /**
	  * set the value from the src.
	  */
      public:
	 Buffer& set(const SubString& src) ;

	 //--------------------------------------------------
	 // used/body/rest
	 /**
	  * get used area as a SubString.
	  * The used area is from the original-head to the reader-head.
	  */
      public:
	 SubString used() const {
	    return SubString(original.head,head) ;
	 } ;

	 /*--------------------*/
	 /**
	  * get length of used area.
	  * (see used())
	  */
      public:
	 UInt usedLength() const {
	    return used().length() ;
	 } ;

	 /*--------------------*/
	 /**
	  * get body area as a SubString.
	  * The body area is from the reader-head to writer-head.
	  */
      public:
	 SubString body() const {
	    return SubString(head,tail) ;
	 } ;

	 /*--------------------*/
	 /**
	  * get the length of the body area.
	  * (see body()).
	  */
      public:
	 UInt bodyLength() const {
	    return length() ;
	 } ;

	 /*--------------------*/
	 /**
	  * get rest area as a SubString.
	  * The rest area is from the writer-head to the original-tail.
	  */
      public:
	 SubString rest() const {
	    return SubString(tail,original.tail) ;
	 } ;

	 /*--------------------*/
	 /**
	  * get the length of the rest area.
	  * (see rest()).
	  */
      public:
	 UInt restLength() const {
	    return rest().length() ;
	 } ;

	 //--------------------------------------------------
	 // safety put
	 /**
	  * put a character safely.
	  * If the buffer is overflow, then it allocates new wide area
	  * and copy the original to the new area, and add the character c.
	  */
      public:
	 char putCharSafe(const char c) ;

	 //--------------------------------------------------
	 // convert to strstream
	 /**
	  * make a new istrstream for the buffer.
	  */
      public:
	 istrstream * newIStrStream() {
	    return new istrstream((const char*)head,bodyLength()) ;
	 } ;
	 
	 /*--------------------*/
	 /**
	  * destroy the istrstream developped by newIStrStream().
	  */
      public:
	 void deleteIStrStream(istrstream * istr) {
	    delete istr ;
	    return ;
	 } ;

	 /*--------------------*/
	 /**
	  * make a new ostrstream for the buffer.
	  */
      public:
	 ostrstream * newOStrStream() { // dangeourous.  should not use
	    return new ostrstream(tail,restLength()) ;
	 } ;
	 
	 /*--------------------*/
	 /**
	  * destroy the istrstream developped by newOStrStream().
	  */
      public:
	 void deleteOStrStream(ostrstream * ostr) {
	    ostr->freeze() ;
	    Int n = ostr->pcount() ;
	    delete ostr ;
	    setLength(n) ;
	    terminate() ;
	    return ;
	 } ;

	 //--------------------------------------------------
	 // describe
	 /**
	  * describe the buffer.
	  */
      public:
	 void describe(ostream& o, Bool detailp = True) const {
	    if(detailp) {
	       o << "#Buffer[<" << (void * const)(original.head) << ">-<" 
		 << (void * const)(original.tail)
		 << ">(len=" << length() << "):" 
		 << "hp=<" << (void * const)head << ">,tp=<" 
		 << (void * const)tail << ">:" ;
	    } 
	    if(isNull(original.head)) {
	       o << "(null)" ;
	    } else {
	       for(char* s = head ; s < tail ; s++) {
		  o << *s ;
	       }
	    }
	    if(detailp) {
	       o << "]" << endl ;
	    }
	 } ;

	 friend ostream& operator<< (ostream& o, const Buffer& b) {
	    b.describe(o,False) ;
	    return o ;
	 } ;

   } ;
} ;


//////////////////////////////////////////////////////////////////////
#endif
