// -*- Mode:C++ -*-
//Header:
//File: Geo2D.h
//Author: NODA, Itsuki
//Date: 2001/03/05
//

//ModifyHistory:
// 2001/03/05: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_Geo2D_h_
#define _itk_Geo2D_h_
////////////////////////////////////////////////////////////////////////

#include "itk/geoutility.h"

//======================================================================
// Itk namespace :: Geo2D namespace
//----------------------------------------------------------------------
namespace Itk {
   namespace Geo2D {

      /*--------------------*/
      /**
       *
       */

      typedef GeoVal Val ;

      //----------------------------------------------------------------------
      // Vector

      /*--------------------*/
      /**
       *
       */

      class Vec { 
	 public:
	    Val x ;
	    Val y ;

	    //--------------------------------------------------
	    // constructor
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec() { x = 0.0 ; y = 0.0 ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec(const Val & vx, const Val & vy) { x = vx ; y = vy ;  } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec(const Vec & v) { x = v.x ; y = v.y ; } ;

	    //--------------------------------------------------
	    // assign
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec & operator = (const Vec & v) { return set(v) ;} ;

	    //--------------------------------------------------
	    // inc/dec/mul/div
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec & operator += (const Vec & v) { 
	       x += v.x ; y += v.y ; 
	       return *this ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec & operator -= (const Vec & v) { 
	       x -= v.x ; y -= v.y ; 
	       return *this ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec & operator *= (const Val & a) { 
	       x *= a ; y *= a ;
	       return *this ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec & operator /= (const Val & a) ; 

	    //--------------------------------------------------
	    // add/sub/mul/div/neg 
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec operator +(const Vec &v) const { return Vec(x+v.x, y+v.y); };

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec operator -(const Vec &v) const { return Vec(x-v.x, y-v.y); };

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec operator *(const Val &a) const { return Vec(x*a,   y*a  ); };

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec operator *(const Vec &a) const { return Vec(x*a.x, y*a.y); };

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec operator /(const Val &a) const ; 

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec operator /(const Vec &a) const ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec operator -()             const { return Vec(-x,    -y   ); };

	    //--------------------------------------------------
	    // diff
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec diffTo(const Vec & dst) const {
	       return dst - *this ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec diffFrom(const Vec & org) const {
	       return *this - org ;
	    } ;

	    //--------------------------------------------------
	    // norm
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Val sqnorm() const { return sq(x) + sq(y) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Val norm() const { return sqrt(sqnorm()) ; } ;

	    //--------------------------------------------------
	    // distance
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Val distance(const Vec & org) const {
	       return diffFrom(org).norm() ;
	    } ;

	    //--------------------------------------------------
	    // inter product
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Val innerProd(const Vec & ref) const {
	       return x * ref.x + y * ref.y ;
	    } ;

	    //--------------------------------------------------
	    // unitVector / normalize
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec unit(Val unit = 1.0) const {
	       Vec v = *this ;
	       return v.normalize(unit) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec & normalize(Val unit = 1.0) {
	       Val l = norm() ;
	       if (l < ITK_EPS) return *this ;
	       *this *= (unit/l) ;
	       return *this ;
	    } ;

	    //--------------------------------------------------
	    // Angle
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angle() const { return ATan(y,x) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angle(const Angle & ref) const {
	       return normalizeAngle(angle() - ref) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleRad() const { return atan2(y,x) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleRad(const Angle & ref) const {
	       return normalizeAngleRad(angleRad() - ref) ;
	    } ;

	    //--------------------------------------------------
	    // Angle to a point
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleTo(const Vec & dst, const Angle & ref) const {
	       return diffTo(dst).angle(ref) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleTo(const Vec & dst) const { return angleTo(dst,0) ; } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleTo(const Vec & dst, const Vec & ref) const {
	       return angleTo(dst,angleTo(ref)) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleFrom(const Vec & org, const Angle & ref) const {
	       return diffFrom(org).angle(ref) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleFrom(const Vec & dst) const { return angleFrom(dst,0);};


	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleFrom(const Vec & org, const Vec & ref) const {
	       return angleFrom(org,org.angleTo(ref)) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleToRad(const Vec & dst, const Angle & ref) const {
	       return diffTo(dst).angleRad(ref) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleToRad(const Vec & dst) const { 
	       return angleToRad(dst,0);
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleFromRad(const Vec & org, const Angle & ref) const {
	       return diffFrom(org).angleRad(ref) ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Angle angleFromRad(const Vec & dst) const { 
	       return angleFromRad(dst,0);
	    } ;

	    //--------------------------------------------------
	    // rotate (in degree)
	    /*--------------------*/
	    /**
	     *
	     */
	 public:	
	    Vec rotate(Angle a) const {
	       Vec r ;
	       r.x =   x * Cos(a) + y * Sin(x) ;
	       r.y = - x * Sin(a) + y * Cos(x) ;
	       return r ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:	
	    Vec & rotateSelf(Angle a) {
	       *this = rotate(a) ;
	       return *this ;
	    } ;

	    //--------------------------------------------------
	    // set 
	    /*--------------------*/
	    /**
	     *
	     */
	 public:	
	    Vec & set(const Val & vx, const Val & vy) {
	       x = vx ; y = vy ; return *this ;
	    } ;

	    /*--------------------*/
	    /**
	     *
	     */
	 public:	
	    Vec & set(const Vec & v) {
	       x = v.x ; y = v.y ; return *this ;
	    } ;

	    //--------------------------------------------------
	    // set by polar
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Vec & setByPolar(const Val & r, const Angle & a) {
	       x = r * Cos(a) ;
	       y = r * Sin(a) ;
	       return *this ;
	    } ;

	    //--------------------------------------------------
	    // check finite value or NaN
	    /*--------------------*/
	    /**
	     *
	     */
	 public:
	    Bool isnan() const { return ::isnan(x) || ::isnan(y) ; } ;
	    Bool finite() const { return ::finite(x) && ::finite(y) ; } ;
	    Bool isinf() const { return ::isinf(x) || ::isinf(y) ; } ;
      } ;

      //------------------------------------------------------------
      // output

      /*--------------------*/
      /**
       *
       */

      inline ostream & operator<< (ostream & ostr, const Vec & v) {
	 return ostr << "#Vec[" << v.x << "," << v.y << "]" ;
      } ;

      //------------------------------------------------------------
      // vector divide

      //------------------------------
      // =/

      /*--------------------*/
      /**
       *
       */

      inline Vec & Vec::operator /= (const Val & a) { 
	 if(Abs(a) > ITK_EPS) {
	    x /= a ; y /= a ;
	 } else {
	    ITK_WRN("a vector " << *this 
		    << " is divided by zero (" << a << ")") ;
	    Flt b = Sign(a) * ITK_EPS ;
	    x /= b ; y /= b ;
	 }
	 return *this ;
      } ;

      //------------------------------
      // vec/val

      /*--------------------*/
      /**
       *
       */

      inline Vec Vec::operator /(const Val &a) const { 
	 Flt vx, vy ;
	 if(Abs(a) > ITK_EPS) {
	    vx = x / a ; vy = y / a ;
	 } else {
	    ITK_WRN("a vector " << *this 
		    << " is divided by zero (" << a << ")") ;
	    Flt b = Sign(a) * ITK_EPS ;
	    vx = x / b ; vy = y / b ;
	 }
	 return Vec(vx,vy) ;
      };

      //------------------------------
      // vec/vec

      /*--------------------*/
      /**
       *
       */

      inline Vec Vec::operator /(const Vec &a) const { 
	 Flt vx;
	 if(Abs(a.x) > ITK_EPS) vx = x/a.x ;
	 else {
	    ITK_WRN("a vector " << *this 
		    << " is divided by zero (" << a << ")") ;
	    Flt b = Sign(a.x) * ITK_EPS ;
	    vx = x / b ; 
	 }
	 Flt vy;
	 if(Abs(a.y) > ITK_EPS) vy = y/a.y ;
	 else {
	    ITK_WRN("a vector " << *this 
		    << " is divided by zero (" << a << ")") ;
	    Flt b = Sign(a.y) * ITK_EPS ;
	    vy = y / b ; 
	 }
	 return Vec(vx,vy) ;
      } ;

      //------------------------------------------------------------
      // convert from polar

      /*--------------------*/
      /**
       *
       */

      inline Vec polar2Vec(Val radius, Angle ang) {
	 return Vec(radius * Cos(ang), radius * Sin(ang));
      } ;

      /*--------------------*/
      /**
       *
       */

      inline Vec polar2VecRad(Val radius, Angle ang) {
	 return Vec(radius * cos(ang), radius * sin(ang));
      } ;

      //------------------------------------------------------------
      // diff / distance / innerproduct

      /*--------------------*/
      /**
       *
       */

      inline Vec diff(const Vec& from, const Vec& to) {
	 return to.diffFrom(from) ;
      }

      /*--------------------*/
      /**
       *
       */

      inline Val distance(const Vec& from, const Vec& to) {
	 return diff(from,to).norm() ;
      } 

      /*--------------------*/
      /**
       *
       */

      inline Val innerproduct(const Vec& p, const Vec& q) {
	 return p.x * q.x + p.y * q.y ; 
      } 

   };
} ;
////////////////////////////////////////////////////////////////////////
#endif


