// -*- Mode:C++ -*-
//Header:
//File: SubString.cc
//Author: NODA, Itsuki
//Date: 2000/03/12
//

//ModifyHistory:
// 2000/03/12: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#include "itk/SString.h"

namespace Itk {

   //--------------------------------------------------
   // class statics

   SubString SubString::null = SubString() ;

   //--------------------------------------------------
   // operator==

   Bool SubString::eq(const SubString& r) const { 
      return (head == r.head) && (tail == r.tail) ;
   }

   Bool SubString::equal(const SubString& r) const { 
      UInt l = length() ;
      return (l == r.length()) && !strncmp(head,r.head,l) ;
   } 

   Bool SubString::operator== (const SubString& r) const { 
      return equal(r) ; 
   } 

   Bool SubString::operator== (const char* ref) const { 
      UInt l = length() ;
      return !strncmp(head,ref,l) ;
   } 

   //--------------------------------------------------
   // less  // used in map

   Bool SubString::less(const SubString & str0, 
			const SubString & str1) {
      Int l0 = str0.length() ;
      Int l1 = str1.length() ;
      Int l = Min(l0,l1) ;

      Int r = strncmp(str0.head, str1.head, l) ;

      if      (r < 0) return True ;
      else if (r > 0) return False ;
      else            return (l0 < l1) ;
   } ;

   Bool SubString::lessequal(const SubString & str0, 
			     const SubString & str1) {
      Int l0 = str0.length() ;
      Int l1 = str1.length() ;
      Int l = Min(l0,l1) ;

      Int r = strncmp(str0.head, str1.head, l) ;

      if      (r < 0) return True ;
      else if (r > 0) return False ;
      else            return (l0 <= l1) ;
   } ;

   Bool SubString::greater(const SubString & str0, 
			   const SubString & str1) {
      Int l0 = str0.length() ;
      Int l1 = str1.length() ;
      Int l = Min(l0,l1) ;

      Int r = strncmp(str0.head, str1.head, l) ;

      if      (r > 0) return True ;
      else if (r < 0) return False ;
      else            return (l0 > l1) ;
   } ;

   Bool SubString::greaterequal(const SubString & str0, 
			       const SubString & str1) {
      Int l0 = str0.length() ;
      Int l1 = str1.length() ;
      Int l = Min(l0,l1) ;

      Int r = strncmp(str0.head, str1.head, l) ;

      if      (r > 0) return True ;
      else if (r < 0) return False ;
      else            return (l0 >= l1) ;
   } ;

   Bool SubString::operator<(const SubString & ref) const {
      return less(*this,ref) ;
   } ;

   Bool SubString::operator<=(const SubString & ref) const {
      return lessequal(*this,ref) ;
   } ;

   Bool SubString::operator>(const SubString & ref) const {
      return greater(*this,ref) ;
   } ;

   Bool SubString::operator>=(const SubString & ref) const {
      return greaterequal(*this,ref) ;
   } ;

  Bool SubString::operator<(const char* ref) const {
    return less(*this, SubString(const_cast<char*>(ref)));
  };

  Bool SubString::operator<=(const char* ref) const {
    return lessequal(*this, SubString(const_cast<char*>(ref)));
  };

  Bool SubString::operator>(const char* ref) const {
    return greater(*this, SubString(const_cast<char*>(ref)));
  };
  
  Bool SubString::operator>=(const char* ref) const {
    return greaterequal(*this, SubString(const_cast<char*>(ref)));
  };

   //--------------------------------------------------
   // prefix check

   Bool SubString::isPrefixOf(const SubString& ref) const {
      UInt l = length() ;
      return l <= ref.length() && !strncmp(head,ref.head,length()) ;
   } 

   Bool SubString::isPrefixOf(const char* const ref) const {
      return isPrefixOf(SubString(const_cast<char*>(ref))) ;
   }

   Bool SubString::hasPrefix(const SubString& prefix) const {
      return prefix.isPrefixOf(*this) ;
   } 

   Bool SubString::hasPrefix(const char* const ref) const {
      return hasPrefix(SubString(const_cast<char*>(ref))) ;
   } 

   Bool SubString::checkPopPrefix(const SubString & prefix) {
      if(hasPrefix(prefix)) {
	 popSubString(prefix.length()) ;
	 return True ;
      } else {
	 return False ;
      }
   } 

   Bool SubString::checkPopPrefix(const char * const prefix) {
      return checkPopPrefix(SubString(const_cast<char *>(prefix))) ;
   }

   //--------------------------------------------------
   // index (return -1 if not exist)

   Int SubString::index(const char c) const {
      Int i = 0 ;
      for(const char * cp = head ; cp < tail ; cp++) {
	 if(*cp == c) return i ;
	 i++ ;
      }
      return -1 ;
   } 

   Bool SubString::has(const char c) const {
      if(index(c) < 0) return False ;
      else return True ;
   } 
	    
   //--------------------------------------------------
   // alloc

   Bool SubString::alloc(const UInt l, Bool forcep) {
      if(!forcep && length() > l) return True ;
      head = new char[l] ;
      if(isNull(head)) {
	 tail = ITK_NULLPTR ;
	 return False ;
      } else {
	 set(head,l) ;
	 return True ;
      }
   } 

   //--------------------------------------------------
   // length

   void SubString::setLength(const UInt l) { 
      tail = &(head[l]) ; 
   }
      
   UInt SubString::strlength() const { 
      UInt l = 0 ;
      for(char* s = head ; s < tail ; s++) {
	 if(isNull(*s)) return l ;
	 l++ ;
      } 
      return l ;
   }

   void SubString::adjustLength() {
      setLength(strlen(head)) ;
   } 
   
   //--------------------------------------------------
   // copy

   Bool SubString::copy(const SubString& src, Bool allocp) {
      if(isNull(head) || allocp) alloc(src.length()+1) ;
      tail = head ;
      for(char* s = (char*)src ; s < src.tail ; s++) {
	 *tail = *s ; tail++ ;
      }
      terminate() ;
      return True ;
   } 

   Bool SubString::copy(char* h, char* t, Bool allocp = False) {
      return copy(SubString(h,t),allocp) ;
   } 

   Bool SubString::copy(const char * h, const char * t) {
      if(isNull(t)) return copy(SubString(const_cast<char *>(h)),False) ;
      else return copy(const_cast<char *>(h), const_cast<char *>(t),False) ;
   }

   //--------------------------------------------------
   // dup

   SubString SubString::dup() const {
      SubString r ;
      r.copy(*this,True) ;
      return r ;
   } 

   //--------------------------------------------------
   // strdup duplicate and 

   char* SubString::strdup() const {
      return (char*)dup() ;
   } 

   //--------------------------------------------------
   // copy to char*

   Bool SubString::copyTo(char* dst,Int maxlen) const {
      Int l = 0 ;
      for(const char* s = head ; s < tail ; s++) {
	 if(maxlen >= 0 && l >= maxlen) {
	    dst-- ;
	    break ;
	 }
	 *dst = *s ; dst++ ; l++ ;
      }
      *dst = ITK_NULLCHR ;

      if(l == maxlen) return False ;  /* dst is ovewflowed */
      else return True ;              /* copy succeeds */
      
   } 
   
   //--------------------------------------------------
   // basic operation (get)

   char SubString::popChar() { 
      char c = first() ; head++ ; return c ;
   } 

   Bool SubString::skipNChar(UInt n = 1) {
      if(n > length()) return False ;
      head = &(head[n]) ;
      return True ;
   } 

   UInt SubString::skipWhile(const SubString & charlist) {
      UInt c = 0 ;
      for( ; !tailp() ; ignoreChar() ) {
	 if(! charlist.has(first())) break ;
	 c++ ;
      }
      return c ;
   } 

   UInt SubString::skipUntil(const SubString & charlist) {
      UInt c = 0 ;
      for( ; !tailp() ; ignoreChar() ) {
	 if( charlist.has(first())) break ;
	 c++ ;
      }
      return c ;
   }

   SubString SubString::sub(UInt offset, UInt length) const {
      return SubString(&(head[offset]),&(head[offset+length])) ;
   } 

   SubString SubString::popSubString(UInt offset, UInt length) {
      SubString s = sub(offset,length) ;
      head = s.tail ;
      return s ;
   } 

   //--------------------------------------------------
   // basic operation (put)

   void SubString::pushChar(const char c) { 
      head-- ;
      *head = c ;
   } 

   void SubString::putChar(const char c) {
      *tail = c ;
      tail++ ;
   } 

   void SubString::put(const SubString& str) {
      for(char* s = str.head ; s < str.tail ; s++) {
	 putChar(*s) ;
      } 
   } 

   //--------------------------------------------------
   // printf

   Int SubString::printf(const char * format, ...) {
      va_list ap ;
      va_start(ap,format) ;
      Int n = ::vsprintf(tail,format,ap) ;
      va_end(ap) ;
      tail = &(tail[n]) ;
      return n ;
   }

   //--------------------------------------------------
   // printf1 

   Int SubString::printf1(const char* format) {
      int n = sprintf(tail,format) ;
      tail = &(tail[n]) ;
      return n ;
   }

   Int SubString::printf1(const char* format, const char * x) {
      int n = sprintf(tail,format,x) ;
      tail = &(tail[n]) ;
      return n ;
   }

   Int SubString::printf1(const char* format, Int x) {
      int n = sprintf(tail,format,x) ;
      tail = &(tail[n]) ;
      return n ;
   }

   Int SubString::printf1(const char* format, UInt x) {
      int n = sprintf(tail,format,x) ;
      tail = &(tail[n]) ;
      return n ;
   }

   Int SubString::printf1(const char* format, Flt x) {
      int n = sprintf(tail,format,x) ;
      tail = &(tail[n]) ;
      return n ;
   }
	 
   //--------------------------------------------------
   // concatination

   SubString SubString::append(const SubString & rest) const {
      UInt l1 = length() ;
      UInt l2 = rest.length() ;
      SubString r ;
      r.head = new char[l1+l2+1] ;
      r.tail = &(r.head[l1+l2]) ;
      char* dst = r.head ;
      for(char * src = head ; src < tail ; src++) {
	 *dst = *src ; dst++ ;
      } ;
      for(char * src = rest.head ; src < rest.tail ; src++) {
	 *dst = *src ; dst++ ;
      } ;
      *(r.tail) = ITK_NULLCHR ;

      return r ;
   } 

   //--------------------------------------------------
   // scan by char set

   SubString SubString::scanInCharSet(const SubString & charSet) {
      SubString r ;
      r.head = head ;
      for(r.head = head ; head < tail ; head++) {
	 if(!charSet.has(*head)) break ;
      }
      r.tail = head ;
      return r ;
   } 

   Bool SubString::skipInCharSet(const SubString & charSet) {
      SubString r = scanInCharSet(charSet) ;
      if(r.isNullStr()) return False ;
      else return True ;
   } 

   //--------------------------------------------------
   // scan by delimiter

   SubString SubString::scanByDelimiter(const SubString & delimiter) {
      SubString r ;
      r.head = head ;
      for(r.head = head ; head < tail ; head++) {
	 if(delimiter.has(*head)) break ;
      }
      r.tail = head ;
      return r ;
   } 

   //--------------------------------------------------
   // scan as number value

   Flt SubString::scanAsFlt() {
      char *endp ;
      Flt r = strtod(head, &endp) ;
      head = endp ;
      return r ;
   } 

   Int SubString::scanAsInt() {
      char *endp ;
      Int r = strtol(head, &endp, 10) ;
      head = endp ;
      return r ;
   } 

   UInt SubString::scanAsUInt() {
      char *endp ;
      UInt r = strtoul(head, &endp, 10) ;
      head = endp ;
      return r ;
   } 

   Bool SubString::popIntVal(Int& i) {
      int l ;
      if(1 > sscanf(head, "%ld%n", &i, &l)) return False ;
      return skipNChar(l) ;
   } ;

   Bool SubString::popUIntVal(UInt& i) {
      int l ;
      if(1 > sscanf(head, "%ld%n", &i, &l)) return False ;
      return skipNChar(l) ;
   } ;

   Bool SubString::popFloatVal(Flt& v) {
      int l ;
      if(1 > sscanf(head, "%lf%n", &v, &l)) return False ;
      return skipNChar(l) ;
   } ;

}
