// -*- Mode: c++ -*-
//Header:
//File: SimpleSexp.h
//Author: NODA, Itsuki
//Date: 2000/08/08
//

//ModifyHistory:
// 2000/08/08: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_SimpleSexp_h_
#define _itk_SimpleSexp_h_
//////////////////////////////////////////////////////////////////////
#include "itk/btype.h"
#include "itk/utility.h"
#include "itk/WithDescriber.h"
#include "itk/SString.h"
#include "itk/Buffer.h"
#include "itk/HeapT.h"
#include "itk/Mutex.h"
#include <fstream.h>
#include <ctype.h>

namespace Itk {

   //======================================================================
   // special chars

#define ITK_SEXP_CONS_BEGIN_CHAR	'('
#define ITK_SEXP_CONS_END_CHAR		')'

#define ITK_SEXP_CONS_SEP_CHAR		' '

#define ITK_SEXP_STRING_BEGIN_CHAR	'"'
#define ITK_SEXP_STRING_END_CHAR	'"'
#define ITK_SEXP_STRING_ESCAPE_CHAR	'\\'

#define ITK_SEXP_COMMENT_BEGIN_CHAR	';'
#define ITK_SEXP_COMMENT_END_CHAR	'\n'




   //======================================================================
   // class SimpleSexp

   class SimpleSexp : public WithDescriber {

	 //============================================================
	 // SimpleSexp:: enum Tag

      public:
	 enum Tag {
	    T_Nil,
	    T_Atom,
	    T_Cons,
	    T_Error
	 } ;

	 enum AtomType {
	    AT_None,
	    AT_Symbol,
	    AT_Int,
	    AT_Flt,
	    AT_Error
	 } ;
	 
	 //============================================================
	 // SimpleSexp:: class Generic Input

      public:
	 class GenericInput {
	    public:
	       virtual char get() = 0 ;		// get the next char
	       virtual void unget() = 0 ;	// put back a char
	       virtual void unget(const char & c) = 0 ;	// put back a char
	       virtual Bool isEnd() = 0 ;	// check end
	       virtual void chop() = 0 ;	// cut last char from buffer
	       virtual void ignore() = 0 ;	// ignore one char

	       virtual char peek() { 		// peek the next char
		  char c = get() ; 
		  unget(c) ;
		  return c ; 
	       } ;

	       virtual char * cpos() = 0 ;	// return current pos in buffer
	 } ;

	 //============================================================
	 // SimpleSexp:: class StreamInput

#ifndef ITK_SEXP_BUFSIZE
#define ITK_SEXP_BUFSIZE	8192
#endif

      public:
	 class StreamInput : public GenericInput {
	    public:
	       istream * _istr ;
	       Buffer * buf ;

	       StreamInput() { init(ITK_NULLPTR) ; } ;
	       StreamInput(istream * s) { init(s) ; } ;
	       StreamInput(istream & s) { init(&s) ; } ;
	       StreamInput(istream * s, Buffer * b) { init(s,b) ; } ;
	       StreamInput(istream & s, Buffer & b) { init(&s,&b) ; } ;

	       virtual char * cpos() { return buf->tail ; } ;

	       void init(istream * s, Buffer * b = ITK_NULLPTR) {
		  _istr = s ;
		  buf = b ;
		  if(isNull(buf)) buf = new Buffer(ITK_SEXP_BUFSIZE) ;
	       } ;

	       const istream & str() const { return *_istr ; } ;
	       istream & str() { return *_istr ; } ;
	       
	       virtual char get() { 
		  char c = (char)str().get() ; 
		  buf->putChar(c) ;
		  return c ;
	       } ;
	       virtual void unget() {
		  str().unget() ;
		  buf->chop() ;		/* unsafe */
	       } ;
	       virtual void unget(const char & c) { 
		  str().putback(c) ; 
		  buf->chop() ;		/* unsafe */
	       } ;
	       virtual char peek() { 
		  return str().peek() ; 
	       } ;
	       virtual Bool isEnd() { 
		  return str().eof() ; 
	       } ;
	       virtual void chop() {
		  buf->chop() ; 	/* unsafe */
	       } ;
	       virtual void ignore() { str().get() ; } ;
	 } ;

	 //============================================================
	 // SimpleSexp:: class BufferInput

      public:
	 class BufferInput : public GenericInput {
	    public:
	       Buffer * _buf ;
	       
	       BufferInput() { _buf = ITK_NULLPTR ; } ;
	       BufferInput(Buffer * buf) { _buf = buf ; } ;
	       BufferInput(Buffer & buf) { _buf = &buf ; } ;

	       const Buffer & buf() const { return *_buf ; } ;
	       Buffer & buf() { return *_buf ; } ;

	       virtual char * cpos() { return buf().head ; } ;

	       virtual char get() { return buf().getChar() ; } ;
	       virtual void unget() { buf().ungetChar() ; } ;
	       virtual void unget(const char & c) { buf().pushChar(c) ; } ;
	       virtual char peek() { return buf().first() ; } ;
	       virtual Bool isEnd() { return buf().tailp() ; } ;
	       virtual void chop() {} ;
	       virtual void ignore() { buf().ignoreChar() ; } ;
	 } ;

	 //============================================================
	 // SimpleSexp:: class Heap & heap-operation(static members)
      public:
	 class Heap : public HeapT<SimpleSexp>, public Mutex { 

	       //----------------------------------------
	       // SimpleSexp::Heap:: consts
	    public:
	       static const UInt BufSize = 8192 ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: member
	    public:
	       Buffer strbuf ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: construct
	    public:
	       Heap() { init() ; } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: init
	    public:
	       void init() { strbuf.alloc(BufSize) ; } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: reset
	    public:
	       void clear() { 
		  strbuf.clear() ; 
		  reset() ;  // HeapT<SimpleSexp>:reset() ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: newCell
	    public:
	       SimpleSexp * newCell() { return getPtr() ; } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: atom
	    public:
	       SimpleSexp * newAtom(AtomType at = AT_None) {
		  SimpleSexp * cell = newCell() ;
		  cell->tag = T_Atom ;
		  cell->atype = at ;
		  cell->str.setNull() ;
		  return cell ;
	       } ;
		  
	    public:
	       SimpleSexp * newAtom(const SubString & s, 
				    AtomType at = AT_None) {
		  SimpleSexp * cell = newAtom(at) ;
		  cell->str = s ;
		  return cell ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: new symbol
	    public:
	       SimpleSexp * newSymbol(const SubString & s) {
		  return newAtom(s,AT_Symbol) ;
	       } ;
	    public:
	       SimpleSexp * newSymbol(const char * s, Bool copyp) {
		  SimpleSexp * cell = newAtom(AT_Symbol) ;
		  cell->str.set(const_cast<char*>(s)) ;
		  return cell ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: new int
	    public:
	       SimpleSexp * newInt(const Int & i) {
		  SimpleSexp * cell = newAtom(AT_Int) ;
		  cell->ival = i ;
		  return cell ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: new flt
	    public:
	       SimpleSexp * newFlt(const Flt & f) {
		  SimpleSexp * cell = newAtom(AT_Flt) ;
		  cell->fval = f ;
		  return cell ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: cons
	    public:
	       SimpleSexp * cons(SimpleSexp * car, SimpleSexp * cdr) {
		  SimpleSexp * cell = newCell() ;
		  cell->tag = T_Cons ;
		  cell->str.setNull() ;
		  cell->car = car ;
		  cell->cdr = cdr ;
		  return cell ;
	       } ;
		  
	       //----------------------------------------
	       // SimpleSexp::Heap:: list
	    public:
	       SimpleSexp * list() { return Nil ; } ;
	       SimpleSexp * list(SimpleSexp * s0) { 
		  return cons(s0,list()) ;
	       } ;
	       SimpleSexp * list(SimpleSexp * s0, SimpleSexp * s1) {
		  return cons(s0,list(s1)) ; 
	       } ;
	       SimpleSexp * list(SimpleSexp * s0, 
				 SimpleSexp * s1, 
				 SimpleSexp * s2) {
		  return cons(s0,list(s1,s2)) ; 
	       } ;
	       SimpleSexp * list(SimpleSexp * s0, 
				 SimpleSexp * s1, 
				 SimpleSexp * s2,
				 SimpleSexp * s3) {
		  return cons(s0,list(s1,s2,s3)) ; 
	       } ;
	       SimpleSexp * list(SimpleSexp * s0, 
				 SimpleSexp * s1, 
				 SimpleSexp * s2,
				 SimpleSexp * s3,
				 SimpleSexp * s4) {
		  return cons(s0,list(s1,s2,s3,s4)) ; 
	       } ;
	       SimpleSexp * list(SimpleSexp * s0, 
				 SimpleSexp * s1, 
				 SimpleSexp * s2,
				 SimpleSexp * s3,
				 SimpleSexp * s4,
				 SimpleSexp * s5) {
		  return cons(s0,list(s1,s2,s3,s4,s5)) ; 
	       } ;
	       SimpleSexp * list(SimpleSexp * s0, 
				 SimpleSexp * s1, 
				 SimpleSexp * s2,
				 SimpleSexp * s3,
				 SimpleSexp * s4,
				 SimpleSexp * s5,
				 SimpleSexp * s6) {
		  return cons(s0,list(s1,s2,s3,s4,s5,s6)) ; 
	       } ;
	       SimpleSexp * list(SimpleSexp * s0, 
				 SimpleSexp * s1, 
				 SimpleSexp * s2,
				 SimpleSexp * s3,
				 SimpleSexp * s4,
				 SimpleSexp * s5,
				 SimpleSexp * s6,
				 SimpleSexp * s7) {
		  return cons(s0,list(s1,s2,s3,s4,s5,s6,s7)) ; 
	       } ;


	       //----------------------------------------
	       // SimpleSexp::Heap:: append
	    public:
	       SimpleSexp * append(SimpleSexp * list1, SimpleSexp * list2) ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: addFirst (push & replace)
	    public:
	       SimpleSexp * addFirst(SimpleSexp * top, SimpleSexp * & list) {
		  list = cons(top,list) ;
		  return list ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: addLast
	    public:
	       SimpleSexp * addLast(SimpleSexp * lst, SimpleSexp * lastelm) ;

	       //----------------------------------------
	       // SimpleSexp::Heap:: scan
	    public:
	       SimpleSexp * scan(GenericInput & ginp) {
		  return SimpleSexp::scan(ginp,this) ;
	       } ;
	       SimpleSexp * scan(istream * istr) {
		  return SimpleSexp::scan(istr,this) ;
	       } ;
	       SimpleSexp * scan(istream & istr) {
		  return SimpleSexp::scan(istr,this) ;
	       } ;
	       SimpleSexp * scan(istream * istr, Buffer * b) {
		  return SimpleSexp::scan(istr,b,this) ;
	       } ;
	       SimpleSexp * scan(istream & istr, Buffer & b) {
		  return SimpleSexp::scan(istr,b,this) ;
	       } ;
	       SimpleSexp * scan(Buffer * b) {
		  return SimpleSexp::scan(b,this) ;
	       } ;
	       SimpleSexp * scan(Buffer & b) {
		  return SimpleSexp::scan(b,this) ;
	       } ;
	 } ;
	 
	 //------------------------------------------------------------
	 // SimpleSexp:: static operations for heap
      public:
	 static Heap heap ;

	 static SimpleSexp * get() { return heap.getPtr() ; } ; // not used ?
	 static void cleanHeap() { heap.reset() ; } ; // not used ?

	 static void lock() { heap.lock() ; } ;
	 static void unlock() { heap.unlock() ; } ;


	 //============================================================
	 // SimpleSexp:: body

	 //--------------------------------------------------
	 // SimpleSexp:: static const
      public:
	 static SimpleSexp _Nil ;
	 static SimpleSexp * Nil ;

	 //--------------------------------------------------
	 // SimpleSexp:: member

      public:
	 Tag tag ;
	 AtomType atype ;
	 SubString str ;
	 Int ival ;
	 Flt fval ;

	 SimpleSexp * car ;	// used only if tag == T_Cons 
	 SimpleSexp * cdr ;	// used only if tag == T_Cons 

	 //--------------------------------------------------
	 // SimpleSexp:: constructor
      public:
	 SimpleSexp() {} ;
	 SimpleSexp(Tag t, const SubString & s) {
	    tag = t ;
	    str = s ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp:: scan (static operations)

      public:

	 //------------------------------
	 // SimpleSexp:: scan top level

	 static SimpleSexp * scan(GenericInput & ginp,
				  Heap * hp = ITK_NULLPTR) ;

	 static SimpleSexp * scan(istream * istr, Heap * hp = ITK_NULLPTR) { 
	    return scan(*istr,hp) ; }
	 static SimpleSexp * scan(istream & istr, Heap * hp = ITK_NULLPTR) { 
	    StreamInput si(istr) ;
	    return scan(si,hp) ;
	 } ;
	 static SimpleSexp * scan(istream * istr, Buffer * b, 
				  Heap * hp = ITK_NULLPTR) { 
	    return scan(*istr,*b, hp) ; }
	 static SimpleSexp * scan(istream & istr, Buffer & b, 
				  Heap * hp = ITK_NULLPTR) { 
	    StreamInput si(istr,b) ;
	    b.clear() ;
	    return scan(si, hp) ;
	 } ;

	 static SimpleSexp * scan(Buffer * b, Heap * hp = ITK_NULLPTR) { 
	    return scan(*b, hp) ; } ;
	 static SimpleSexp * scan(Buffer & b, Heap * hp = ITK_NULLPTR) {
	    BufferInput bi(b) ;
	    return scan(bi, hp) ;
	 } ;

	 //------------------------------
	 // SimpleSexp:: scan body

	 static SimpleSexp * scanSexp(GenericInput & gi, Heap * hp) ;
	 static SimpleSexp * scanAtom(GenericInput & gi, Heap * hp) ;
	 static SimpleSexp * scanCons(GenericInput & gi, Heap * hp) ;
	 static SimpleSexp * scanConsBody(char * head, GenericInput & gi,
					  Heap * hp) ;

	 //------------------------------
	 // SimpleSexp:: skips

	 static Bool skipSpaceComment(GenericInput & gi) ;
	 static Bool skipSpace(GenericInput & gi) ;
	 static Bool skipComment(GenericInput & gi) ;

	 //--------------------------------------------------
	 // SimpleSexp:: type check

	 Bool isNullPtr() const { return isNull(this) ; } ;

	 Bool isNil() const { return !isNullPtr() && tag == T_Nil ; } ;
	 Bool isAtom() const { return !isNullPtr() && tag == T_Atom ; } ;
	 Bool isCons() const { return !isNullPtr() && tag == T_Cons ; } ;
	 Bool isError() const { return !isNullPtr() && tag == T_Error ; } ;
	 
	 Bool isString() const { 
	    return isAtom() 
	       && (str.first() == ITK_SEXP_STRING_BEGIN_CHAR) ;
	 } ;

	 Bool isInt() const {
	    Int dummy ;
	    return scanAsIntTo(dummy) ;
	 } ;

	 Bool isFlt() const {
	    Flt dummy ;
	    return scanAsFltTo(dummy) ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp:: scan As Number

	 Int scanAsInt() const ;

	 Bool scanAsIntTo(Int & val) const ;

	 Flt scanAsFlt() const ;

	 Bool scanAsFltTo(Flt & val) const ;
	 
	 //--------------------------------------------------
	 // SimpleSexp:: scan As Symbol

	 SubString scanAsSymbol() const ;

	 const SubString & scanAsConstSymbol() const {
	    return str ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp::scan As SubString (naked if it start quote mark)
	 
	 SubString scanAsString(Bool naked = True, Bool copyp = False) const ;

	 //--------------------------------------------------
	 // SimpleSexp::copy to buffer
	 
	 void copyTo(Buffer & buffer) const ;

	 //--------------------------------------------------
	 // SimpleSexp::list operations

	 UInt length() const ;

	 SimpleSexp * nthcdr(UInt n) ;

	 SimpleSexp * nth(UInt n) ;

	 SimpleSexp * first() { return nth(0) ; } ;
	 SimpleSexp * second() { return nth(1) ; } ;
	 SimpleSexp * rest() { return nthcdr(1) ; } ;
	 SimpleSexp * last() { return nthcdr(length()-1) ; } ;
	 SimpleSexp * lastone() { return nth(length()-1) ; } ;

	 //--------------------------------------------------
	 // SimpleSexp::equal

	 Bool equal(const SimpleSexp * sexp) const { return equal(*sexp) ; } ;

	 Bool equal(const SimpleSexp & sexp) const {
	    if(isNullPtr() || isError()) return False ;
	    switch(tag) {
	       case T_Nil : return sexp.isNil() ; break ;
	       case T_Atom : return sexp.equal(str) ; break ;
	       case T_Cons : {
		  if(sexp.isCons()) {
		     return car->equal(sexp.car) 
			&&  cdr->equal(sexp.cdr) ;
		  } else {
		     return False ;
		  }
	       }
	       default : return False ;
	    }
	 } ;

	 Bool equal(const char * sym) const {
	    return !isNullPtr() && isAtom() && str == sym ;
	 } ;
	 Bool equal(const SubString & sym) const {
	    return !isNullPtr() && isAtom() && str == sym ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp::assoc table operation
      public:
	 SimpleSexp * assoc(const SimpleSexp * const key) const ;
	 SimpleSexp * assoc(const SubString & key) const ;
	 SimpleSexp * assocValue(const SimpleSexp * const key) const ;
	 SimpleSexp * assocValue(const SubString & key) const ;

	 //--------------------------------------------------
	 // SimpleSexp::proparty table operation
	 //	passoc() return value
      public:
	 SimpleSexp * passoc(const SimpleSexp * const key) const ;
	 SimpleSexp * passoc(const SubString & key) const ;

	 //--------------------------------------------------
	 // SimpleSexp::describe

      public:
	 virtual void describe(ostream& o, Bool detailp = True) const ;
	 virtual void describePtr(ostream& o, Bool detailp = True) const {
	    describe(o,detailp) ;
	 } ;
   } ;

} ;


//////////////////////////////////////////////////////////////////////
#endif
