// -*- Mode: c++ -*-
//Header:
//File: SimpleSexp2.cc
//Author: NODA, Itsuki
//Date: 2001/06/26
//

//ModifyHistory:
// 2001/06/26: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#include "SimpleSexp2.h"

namespace Itk {

   //======================================================================
   // class static variables
   
   SimpleSexp2::Heap SimpleSexp2::sharedHeap ;
   SimpleSexp2::Scanner SimpleSexp2::sharedScanner(cin,False) ;

   SimpleSexp2 SimpleSexp2::_Nil(T_Nil,SubString("()")) ;
   SimpleSexp2 * SimpleSexp2::Nil = &SimpleSexp2::_Nil;

   SimpleSexp2 SimpleSexp2::_Eof(T_Atom,SubString("eof")) ;
   SimpleSexp2 * SimpleSexp2::Eof = &_Eof ;

   SimpleSexp2 SimpleSexp2::_Error(T_Error,SubString("error")) ;
   SimpleSexp2 * SimpleSexp2::Error = &_Error ;

   SimpleSexp2 SimpleSexp2::_TrueValue(T_Atom,SubString("T")) ;
   SimpleSexp2 * SimpleSexp2::TrueValue = &_TrueValue ;

   //======================================================================
   // scan As Number

   Int SimpleSexp2::scanAsInt() const {
      Int v ; 
      scanAsIntTo(v) ;
      return v ;
   } ;

   Int SimpleSexp2::scanAsInt() {
      Int v ; 
      scanAsIntTo(v) ;
      return v ;
   } ;

   Bool SimpleSexp2::scanAsIntTo(Int & val) const {
      if(isAtom()) {
	 char buf[ITK_SEXP2_NUMBER_STR_MAX] ;
	 str.copyTo(buf,ITK_SEXP2_NUMBER_STR_MAX) ;
	 char *endptr ;
	 val = strtol(buf,&endptr,0) ;
	 if(isNull(*endptr)) return True ;
      }
      return False ;
   } ;

   Bool SimpleSexp2::scanAsIntTo(Int & val) {
      Bool r = const_cast<const SimpleSexp2 *>(this)->scanAsIntTo(val) ;
      if(r) {
	 tag=T_Int ;
	 ival = val ;
      }
      return r ;
   } ;

   Flt SimpleSexp2::scanAsFlt() const {
      Flt v ; 
      scanAsFltTo(v) ;
      return v ;
   } ;

   Flt SimpleSexp2::scanAsFlt() {
      Flt v ; 
      scanAsFltTo(v) ;
      return v ;
   } ;

   Bool SimpleSexp2::scanAsFltTo(Flt & val) const {
      if(isAtom()) {
	 char buf[ITK_SEXP2_NUMBER_STR_MAX] ;
	 str.copyTo(buf,ITK_SEXP2_NUMBER_STR_MAX) ;
	 char *endptr ;
	 val = strtod(buf,&endptr) ;
	 if(isNull(*endptr)) return True ;
      }
      return False ;
   } ;

   Bool SimpleSexp2::scanAsFltTo(Flt & val) {
      Bool r = const_cast<const SimpleSexp2 *>(this)->scanAsFltTo(val) ;
      if(r) {
	 tag=T_Flt ;
	 fval = val ;
      }
      return r ;
   } ;

   SubString SimpleSexp2::scanAsSymbol() const { return str.dup() ; } ;

   SubString SimpleSexp2::scanAsString(Bool naked, Bool copyp) const {
      SubString r ;
      if(copyp) r.copy(str,True) ;
      else      r = str ;
      if(isString()) {
	 r.ignoreChar() ; // chop the first char
	 r.chop() ;       // chop the last char
      }
      return r ;
   } ;

   //--------------------------------------------------
   // copy to buffer

   void SimpleSexp2::outputTo(Buffer & buffer) const {
      buffer.put(str) ;
   }

   //--------------------------------------------------
   // SimpleSexp2:: set symbol

   SimpleSexp2 * SimpleSexp2::fixTag(Bool recursivep) {
      if(tag == T_Atom) {
	 if(str.isNullStr()) {
	    tag = T_Symbol ;
	 } else {
	    Int iv ;
	    if(!scanAsIntTo(iv)) {
	       Flt fv ;
	       if(!scanAsFltTo(fv)) {
		  tag = T_Symbol ;
	       }
	    }
	 }
      } else if (tag == T_Cons && recursivep) {
	 car()->fixTag(recursivep) ;
	 cdr()->fixTag(recursivep) ;
      }
      return this ;
   } ;

   //======================================================================
   // list operations

   UInt SimpleSexp2::length() const {
      UInt l = 0 ;
      for(const SimpleSexp2 * p = this ; p->isCons() ; p = p->cdr()) {
	 l++ ;
      }
      return l ;
   } ;

   SimpleSexp2 * SimpleSexp2::nthcdr(UInt n) {
      SimpleSexp2 * r = this ;
      for(UInt i = 0 ; i < n ; i++) { 
	 if(isNullPtr() || !isCons()) {
	    r = ITK_NULLPTR ;
	    break ;
	 }
	 r = r->cdr() ; 
      } 
      return r ;
   } ;

   SimpleSexp2 * SimpleSexp2::nth(UInt n) {
      SimpleSexp2 * r = nthcdr(n) ;
      if(r->isCons()) return r->car() ;
      else return ITK_NULLPTR;
   } ;

   //======================================================================
   // table operations

   //------------------------------------------------------------
   // assoc

   SimpleSexp2 * SimpleSexp2::assoc(const SimpleSexp2 * const key) const {
      SimpleSexp2 * l = const_cast<SimpleSexp2*>(this) ;
      SimpleSexp2 * e = ITK_NULLPTR;
      for( ; l->isCons() ; l = l->rest()) {
	 e = l->first() ;
	 if(e->first()->equal(key)) break ;
      }
      if (l->isCons()) return e ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // assoc by string

   SimpleSexp2 * SimpleSexp2::assoc(const SubString & key) const {
      SimpleSexp2 * l = const_cast<SimpleSexp2*>(this) ;
      SimpleSexp2 * e = ITK_NULLPTR;
      for( ; l->isCons() ; l = l->rest()) {
	 e = l->first() ;
	 if(e->first()->equal(key)) break ;
      }
      if (l->isCons()) return e ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // assoc Value

   SimpleSexp2 * SimpleSexp2::assocValue(const SimpleSexp2 * const key) const {
      SimpleSexp2 * r = assoc(key) ;
      if(r->isCons()) return r->second() ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // assoc Value by String

   SimpleSexp2 * SimpleSexp2::assocValue(const SubString& key) const {
      SimpleSexp2 * r = assoc(key) ;
      if(r->isCons()) return r->second() ;
      else return Nil ;
   } ;

   //------------------------------------------------------------
   // passoc (assoc property list)

   SimpleSexp2 * SimpleSexp2::passoc(const SimpleSexp2 * const key) const {
      SimpleSexp2 * r = passocPos(key) ;
      if(r->isCons()) return r->second() ;
      else return Nil ;
   } ;

   SimpleSexp2 * SimpleSexp2::passocPos(const SimpleSexp2 * const key) const {
      SimpleSexp2 * l = const_cast<SimpleSexp2*>(this) ;
      for( ; l->isCons() ; l = l->rest()->rest()) {
	 if(l->first()->equal(key)) break ;
      }
      return l ;
   } ;

   //------------------------------------------------------------
   // passoc (assoc property list) by string

   SimpleSexp2 * SimpleSexp2::passoc(const SubString & key) const {
      SimpleSexp2 * r = passocPos(key) ;
      if (r->isCons()) return r->second() ;
      else return Nil ;
   } ;
      
   SimpleSexp2 * SimpleSexp2::passocPos(const SubString & key) const {
      SimpleSexp2 * l = const_cast<SimpleSexp2*>(this) ;
      for( ; l->isCons() ; l = l->rest()->rest()) {
	 if(l->first()->equal(key)) break ;
      }
      return l ;
   } ;

   //======================================================================
   // duplicate

   SimpleSexp2 * SimpleSexp2::dup(Heap & heap, Bool strcopyp) const {
      if      (isNil()) {
	 return Nil ;
      }
      else if (isAtom()) {
	 SimpleSexp2 * cell = heap.newCell() ;
	 if (isNumber()) {
	    cell->set(this) ;
	    cell->fixTag() ;
	    cell->str.setNull() ;
	 } else {
	    cell->set(this,strcopyp,&heap) ;
	 }
	 return cell ;
      } 
      else if (isCons()) {
	 SimpleSexp2 * carval = car()->dup(heap);
	 SimpleSexp2 * cdrval = cdr()->dup(heap);
	 return heap.cons(carval,cdrval) ;
      }
      else {
	 ITK_ERR("illegal SimpleSexp2 object to duplicate:" << *this) ;
	 return Nil ;
      }
   } ;

   //======================================================================
   // describe

   void SimpleSexp2::describe(ostream& o, Bool detailp) const {
      if(detailp) {
	 o << "#SimpleSexp2[<" << (void*)this << ">:tag=" ;
	 switch(tag) {
	    case T_Atom   : o << "atom" ; break ;
	    case T_Int    : o << "int"  ; break ;
	    case T_Flt    : o << "flt"  ; break ;
	    case T_Symbol : o << "sym"  ; break ;
	    case T_Nil    : o << "nil"  ; break ;
	    case T_Cons   : o << "cons" ; break ;
	    case T_Error  : o << "error" ; break ;
	    default       : o << "other" ; break ;
	 }
	 switch(tag) {
	    case T_Int : {
	       o << ",ival=" << ival ;
	       break ;
	    } 
	    case T_Flt : {
	       o << ",fval=" << fval ;
	       break ;
	    }
	    default : break ;
	 }
	 o << ",str=\"" << str << "\"]" << endl ;
	 if(tag == T_Cons) {
	    o << "	car=" ;
	    car()->describe(o,detailp) ;
	    o << "	cdr=" ;
	    cdr()->describe(o,detailp) ;
	 }
      } else {
	 switch(tag) {
	    case T_Int : {
	       o << ival ;
	       break ;
	    } ;
	    case T_Flt : {
	       o << fval ;
	       break ;
	    } ;
	    case T_Symbol :
	    case T_Atom : {
	       o << str ;
	       break ;
	    } ;
	    case T_Nil :
	    case T_Cons : {
	       o << ITK_SEXP2_CONS_BEGIN_CHAR ;
	       for(const SimpleSexp2 * sx = this ; sx->tag == T_Cons ; 
		   sx = sx->cdr()) {
		  if(sx != this) o << ITK_SEXP2_CONS_SEP_CHAR ;
		  sx->car()->describe(o,detailp) ;
	       }
	       o << ITK_SEXP2_CONS_END_CHAR ;
	       break ;
	    } ;
	    case T_Error : 
	    default : {
	       o << "#SimpleSexp2Error[" << str << "]" ;
	    }
	 }
      }
   } ;
      
   //======================================================================
   // SimpleSexp2::Heap

   //------------------------------------------------------------
   // SimpleSexp2::Heap:: releaseCell

   void SimpleSexp2::Heap::releaseCell(SimpleSexp2 * cell, 
				       Bool recursivep) {
      if(isNull(cell) || cell->isNil()) return ;
      if(recursivep && cell->isCons()) {
	 releaseCell(cell->car(), recursivep) ;
	 releaseCell(cell->cdr(), recursivep) ;
      }
      putPtr(cell) ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp2::Heap:: append

   SimpleSexp2 * SimpleSexp2::Heap::append(SimpleSexp2 * list1, 
					   SimpleSexp2 * list2) {
      if(!list1->isCons()) 
	 return list2 ;
      else
	 return cons(list1->car(),append(list1->cdr(),list2)) ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp2::Heap:: concat

   SimpleSexp2 * SimpleSexp2::Heap::concat(SimpleSexp2 * list1, 
					   SimpleSexp2 * list2) {
      if(!list1->isCons()) {
	 list1 = list2 ;
	 // this has no effect.
      } else {
	 SimpleSexp2 * cell ;
	 for(cell = list1 ; cell->cdr()->isCons() ; cell = cell->cdr()) ;
	 cell->setCdr(list2) ;
      }
      return list1 ;
   } ;

   //------------------------------------------------------------
   // SimpleSexp2::Heap:: append

   SimpleSexp2 * SimpleSexp2::Heap::addLast(SimpleSexp2 * lst, 
					    SimpleSexp2 * lastelm) {
      SimpleSexp2 * l = lst ;
      for(; l->cdr()->isCons() ; l = l->cdr()) {} ;
      l->setCdr(list(lastelm)) ;
      return lst ;
   } ;
} ;
