// -*- Mode: c++ -*-
//Header:
//File: SimpleSexp2.h
//Author: NODA, Itsuki
//Date: 2001/06/26
//

//ModifyHistory:
// 2001/06/26: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_SimpleSexp2_h_
#define _itk_SimpleSexp2_h_
//////////////////////////////////////////////////////////////////////

#include "itk/btype.h"
#include "itk/utility.h"
#include "itk/TimeVal.h"
#include "itk/WithDescriber.h"
#include "itk/SString.h"
#include "itk/Buffer.h"
#include "itk/RcylHeapT.h"
#include "itk/Mutex.h"
#include <fstream.h>
#include <ctype.h>
#include <strstream.h>


//----------------------------------------------------------------------
// char definitions

#define ITK_SEXP2_CONS_BEGIN_CHAR	'('
#define ITK_SEXP2_CONS_END_CHAR		')'

#define ITK_SEXP2_CONS_SEP_CHAR		' '

#define ITK_SEXP2_STRING_BEGIN_CHAR	'"'
#define ITK_SEXP2_STRING_END_CHAR	'"'
#define ITK_SEXP2_STRING_ESCAPE_CHAR	'\\'

#define ITK_SEXP2_COMMENT_BEGIN_CHAR	';'
#define ITK_SEXP2_COMMENT_END_CHAR	'\n'

#define ITK_SEXP2_NUMBER_STR_MAX	128

//----------------------------------------------------------------------
// interface to bison and flex

namespace Itk { class SimpleSexp2 ;} ;

//#define YYSTYPE Itk::Int
#define YYSTYPE Itk::SimpleSexp2 *

extern int SimpleSexp2yyerror(char * s) ;
extern int SimpleSexp2yylex(YYSTYPE *lvalp, void *scanner) ;
extern int SimpleSexp2yyparse(void * scanner) ;

namespace Itk {

   class SimpleSexp2Lexer ;

   /*--------------------*/
   /**
    *
    */

   extern SimpleSexp2Lexer * newSimpleSexp2Lexer() ;

   /*--------------------*/
   /**
    *
    */

   extern void deleteSimpleSexp2Lexer(SimpleSexp2Lexer * lexer) ;

   /*--------------------*/
   /**
    *
    */

   extern void assignSimpleSexp2LexerScanner(SimpleSexp2Lexer * lexer,
					     void * scanner) ;
   
   //======================================================================
   // class SimpleSexp2
   /*--------------------*/
   /**
    *
    */

   class SimpleSexp2 : public WithDescriber {

	 //--------------------------------------------------
	 // SimpleSexp2:: local types

	 //----------------------------------------
	 // SimpleSexp2:: enum Tag
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 enum Tag {
	    T_Nil,
	    T_Cons,
	    T_Atom,	// generic atom
	    T_Symbol,	// Symbol Atom
	    T_Int,	// Int Atom
	    T_Flt,	// Flt Atom
	    T_Error
	 } ;

	 //----------------------------------------
	 // SimpleSexp2:: class Heap
      public:
	 class Heap ; // defined below

	 //----------------------------------------
	 // SimpleSexp2:: class Scanner
      public:
	 class Scanner ; // defined below

	 //==================================================
	 //  SimpleSexp2:: body

	 //--------------------------------------------------
	 // SimpleSexp2:: static const
	 /*--------------------*/
	 /**
	  *
	  */
      public:

	 static SimpleSexp2  _Nil ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * Nil ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2  _Eof ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * Eof ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2  _Error ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * Error ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2  _TrueValue ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * TrueValue ;

	 //--------------------------------------------------
	 // SimpleSexp2:: member

      public:
	 Tag tag ;
	 SubString str ;

	 Int ival ;
	 Flt fval ;

	 SimpleSexp2 * car_ ;	// used only if tag == T_Cons 
	 SimpleSexp2 * cdr_ ;	// used only if tag == T_Cons 

	 //--------------------------------------------------
	 // SimpleSexp2:: constructor
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2() {} ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2(Tag t, const SubString & s) {
	    tag = t ;
	    str = s ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2(Tag t, const char * s) {
	    tag = t ;
	    str.copy(s) ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: let
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * set(const SimpleSexp2 * original, 
			   Bool strcopyp = False,
			   Heap * heap = &sharedHeap) {
	    setWithoutStr(original) ;
	    if(strcopyp) {
	       str = heap->getSString(original->str.length()) ;
	       str.copy(original->str) ;
	    } else {
	       str = original->str ;
	    }
	    
	    return this ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * setWithoutStr(const SimpleSexp2 * original) {
	    tag = original->tag ;
	    ival = original->ival ;
	    fval = original->fval ;
	    car_ = original->car_ ;
	    cdr_ = original->cdr_ ;
	    return this ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: type check
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isNullPtr() const { return isNull(this) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isNil() const { return !isNullPtr() && tag == T_Nil ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isCons() const { return !isNullPtr() && tag == T_Cons ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isError() const { return !isNullPtr() && tag == T_Error ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isEof() const { return this == Eof ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isAtom() const { 
	    return !isNullPtr() 
	       && (tag == T_Atom ||
		   tag == T_Int ||
		   tag == T_Flt ||
		   tag == T_Symbol) ;
	 } ;
	 
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isInt(Bool tryscan = True) const {
	    if(isNullPtr()) return False ;
	    if(tag == T_Int) return True ;
	    if(tryscan) {
	       Int dummy ;
	       return scanAsIntTo(dummy) ;
	    } else {
	       return False ;
	    }
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isFlt(Bool tryscan = True) const {
	    if(isNullPtr()) return False ;
	    if(tag == T_Flt) return True ;
	    if(tryscan) {
	       Flt dummy ;
	       return scanAsFltTo(dummy) ;
	    } else {
	       return False ;
	    }
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isNumber(Bool tryscan = True) const {
	    return isInt(tryscan) || isFlt(tryscan) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isSymbol(Bool tryscan = True) const { 
	    if(isNullPtr()) return False ;
	    if(tag == T_Symbol) return True ;
	    if(tryscan) {
	       return !isInt() && !isFlt() ;
	    } else {
	       return False ;
	    }
	 } ;
	 
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isString() const { 
	    return isAtom()
	       && (str.first() == ITK_SEXP2_STRING_BEGIN_CHAR) ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: newcells (static operations)

	 //----------------------------------------
	 // SimpleSexp2:: newatom (static operations)
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * newAtom(SubString s, 
				      Heap * h = &sharedHeap) {
	    return h->newAtom(s) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * newAtom(char * s, 
				      Heap * h = &sharedHeap) {
	    return newAtom(SubString(s),h) ;
	 } ;
	 
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * newAtom(SubString s, Scanner * scanner) {
	    return newAtom(s,scanner->heap) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * newAtom(char * s, Scanner * scanner) {
	    return newAtom(SubString(s),scanner) ;
	 } ;

	 //----------------------------------------
	 // SimpleSexp2:: cons (static operations)
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * cons(SimpleSexp2 * carVal,
				   SimpleSexp2 * cdrVal,
				   Heap * h = &sharedHeap) {
	    return h->cons(carVal,cdrVal) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * cons(SimpleSexp2 * carVal,
				   SimpleSexp2 * cdrVal,
				   Scanner * scanner) {
	    return cons(carVal,cdrVal,scanner->heap) ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: output to buffer  // ??? work?
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void outputTo(Buffer & buffer) const ;

	 //--------------------------------------------------
	 // SimpleSexp2:: name
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SubString pname() const { return str ; } ;
	 
	 //------------------------------------------------------------
	 // SimpleSexp2:: set tag and value

	 //----------------------------------------
	 // SimpleSexp2:: set intval
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2* setIntVal(Int iv) {
	    tag = T_Int ;
	    ival = iv ;
	    str.setNull() ;
	    return this ;
	 } ;

	 //----------------------------------------
	 // SimpleSexp2:: set fltval
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2* setFltVal(Flt fv) {
	    tag = T_Flt ;
	    fval = fv ;
	    str.setNull() ;
	    return this ;
	 } ;

	 //----------------------------------------
	 // SimpleSexp2:: set symbol
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2* setSymVal(SubString sv) {
	    tag = T_Symbol ;
	    str.set(sv) ;
	    return this ;
	 } ;

	 //----------------------------------------
	 // SimpleSexp2:: set symbol
	 /**
	  *  fix ambiguous tag (T_Atom) to specific type (Int, Flt, Symbol)
	  */
      public:
	 SimpleSexp2 * fixTag(Bool recursivep = True) ;

	 //------------------------------------------------------------
	 // SimpleSexp2:: list operations


	 //--------------------------------------------------
	 // SimpleSexp2:: car & cdr
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * car() const { return car_ ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * cdr() const { return cdr_ ; } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: rplaca & rplacd
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * rplaca(SimpleSexp2 * v) { return car_ = v ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * rplacd(SimpleSexp2 * v) { return cdr_ = v ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * setCar(SimpleSexp2 * v) { return rplaca(v) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * setCdr(SimpleSexp2 * v) { return rplacd(v) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * setCarCdr(SimpleSexp2 * carval, SimpleSexp2 * cdrval) {
	    setCar(carval) ; 
	    setCdr(cdrval) ;
	    return this ; 
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * setCons(SimpleSexp2 * carval, SimpleSexp2 * cdrval) {
	    tag = T_Cons ;
	    return setCarCdr(carval,cdrval) ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: list operations
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * nthcdr(UInt n) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * nth(UInt n) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * first() { return nth(0) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * second() { return nth(1) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * rest() { return nthcdr(1) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * last() { return nthcdr(length()-1) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * lastone() { return nth(length()-1) ; } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: equal
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool equal(const SimpleSexp2 * sexp) const { return equal(*sexp) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool equal(const SimpleSexp2 & sexp) const {
	    if(isNullPtr() || isError()) return False ;
	    switch(tag) {
	       case T_Nil : return sexp.isNil() ; break ;
	       case T_Symbol :
	       case T_Atom : return sexp.equal(str) ; break ;
	       case T_Int : return sexp.isInt() && ival == sexp.ival ;
		  break ;
	       case T_Flt : return sexp.isFlt() && fval == sexp.fval ;
		  break ;
	       case T_Cons : {
		  if(sexp.isCons()) {
		     return car()->equal(sexp.car()) 
			&&  cdr()->equal(sexp.cdr()) ;
		  } else {
		     return False ;
		  }
	       }
	       default : return False ;
	    }
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool equal(const char * sym) const {
	    return !isNullPtr() && isAtom() && str == sym ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool equal(const SubString & sym) const {
	    return !isNullPtr() && isAtom() && str == sym ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: length
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 UInt length() const ;

	 //--------------------------------------------------
	 // SimpleSexp2:: assoc table operation
	 //	assoc() returns (key . value) pair
	 //	assocValue() return value.
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * assoc(const SimpleSexp2 * const key) const ;
	 SimpleSexp2 * assoc(const SubString & key) const ;
	 SimpleSexp2 * assocValue(const SimpleSexp2 * const key) const ;
	 SimpleSexp2 * assocValue(const SubString & key) const ;

	 //--------------------------------------------------
	 // SimpleSexp2::proparty table operation
	 //	passoc() return value
	 //	passocPos() return (key value . <rest pairs>)
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * passoc(const SimpleSexp2 * const key) const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * passoc(const SubString & key) const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * passocPos(const SimpleSexp2 * const key) const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * passocPos(const SubString & key) const ;

	 //--------------------------------------------------
	 // SimpleSexp2:: duplicate
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * dup(Heap & heap = sharedHeap,
			   Bool strcopyp = True) const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SimpleSexp2 * dup(Heap * heap = &sharedHeap,
			   Bool strcopyp = True) const {
	    return dup(*heap,strcopyp) ;
	 } ;

	 //------------------------------------------------------------
	 // SimpleSexp2:: get values

	 //----------------------------------------
	 // SimpleSexp2:: get int value
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int intVal(Int errorVal = 0) {
	    if(isInt(False)) {
	       return ival ;
	    } else if (isFlt(False)) {
	       return (Int)fval ;
	    } else {
	       Bool r = scanAsIntTo(ival) ;
	       if(r) {
		  tag = T_Int ;
		  return ival ;
	       } else {
		  return errorVal ;
	       }
	    }
	 } ;

	 //----------------------------------------
	 // SimpleSexp2:: get int value
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Flt fltVal(Flt errorVal = 0.0) {
	    if(isFlt(False)) {
	       return fval ;
	    } else if(isInt(False)) {
	       return Flt(ival) ;
	    } else {
	       Bool r = scanAsFltTo(fval) ;
	       if(r) {
		  tag = T_Flt ;
		  return fval ;
	       } else {
		  return errorVal ;
	       }
	    }
	 } ;

	 //----------------------------------------
	 // SimpleSexp2:: get int value
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 const SubString & symVal() {
	    if(isSymbol(False)) {
	       return str ;
	    } else {
	       if(isAtom()) {
		  tag = T_Symbol ;
		  return str ;
	       } else {
		  return SString::null ;
	       }
	    }
	 } ;

	 //------------------------------------------------------------
	 // SimpleSexp2:: scan 

	 //--------------------------------------------------
	 // SimpleSexp2:: scan (static operations) with scanner
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * scan(Scanner * scanner, Bool resetp = True,
				   Heap * heap = ITK_NULLPTR) {
	    Heap * origheap = scanner->heap ;

	    if(!isNull(heap)) scanner->heap = heap ;
	    SimpleSexp2 * r = scanner->scan(resetp) ;
	    if(!isNull(heap)) scanner->heap = origheap ;

	    return r ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * scan(Scanner & scanner, Bool resetp = True,
				   Heap * heap = ITK_NULLPTR) {
	    return scan(&scanner,resetp,heap) ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: scan (static operations) 
	 //		without scanner (using shared scanner)
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * scan(istream * istr, Bool resetp = True,
				   Heap * heap = ITK_NULLPTR) {
	    sharedScanner.reset(istr,False) ;
	    if(!isNull(heap)) sharedScanner.heap = heap ;

	    return sharedScanner.scan(resetp) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * scan(istream & istr, Bool resetp = True,
				   Heap * heap = ITK_NULLPTR) {
	    return scan(&istr,resetp,heap) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * scan(Buffer & buf, Bool resetp = True,
				   Heap * heap = ITK_NULLPTR) {
	    istrstream str((const char *)buf.head,buf.bodyLength()) ;
	    return scan(str,resetp,heap) ;
	 } ;
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * scan(Buffer * buf, Bool resetp = True,
				   Heap * heap = ITK_NULLPTR) {
	    return scan(*buf,resetp,heap) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * scan(const char * str, Bool resetp = True,
				   Heap * heap = ITK_NULLPTR) {
	    SubString s(const_cast<char *>(str)) ;
	    Buffer buf(s) ;
	    return scan(buf,resetp,heap) ;

	    // BUG?
	    // If we use 'scan(Buffer(SubString(s)),resetp)', then
	    // it calls 'scan((char*)Buffer(SubString(s)),resetp)',
	    // and enter infinite loop.
	 } ;
	 
	 //--------------------------------------------------
	 // SimpleSexp2:: scan As Number
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int scanAsInt() const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int scanAsInt() ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool scanAsIntTo(Int & val) const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool scanAsIntTo(Int & val) ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Flt scanAsFlt() const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Flt scanAsFlt() ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool scanAsFltTo(Flt & val) const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool scanAsFltTo(Flt & val) ;
	 
	 //--------------------------------------------------
	 // SimpleSexp2:: scan As Symbol
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SubString scanAsSymbol() const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 const SubString & scanAsConstSymbol() const {
	    return str ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: scan As SubString (naked if it start quote mark)
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 SubString scanAsString(Bool naked = True, Bool copyp = False) const ;

	 //------------------------------------------------------------
	 // SimpleSexp2:: describe
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 virtual void describe(ostream& o, Bool detailp = True) const ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 virtual void describePtr(ostream& o, Bool detailp = True) const {
	    describe(o,detailp) ;
	 } ;
     
	 //============================================================
	 // SimpleSexp2:: class Heap & operations
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 class Heap : public RcylHeapT<SimpleSexp2>, public Mutex { 

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: member
	    public:
	       static const UInt StrBufSize = 8192 ;
	       Buffer strbuf ;
	       Scanner * scanner ;	// local scanner (used in scan())

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: construct
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       Heap() { init() ; } ;

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: init
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void init() { 
		  strbuf.alloc(StrBufSize) ; 
		  scanner = ITK_NULLPTR ;
	       } ;

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: clear
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void clear() { 
		  strbuf.clear() ;
		  RcylHeap::clear() ;  // HeapT<SimpleSexp2>:reset() ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: get SString & char* from strbuf
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SubString getSString(Int len) {
		  return strbuf.popSubString(len+1) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       char * getCString(Int len) {
		  SubString s = getSString(len) ;
		  s.terminate() ;
		  return (char *)s ;
	       } ;

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: get newcell
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newCell() { return getPtr() ; } ;

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: release newcell
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void releaseCell(SimpleSexp2 * cell, Bool recursivep = False) ;

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: new atom (empty)
	       //    atype should be one of T_Atom, T_Symbol, T_Int, T_Flt
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newAtom(Tag atype = T_Atom) {
		  SimpleSexp2 * cell = newCell() ;
		  cell->tag = atype ;
		  cell->str.setNull() ;
		  return cell ;
	       } ;

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: new atom with SString
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newAtom(const SubString & s, 
				     Tag atype = T_Atom) {
		  return newAtom(const_cast<SubString&>(s),True,atype) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newAtom(SubString & s, 
				     Bool copyp = True,
				     Tag atype = T_Atom) {
		  SimpleSexp2 * cell = newAtom(atype) ;
		  if(copyp) {
		     cell->str = getSString(s.length()) ;
		     cell->str.copy(s) ;
		  } else {
		     cell->str = s ;
		  }
		  return cell ;
	       } ;

	       //--------------------------------------------------
	       // SimpleSexp2::Heap:: new atom with char*
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newAtom(const char * s, 
				     Bool copyp = True,
				     Tag atype = T_Atom) {
		  SimpleSexp2 * cell = newAtom(atype) ;
		  if(copyp) {
		     cell->str = getSString(strlen(s)) ;
		     cell->str.copy(s) ;
		  } else {
		     cell->str = const_cast<char*>(s) ;
		  }
		  return cell ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: new symbol
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newSymbol(const SubString & s) {
		  return newSymbol(const_cast<SubString &>(s),True) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newSymbol(SubString & s, Bool copyp = True){
		  return newAtom(s,copyp,T_Symbol) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newSymbol(const char * s, Bool copyp = True) {
		  return newAtom(s,copyp,T_Symbol) ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: new int
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newInt(const Int & i) {
		  SimpleSexp2 * cell = newAtom(T_Int) ;
		  return cell->setIntVal(i) ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: new flt
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newFlt(const Flt & f) {
		  SimpleSexp2 * cell = newAtom(T_Flt) ;
		  return cell->setFltVal(f) ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: cons
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * cons(SimpleSexp2 * car, SimpleSexp2 * cdr) {
		  SimpleSexp2 * cell = newCell() ;
		  cell->tag = T_Cons ;
		  cell->str.setNull() ;
		  cell->setCarCdr(car,cdr) ;
		  return cell ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: list
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list() { return Nil ; } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0) { 
		  return cons(s0,list()) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0, SimpleSexp2 * s1) {
		  return cons(s0,list(s1)) ; 
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0, 
				  SimpleSexp2 * s1, 
				  SimpleSexp2 * s2) {
		  return cons(s0,list(s1,s2)) ; 
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0, 
				  SimpleSexp2 * s1, 
				  SimpleSexp2 * s2,
				  SimpleSexp2 * s3) {
		  return cons(s0,list(s1,s2,s3)) ; 
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0, 
				  SimpleSexp2 * s1, 
				  SimpleSexp2 * s2,
				  SimpleSexp2 * s3,
				  SimpleSexp2 * s4) {
		  return cons(s0,list(s1,s2,s3,s4)) ; 
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0, 
				  SimpleSexp2 * s1, 
				  SimpleSexp2 * s2,
				  SimpleSexp2 * s3,
				  SimpleSexp2 * s4,
				  SimpleSexp2 * s5) {
		  return cons(s0,list(s1,s2,s3,s4,s5)) ; 
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0, 
				  SimpleSexp2 * s1, 
				  SimpleSexp2 * s2,
				  SimpleSexp2 * s3,
				  SimpleSexp2 * s4,
				  SimpleSexp2 * s5,
				  SimpleSexp2 * s6) {
		  return cons(s0,list(s1,s2,s3,s4,s5,s6)) ; 
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * list(SimpleSexp2 * s0, 
				  SimpleSexp2 * s1, 
				  SimpleSexp2 * s2,
				  SimpleSexp2 * s3,
				  SimpleSexp2 * s4,
				  SimpleSexp2 * s5,
				  SimpleSexp2 * s6,
				  SimpleSexp2 * s7) {
		  return cons(s0,list(s1,s2,s3,s4,s5,s6,s7)) ; 
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: append & concat
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * append(SimpleSexp2 * list1, SimpleSexp2 * list2) ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * concat(SimpleSexp2 * list1, SimpleSexp2 * list2) ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: addFirst (push & replace)
	    public:
	       SimpleSexp2 * addFirst(SimpleSexp2 * top, 
				      SimpleSexp2 * & list) {
		  list = cons(top,list) ;
		  return list ;
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: addLast
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * addLast(SimpleSexp2 * lst, SimpleSexp2 * lastelm);

	       //----------------------------------------
	       // SimpleSexp2::Heap:: duplicate
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * dup(SimpleSexp2 * orginal, 
				 Bool strcopyp = True) {
		  return orginal->dup(this,strcopyp);
	       } ;

	       //----------------------------------------
	       // SimpleSexp2::Heap:: scan
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * scan(istream * istr, Bool clearheapp = False) {
		  if(isNull(scanner)) scanner = new Scanner(this) ;
		  scanner->istr = istr ;
		  return scanner->scan(clearheapp) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * scan(istream & istr, Bool clearheapp = False) {
		  return scan(&istr,clearheapp) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * scan(Buffer * b, Bool clearheapp = False) {
		  if(isNull(scanner)) scanner = new Scanner(this) ;
		  scanner->reset(b) ;
		  return scanner->scan(clearheapp) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * scan(Buffer & b, Bool clearheapp = False) {
		  return scan(&b,clearheapp) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * scan(const char * str, Bool clearheapp = False) {
		  SubString s(const_cast<char*>(str)) ;
		  Buffer buf(s) ;
		  return scan(buf,clearheapp) ;
	       } ;
	 } ;
	 
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static Heap sharedHeap ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static SimpleSexp2 * get(Heap * heap = &sharedHeap) {
	    return heap->newCell() ; 
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static void cleanHeap() { sharedHeap.clear() ; } ; // not used?

	 //============================================================
	 // SimpleSexp2:: class Scanner
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 class Scanner : public Mutex {
	       //------------------------------
	       // SimpleSexp2::Scanner:: typedef outside defs
	    public:
	       typedef SimpleSexp2Lexer Lexer ;

	       static Lexer * newLexer() { return newSimpleSexp2Lexer() ;} ;
	       static void deleteLexer(SimpleSexp2Lexer * lexer) { 
		  return deleteSimpleSexp2Lexer(lexer) ;
	       } ;

	       //------------------------------
	       // SimpleSexp2::Scanner:: members
	    public:
	       Lexer * lexer ;
	       Heap * heap ;
	       istream * istr ;
	       SimpleSexp2 * result ;
	       Bool deletablep ;  // istr is deletable or not
	       Bool ownheap ;     // heap is owned by scanner or not

	       //------------------------------
	       // SimpleSexp2::Scanner:: constructor
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       /// construct with nothing
	       Scanner() {
		  init((istream*)ITK_NULLPTR,False,ITK_NULLPTR) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       /// construct from heap
	       Scanner(Heap * h) {
		  init((istream*)ITK_NULLPTR,False,h) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       Scanner(istream * istr,
		       Bool deletablep = False,
		       Heap * h = ITK_NULLPTR) {
		  init(istr,deletablep,h) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       Scanner(istream & istr, 
		       Bool deletablep = False,
		       Heap  * h = ITK_NULLPTR) {
		  init(istr,deletablep,h) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       Scanner(Buffer * b, 
		       Bool deletablep = False,
		       Heap * h = ITK_NULLPTR) {
		  init(b,deletablep,h) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       Scanner(Buffer & b, 
		       Bool deletablep = False,
		       Heap * h = ITK_NULLPTR) {
		  init(b,deletablep,h) ;
	       } ;

	       //------------------------------
	       // SimpleSexp2::Scanner:: desctuctor
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       ~Scanner() {
		  deleteLexer(lexer) ;
		  if(!isNull(istr) && deletablep) delete istr ;
		  if(ownheap) delete heap ;
	       } ;

	       //------------------------------
	       // SimpleSexp2::Scanner:: init
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void init(istream * is = ITK_NULLPTR, 
			 Bool deletable = True,
			 Heap * h = ITK_NULLPTR) {
		  lock() ; {
		     lexer = newLexer() ;
		     assignSimpleSexp2LexerScanner(lexer,this) ;
		     istr = is ;
		     deletablep = deletable ;
		     heap = h ;
		     ownheap = False ;
		     if(isNull(heap)) {
			heap = new Heap ;
			ownheap = True ;
		     }
		  } ; unlock() ;
	       } ;
	       
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void init(istream & istr, 
			 Bool deletable = True,
			 Heap  * h = ITK_NULLPTR) {
		  init(&istr,h) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void init(Buffer * b, 
			 Bool deletable = True,
			 Heap * h = ITK_NULLPTR) {
		  istrstream * str = b->newIStrStream() ;
		  init(str,h) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void init(Buffer & b, 
			 Bool deletable = True,
			 Heap * h = ITK_NULLPTR) {
		  init(&b,h) ;
	       } ;

	       //------------------------------
	       // SimpleSexp2::Scanner:: reset
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void reset(istream * is, Bool deletable = True) {
		  lock() ; {
		     if(!isNull(istr) && deletablep) delete istr ;
		     istr = is ;
		     deletablep = deletable ;
		  } ; unlock() ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void reset(Buffer * b) {
		  istrstream * str = b->newIStrStream() ;
		  reset(str,True) ;
	       } ;
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       void reset(Buffer & b) { return reset(&b) ; } ;
		  
	       //------------------------------
	       // SimpleSexp2::Scanner:: call parser
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * scan(Bool resetp = True) {
		  lock() ; {
		     if(resetp) {
			heap->clear() ;
		     }
		     if(SimpleSexp2yyparse(this)) {
			if(!result->isEof()) {
			   ITK_WRN(ITK_HERE << 
				   "SimpleSexp2::scan(): yyparse error.") ;
			   result = Error ;
			} else {
			   // in case of EOF, do nothing
			}
		     } else {
			// in the nomal case, do nothing
		     } 
		  } ; unlock() ;
		  return result ;
	       } ;
	       //------------------------------
	       // SimpleSexp2::Scanner:: new Atom 
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newAtom(const char * text, Bool copyp = True) {
		  return heap->newAtom(text,copyp) ;
	       } ;

	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * newAtom(SubString str, Bool copyp = True) {
		  return heap->newAtom(str,copyp) ;
	       } ;

	       //------------------------------
	       // SimpleSexp2::Scanner:: new Cons
	       /*--------------------*/
	       /**
		*
		*/
	    public:
	       SimpleSexp2 * cons(SimpleSexp2 * car, SimpleSexp2 * cdr) {
		  return heap->cons(car,cdr) ;
	       } ;
	 } ;

	 //--------------------------------------------------
	 // SimpleSexp2:: Scanner Operation
	 /*--------------------*/
	 /**
	  *
	  */
   public:
     static Scanner sharedScanner ;
   };
} ;	 
	 
	    

//////////////////////////////////////////////////////////////////////
#endif
