// -*- Mode:C++ -*-
//Header:
//File: Socket.cc
//Author: NODA, Itsuki
//Date: 1999/05/19
//

//ModifyHistory:
// 1999/05/19: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#include "itk/Socket.h"

namespace Itk {

//----------------------------------------------------------------------
// init & setup

//--------------------------------------------------
// remote

   Bool Socket::setupRemote(char* host, const UInt port) {
   
      ///// clean up remote addr
      bzero((char *)(&remoteAddr),sizeof(remoteAddr)) ;

      ///// set inet family flag 
      remoteAddr.sin_family = AF_INET ;

      ///// find srever host address 
      unsigned long int s_addr = inet_addr(host) ;
      if(s_addr == (unsigned long int)-1) {
	 struct hostent *h = (struct hostent *)gethostbyname(host) ;
	 if(isNull(h)) {
	    ITK_ERR("Wrong host name: " << host) ;
	    return False ;
	 }
	 s_addr = 
	    inet_addr(inet_ntoa(*(struct in_addr *)(h->h_addr_list[0]))) ;
      } 
      remoteAddr.sin_addr.s_addr = s_addr ;

      ///// set port number
      remoteAddr.sin_port = htons(port) ;

      return True ;

   }
   
//--------------------------------------------------
// local

   Bool Socket::setupLocal(const Protocol proto, const UInt port) {

      protocol = proto ;

      //--------------------------------------------------
      // setup socketfd

      ///// get socketfd
      if        (protocol == UDP) {
	 socketfd = socket(AF_INET,SOCK_DGRAM,0) ;
      } else if (protocol == TCP) {
	 socketfd = socket(AF_INET,SOCK_STREAM,0) ;
      } else {
	 ITK_ERR("This protocol (" << (int)protocol << 
		 ") is not supported yet.") ;
      }
      if(socketfd < 0) {
	 ITK_ERR("Can't open socket.") ;
	 return False ;
      }

      //--------------------------------------------------
      // setup socketfd

      bzero((char*)&localAddr,sizeof(localAddr)) ;
      localAddr.sin_family = AF_INET ;
      localAddr.sin_addr.s_addr = htonl(INADDR_ANY) ;
      localAddr.sin_port = htons((short unsigned int)port) ;
      
      if(bind(socketfd,(struct sockaddr*)&localAddr,sizeof(localAddr)) < 0) {
	 ITK_ERR("Can't bind local address.") ;
	 return False ;
      }

      socklen_t l = sizeof(localAddr) ;
      getsockname(socketfd,(struct sockaddr*)&localAddr,&l) ;

      return True ;
   }

//--------------------------------------------------
// Blocking Mode

   Bool Socket::setBlocking(const Bool blockingp) {
      int flag ;
      if(blockingp) flag = 0 ;
      else          flag = 1 ;

      if(ioctl(socketfd,FIONBIO,&flag) < 0) {
	 ITK_ERR("Can't ioctl on socket.") ;
	 return False ;
      }
      return True ;
   }
      
//--------------------------------------------------
// send / receive

   Bool Socket::send(const Buffer& buf) {
      SubString rbuf = buf.body() ;
      UInt l = buf.length() ;
      if (sendto(socketfd, (char*)rbuf, l, 0, (struct sockaddr*)&remoteAddr,
		 sizeof(remoteAddr)) != (int)l) {
	 ITK_WRN(*this << ": send Error. [" << errno << "]" 
		 << getSystemErrorMsg(errno)) ;
	 return False ;
      } else {
	 return True ;
      }
   }

   Bool Socket::receive(Buffer& buf, UInt const size) {
      if(size > buf.restLength()) {
	 ITK_WRN("buffer size is too small.") ;
	 return False ;
      }

      socklen_t saddrlen = sizeof(remoteAddr) ;

      ///// receive
      Int r = recvfrom(socketfd, buf.tail, buf.restLength(), 0,
		       (struct sockaddr *)&remoteAddr, &saddrlen) ;

      ///// check error
      if (r < 0) {
	 if (errno != ENOTSOCK && errno != EWOULDBLOCK && errno != 0) {
	    ITK_WRN(*this << ": recvfrom Error. [" 
		    << errno << "]" << getSystemErrorMsg(errno)) ;
	 }
	 return False ;
      } 

      //// terminate buffer
      buf.tail = &(buf.tail[r]) ;
      buf.terminate() ;

      ///// return
      if(r == 0)
	 return False ;
      else
	 return True ;
   }

//--------------------------------------------------
// send from / receive to data

   Bool Socket::send(void* data, const UInt size) {
      Int r = sendto(socketfd, data, size, 0, 
		     (struct sockaddr*)&remoteAddr, sizeof(remoteAddr)) ;
      //cerr << "Socket::send" << " size=" << size 
      //	<< ", r=" << r << endl ;
      if(r != (int)size) {
	 ITK_WRN(*this << ": send Error. [" << errno << "]" 
		 << getSystemErrorMsg(errno)) ;
	 return False ;
      } else {
	 return True ;
      }
   } 

   Bool Socket::receive(void* data, const UInt size) {
      UInt n = 0 ;
      while(n < size) {
	 char* d = (char*)data ;
	 socklen_t l = sizeof(remoteAddr) ;
	 Int r = recvfrom(socketfd, &(d[n]), size-n, 0,
			  (struct sockaddr*)&remoteAddr, &l) ;
      
	 if (r < 0) {
	    if (errno != ENOTSOCK && errno != EWOULDBLOCK && errno != 0) {
	       ITK_WRN(*this << "receive Error. [" << errno << "]" 
		       << getSystemErrorMsg(errno)) ;
	    }
	    return False ;
	 } 
	 n += r ;
	 //cerr << "Socket::receive: size=" << size 
	 //   << ", r=" << r 
	 //   << ", n=" << n << endl ;
      }
      ///// return
      // if(r == 0)
      if(n == size)
	 return False ;
      else
	 return True ;
   } 

//--------------------------------------------------
// write from / read to Buffer

   Bool Socket::write(const Buffer& buf) {
      SubString rbuf = buf.body() ;
      Int l = buf.length() ;
      if (::write(socketfd, (char*)rbuf, l) != l)
	 return False ;
      else
	 return True ;
   }

   Bool Socket::read(Buffer& buf, UInt const size) {
      if(size > buf.restLength()) {
	 ITK_WRN("buffer size is too small") ;
	 return False ;
      }

      ///// receive
      Int r = ::read(socketfd, buf.tail, size) ;

      ///// check error
      if (r < 0) {
	 if (errno != ENOTSOCK && errno != EWOULDBLOCK && errno != 0) {
	    ITK_WRN(*this << "recvfrom Error. [" << errno << "]" 
		    << getSystemErrorMsg(errno)) ;
	 }
	 return False ;
      } 

      //// terminate buffer
      buf.tail = &(buf.tail[r]) ;
      buf.terminate() ;

      ///// return
      if(r == 0)
	 return False ;
      else
	 return True ;
   }

//--------------------------------------------------
// write from / read to data

   Bool Socket::write(void* data, const UInt size) {
      Int r = ::write(socketfd, data, size) ;
      //cerr << "Socket::write" << " size=" << size 
      //	<< ", r=" << r << endl ;
      if(r != (int)size)
	 return False ;
      else 
	 return True ;
   } 

   Bool Socket::read(void* data, const UInt size) {
      UInt n = 0 ;
      while(n < size) {
	 char* d = (char*)data ;
	 Int r = ::read(socketfd, &(d[n]), size-n) ;
      
	 if (r < 0) {
	    if (errno != ENOTSOCK && errno != EWOULDBLOCK && errno != 0) {
	       ITK_WRN(*this << "Warning: read Error. [" << errno << "]" 
		       << getSystemErrorMsg(errno)) ;
	    }
	    return False ;
	 } 
	 n += r ;
	 //cerr << "Socket::read: size=" << size 
	 //       << ", r=" << r 
	 //       << ", n=" << n << endl ;
      }
      ///// return
      // if(r == 0)
      if(n == size)
	 return False ;
      else
	 return True ;
   } 

//--------------------------------------------------
// discard input

   void Socket::discard(const UInt size) {
      char dummy[1] ;
      for(UInt i = 0 ; i < size ; i++) {
	 ::read(socketfd, dummy, 1) ;
      }
   }

//======================================================================
// class ServerSocket

//--------------------------------------------------
// accept

   Bool ServerSocket::accept(ServerSocket& child) {
      child.copy(*this) ;
      if(isTCP()) {
	 socklen_t s = sizeof(child.remoteAddr) ;
	 child.socketfd = ::accept(socketfd,
				   (struct sockaddr*)&(child.remoteAddr),
				   &s) ;
	 if(child.socketfd < 0) return False ;
	 getsockname(child.socketfd,
		     (struct sockaddr*)&(child.localAddr),&s) ;
	 return True ;
      } else if (isUDP()) {
	 return True ;
      } else 
	 return False ;
   } 


} ;
