// -*- Mode: c++ -*-
//Header:
//File: TimeVal.h
//Author: NODA, Itsuki
//Date: 1999/05/18
//

//ModifyHistory:
// 1999/05/18: Start to create this file
// 1999/11/07: introduce Itk namespace
// 1999/11/07: Divide each class to separate file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 * Copyright (C) 1999, 2000 Itsuki Noda, Electrotechnical Laboratory, Japan
 */

#ifndef _itk_TimeVal_h_
#define _itk_TimeVal_h_
//////////////////////////////////////////////////////////////////////

extern "C" {
#include <math.h>
#include <sys/time.h>
#include <unistd.h>
#include <time.h>
}

#include <iostream.h>
#include <iomanip.h>

#include "itk/btype.h"
#include "itk/WithDescriber.h"

namespace Itk {
   
   //======================================================================
   // class TimeVal 

   /*--------------------*/
   /**
    *
    */

   class TimeVal : public WithDescriber {
      public:  
	 /* This should be "private", but it is used in Select.
	    So now, I change public. */
	 struct timeval tval ;

	 //--------------------------------------------------
	 // access to sec part & usec part

	 typedef TIMEVAL_INNER_TYPE _timeval_inner_t ;

      public:
	 const _timeval_inner_t& _usec() const { return tval.tv_usec ; } ;
	 const _timeval_inner_t& _sec() const { return tval.tv_sec ; } ;
	 _timeval_inner_t& _usec() { return tval.tv_usec ; } ;
	 _timeval_inner_t& _sec() { return tval.tv_sec ; } ;

	 //--------------------------------------------------
	 // constructor

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal(const Int& sec = 0, const Int& msec = 0, 
		 const Int& usec = 0) {
	    set(sec,msec,usec) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal(const TimeVal& tv) { set(tv) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal(const Flt& sec) { set(sec) ; } ;
	 
	 //--------------------------------------------------
	 // get time of day

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& getTimeOfDay() {
	    gettimeofday(&tval,NULL) ;
	    return *this ;
	 } ;
	 TimeVal& update() { return getTimeOfDay() ; } ;
	 
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 static TimeVal now() {
	    TimeVal tv ;
	    return tv.update() ;
	 } ;

	 //--------------------------------------------------
	 // set

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& operator=(const TimeVal& tv) { return set(tv) ; } ;
	 
	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& set(const Int& sec = 0, const Int& msec = 0, 
		      const Int& usec = 0) {
	    Int us = msec * 1000 + usec ;
	    _usec() = (_timeval_inner_t)(us % 1000000) ;
	    _sec() = (_timeval_inner_t)(sec + (us / 1000000)) ;
	    return *this ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& set(const struct timeval& tv) {
	    tval = tv ;
	    return *this ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& set(const TimeVal& tv) {
	    return set(tv.tval) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& set(const Flt& sec) {
	    Int s = (Int)floor(sec) ;
	    Int u = (Int)floor((sec - (Flt)s) * 1000000.0) ;
	    return set(s,0,u) ;
	 } ;

	 //--------------------------------------------------
	 // get total value in various unit

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int inUSec() const { 
	    return (Int)(_usec() + (_sec() * 1000000)) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int inMSec() const {
	    return (Int)((_usec() / 1000) + (_sec() * 1000)) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int inSec() const { return _sec() ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int inMin() const { return (inSec() / 60) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int inHour() const { return (inMin() / 60) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int inDay() const { return (inHour() / 24) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int inYear() const { return (inDay() / 365) ; } ;

	 //--------------------------------------------------
	 // get float value

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Flt fltSec() const {
	    return (((Flt)_sec()) + (((Flt)_usec())/1000000.0)) ;
         } ;

	 //--------------------------------------------------
	 // get struct timeval and struct timespec

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 struct timeval in_struct_timeval() const { return tval ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 struct timespec in_struct_timespec() const {
	    struct timespec tspec ;
	    tspec.tv_nsec = _usec() * 1000 ;
	    tspec.tv_sec = _sec() ;
	    return tspec ;
	 } ;

	 //--------------------------------------------------
	 // get value in various unit in suitable range

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int usec() const { return (_usec() % 1000) ; } ; // 0-1000

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int msec() const { return (_usec() / 1000) ; } ; // 0-1000

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int sec() const { return (_sec() % 60) ; } ;     // 0-60

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int min() const { return (inMin() % 60) ; } ;         // 0-60

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int hour() const { return (inHour() % 24) ; } ;        // 0-24

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int day() const { return inDay() % 365 ; } ;          // 0-365

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int year() const { return inYear() ; } ;              // 0-?

	 //--------------------------------------------------
	 // inc / dec / diff / mul / div

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& inc(const Int sec = 1, 
		      const Int msec = 0, const Int usec = 0) {
	    return set(_sec() + sec, msec, _usec() + usec) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& inc(const TimeVal& x) { 
	    return inc(x._sec(),0,x._usec()) ; 
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& dec(const Int sec = 1, 
		      const Int msec = 0, const Int usec = 0) {
	    UInt d = (msec * 1000) + usec ;
	    UInt over = 0 ;
	    if(d > (UInt)_usec()) over++ ;
	    over += d/1000000 ;
	    _usec() += (over * 1000000 - d) ;
	    _sec() -= (over + sec) ;
	    return *this ;
	 } ; 

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& dec(const TimeVal& x) {
	    return dec(x._sec(),0,x._usec()) ;
	 } ;

	 //--------------------------------------------------
	 // diff

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal diff(const TimeVal& x) const {
	    TimeVal v = x ;
	    return v.dec(*this) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int diffSec(const TimeVal& x) const { 
	    return diff(x).inSec() ; 
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int diffMSec(const TimeVal& x) const {
	    return diff(x).inMSec() ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Int diffUSec(const TimeVal& x) const {
	    return diff(x).inUSec() ;
	 } ;

	 //--------------------------------------------------
	 // mul / div

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& mul(const Flt& m) { return set(fltSec() * m) ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& div(const Flt& m) { return set(fltSec() / m) ; } ;

	 
	 //--------------------------------------------------
	 // arith. operator

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal operator+(const TimeVal& tv) const {
	    TimeVal r = *this ;
	    return r.inc(tv) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal operator-(const TimeVal& tv) const {
	    TimeVal r = *this ;
	    return r.dec(tv) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal operator*(const Flt m) const {
	    TimeVal r = *this ;
	    return r.mul(m) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal operator/(const Flt m) const {
	    TimeVal r = *this ;
	    return r.div(m) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal operator/(const TimeVal& tv) const {
	    return fltSec() / tv.fltSec() ;
	 } ;

	 //--------------------------------------------------
	 // arith. & assign operator

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& operator+=(const TimeVal& tv) {
	    return inc(tv) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& operator-=(const TimeVal& tv) {
	    return dec(tv) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& operator*(const Flt m) {
	    return mul(m) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 TimeVal& operator/(const Flt m) {
	    return div(m) ;
	 } ;

         //--------------------------------------
         // Comparison
   
	 /*--------------------*/
	 /**
	  *
	  */
      public:
         Bool operator> (const TimeVal& tv) const {
            return (_sec() > tv._sec() ||
                    (_sec() == tv._sec() && _usec() > tv._usec())) ;
         } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
         Bool operator>= (const TimeVal& tv) const {
            return (_sec() > tv._sec() ||
                    (_sec() == tv._sec() && _usec() >= tv._usec())) ;
         } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
         Bool operator< (const TimeVal& tv) const {
            return (_sec() < tv._sec() ||
                    (_sec() == tv._sec() && _usec() < tv._usec())) ;
         } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
         Bool operator<= (const TimeVal& tv) const {
            return (_sec() < tv._sec() ||
                    (_sec() == tv._sec() && _usec() <= tv._usec())) ;
         } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
         Bool operator== (const TimeVal& tv) const {
            return (_sec() == tv._sec() && _usec() == tv._usec()) ;
         } ;
   
         //--------------------------------------
         // Comparison with Now

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isPast() const { return *this < now() ; } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 Bool isFuture() const { return *this > now() ; } ;
   
	 //--------------------------------------------------
	 // transform between network byte order and host byte order

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void transN2H() { 
	    _usec() = (Int)ntohl((UInt)_usec()) ;
	    _sec() = (Int)ntohl((UInt)_usec()) ;
	 } ;

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 void transH2N() { 
	    _usec() = (Int)htonl((UInt)_usec()) ;
	    _sec() = (Int)htonl((UInt)_usec()) ;
	 } ;

      public:
	 //--------------------------------------------------
	 // full timestamp

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 char * stamp() const {
	    const int bufsize = 64 ;
	    char buf[bufsize] ;
	    strftime(buf,bufsize,"#t[%Y/%m/%d.%H:%M:%S]",localtime(&_sec())) ;
	    return strdup(buf) ;
	 } ;

	 //--------------------------------------------------
	 // describer

	 /*--------------------*/
	 /**
	  *
	  */
      public:
	 virtual void describe(ostream& ostr, 
			       const Bool detailp = True) const {
	    if(detailp) {
	       ostr << "#TimeVal[" 
		    << setfill('0') << setw(2) << year() << "/"
		    << setfill('0') << setw(3) << day() << ";"
		    << setfill('0') << setw(2) << hour() << ":" 
		    << setfill('0') << setw(2) << min() << ":" 
		    << setfill('0') << setw(2) << sec() << "."
		    << setfill('0') << setw(3) << msec() << "," 
		    << setfill('0') << setw(3) << usec() 
		    << "]" << endl ;
	    } else {
	       ostr << "#T[" 
		    << setfill('0') << setw(2) << min() << ":" 
		    << setfill('0') << setw(2) << sec() << "."
		    << setfill('0') << setw(3) << msec() << "," 
		    << setfill('0') << setw(3) << usec() 
		    << "]" ;
	    }
	 } ;
   } ;
}


//////////////////////////////////////////////////////////////////////
#endif

