// -*- Mode:C++ -*-
//Header:
//File: EvalEngine.cc
//Author: NODA, Itsuki
//Date: 2001/11/19
//

//ModifyHistory:
// 2001/11/19: Start to create this file
//EndModifyHistory:

#include "EvalEngine.h"

namespace Rescue {

   //======================================================================
   //   : Context : EvalEngine
   // class BindTable 
   
   //------------------------------
   //   : BindTable : Context : EvalEngine
   // clear()

   void 
   EvalEngine::Context::BindTable::
   clear() {
      stack().reset() ;
      sheap().clear() ;
   } ;

   //------------------------------
   //   : BindTable : Context : EvalEngine
   // find()

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::BindTable::
   find(const SubString & varname) {
      Entry * r = ITK_NULLPTR ;
      for(BindStack::Iterator e = stack().begin() ; e < stack().end() ; e++) {
	 if((*e).var == varname) {
	    r = &(*e) ;
	    break ;
	 }
      }
      return r ;
   } ;

   //------------------------------
   //   : BindTable : Context : EvalEngine
   // rfind()

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::BindTable::
   rfind(const SubString & varname) {
      Entry * r = ITK_NULLPTR ;
      for(BindStack::RevIterator e = stack().rbegin() ; 
	  e < stack().rend() ; e++) {
	 if((*e).var == varname) {
	    r = &(*e) ;
	    break ;
	 }
      }
      return r ;
   } ;

   //------------------------------
   //   : BindTable : Context : EvalEngine
   // bind()

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::BindTable::
   bind(const SubString & varname,Sexp * value, Bool copyp) {
      Entry * e = &(stack().pushTop()) ;
      e->var = varname ;
      if(copyp) {
	 e->value = sheap().dup(value,True) ;
      } else {
	 e->value = value ;
      }

      return e ;
   } ;

   //------------------------------
   //   : BindTable : Context : EvalEngine
   // unify()

   Bool 
   EvalEngine::Context::BindTable::
   unify(const SubString & varname, Sexp * value, Bool copyp) {
      Entry * r = find(varname) ;

      if(!isNull(r)) { /* when the var has already bind. */
	 return r->value->equal(value) ;
      } 
      else {           /* when the var has not bind. */
	 bind(varname,value,copyp) ;
	 return True ;
      }
   } ;

   //------------------------------
   //   : BindTable : Context : EvalEngine
   // let()

   void 
   EvalEngine::Context::BindTable::
   let(const SubString & varname, Sexp * value, Bool copyp) {
      Entry * r = find(varname) ;

      if(!isNull(r)) { /* when the var has already bind. */
	 if(copyp) {
	    sheap().releaseCell(r->value) ;
	    r->value = sheap().dup(value,True) ;
	 } else {
	    r->value = value ;
	 }
      } 
      else {           /* when the var has not bind. */
	 bind(varname,value,copyp) ;
      }
   } ;

   //------------------------------
   //   : BindTable : Context : EvalEngine
   // index() 

   EvalEngine::Context::BindTable::BindStackIndex 
   EvalEngine::Context::BindTable::
   index() const {
      return stack().toppos() ;
   } ;

   //------------------------------
   //   : BindTable : Context : EvalEngine
   // rewind() 

   void 
   EvalEngine::Context::BindTable::
   rewind(const BindStackIndex & idx) {
      stack().rewind(idx) ;
   } ;
   
   //======================================================================
   //  : EvalEngine 
   // class Context

   //----------------------------------------
   //   : Context : EvalEngine
   // statics defines

   EvalEngine::Context::BindTable
   EvalEngine::Context::
   commonvar ;

   //----------------------------------------
   //   : Context : EvalEngine
   // getValue() 
   /* [2001.12.26:I.Noda]
    * change semantics.
    * variable name is searched in local bindtable in reverse order
    * so that this find the newest entry.
    */

   Sexp * 
   EvalEngine::Context::
   getValue(const SubString & varname) {
      Sexp * r ;
      //BindTable::Entry * e = localvar().find(varname) ;
      BindTable::Entry * e = localvar().rfind(varname) ;
      if(!isNull(e)) {
	 r = e->value ;
      } else {
	 e = staticvar().find(varname) ;
	 if(!isNull(e)) {
	    r = localsheap().dup(e->value,True) ;
	 } else {
	    e = globalvar().find(varname) ;
	    if(!isNull(e)) {
	       r = localsheap().dup(e->value,True) ;
	    } else {
	       r = Sexp::Nil ;
	    }
	 }
      }
      return r ;
   } ;
   
   //----------------------------------------
   //   : Context : EvalEngine
   // local var operation
   //--------------------
   //   : Context : EvalEngine
   // clearLocal() 

   void
   EvalEngine::Context::
   clearLocal() {
      return localvar().clear() ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // findLocal() 

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::
   findLocal(const SubString & varname) {
      return localvar().find(varname) ;
   } ;
   
   //--------------------
   //   : Context : EvalEngine
   // bindLocal() 

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::
   bindLocal(const SubString & varname, Sexp * value) {
      return localvar().bind(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // unifyLocal() 

   Bool
   EvalEngine::Context::
   unifyLocal(const SubString & varname, Sexp * value) {
      return localvar().unify(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // letLocal() 

   void
   EvalEngine::Context::
   letLocal(const SubString & varname, Sexp * value) {
      return localvar().let(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // newLetLocal() 

   void
   EvalEngine::Context::
   newLetLocal(const SubString & varname, Sexp * value) {
      localvar().bind(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // indexLocal() 

   EvalEngine::Context::BindIndex
   EvalEngine::Context::
   indexLocal() const {
      return localvar().index() ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // rewindLocal() 

   void
   EvalEngine::Context::
   rewindLocal(const EvalEngine::Context::BindIndex & idx) {
      return localvar().rewind(idx) ;
   } ;

   //----------------------------------------
   //   : Context : EvalEngine
   // static var operation
   //--------------------
   //   : Context : EvalEngine
   // clearStatic() 

   void
   EvalEngine::Context::
   clearStatic() {
      return staticvar().clear() ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // findStatic() 

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::
   findStatic(const SubString & varname) {
      return staticvar().find(varname) ;
   } ;
   
   //--------------------
   //   : Context : EvalEngine
   // bindStatic() 

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::
   bindStatic(const SubString & varname, Sexp * value) {
      return staticvar().bind(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // unifyStatic() 

   Bool
   EvalEngine::Context::
   unifyStatic(const SubString & varname, Sexp * value) {
      return staticvar().unify(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // letStatic() 

   void
   EvalEngine::Context::
   letStatic(const SubString & varname, Sexp * value) {
      return staticvar().let(varname, value, True) ;
   } ;

   //----------------------------------------
   //   : Context : EvalEngine
   // global var operation
   //--------------------
   //   : Context : EvalEngine
   // clearGlobal() 

   void
   EvalEngine::Context::
   clearGlobal() {
      return globalvar().clear() ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // findGlobal() 

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::
   findGlobal(const SubString & varname) {
      return globalvar().find(varname) ;
   } ;
   
   //--------------------
   //   : Context : EvalEngine
   // bindGlobal() 

   EvalEngine::Context::BindTable::Entry * 
   EvalEngine::Context::
   bindGlobal(const SubString & varname, Sexp * value) {
      return globalvar().bind(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // unifyGlobal() 

   Bool
   EvalEngine::Context::
   unifyGlobal(const SubString & varname, Sexp * value) {
      return globalvar().unify(varname, value) ;
   } ;

   //--------------------
   //   : Context : EvalEngine
   // letGlobal() 

   void
   EvalEngine::Context::
   letGlobal(const SubString & varname, Sexp * value) {
      return globalvar().let(varname, value, True) ;
   } ;

   //======================================================================
   // class EvalEngine

   //------------------------------------------------------------
   //   : EvalEngine
   // function table

   //------------------------------
   //   : EvalEngine
   // define static vars

   EvalEngine::FuncTable 
   EvalEngine::
   functable_ ;

   EvalEngine::Context
   EvalEngine::
   commonContext ;

   //------------------------------
   //   : EvalEngine
   // register func by SubString

   void
   EvalEngine::
   registerFunc(SubString name,
		FuncType func,
		FuncTable * ftab) {
      (*ftab)[name] = func ;
   } ;

   //------------------------------
   //   : EvalEngine
   // register func char *

   void
   EvalEngine::
   registerFunc(char * name,
		FuncType func,
		FuncTable * ftab) {
      return registerFunc(SubString(name),func, ftab) ;
   } ;

   //--------------------------------------------------
   //   : EvalEngine
   // clear()

   void
   EvalEngine::
   clear() {
      context().clearLocal() ;
      // sheap().clear() ; // executed in the above operation
      // tracelevel = 0 ; /* outside of eval may use trace */ 
   } ;

   //--------------------------------------------------
   //   : EvalEngine
   // rewindpoint()

   EvalEngine::RewindPoint
   EvalEngine::
   rewindpoint() const {
      return context().indexLocal() ;
   } ;

   //--------------------------------------------------
   //   : EvalEngine
   // rewind()

   void
   EvalEngine::
   rewind(RewindPoint point) {
      context().rewindLocal(point) ;
   } ;

   //--------------------------------------------------
   //   : EvalEngine
   // evalTop()

   Sexp *
   EvalEngine::
   evalTop(Sexp * form, Bool clearp, Context * ctx) {

      if(!isNull(ctx)) context_ = ctx ;

      if(clearp) clear() ;

      return eval(form) ;
   } ;

   //--------------------------------------------------
   //   : EvalEngine
   // eval()

   Sexp *
   EvalEngine::
   eval(Sexp * form) {
      
      RSC_EVAL_TRACE_IN(this,form) ;

      Sexp * r ;
      if(form->isCons()) {
	 r = evalFunc(form) ;
      }
      else if(isVar(form)) {
	 r = context().getValue(form->symVal()) ;
      }
      else {
	 r = form ;
      }

      RSC_EVAL_TRACE_OUT(this,r << " [= " << form << "]") ;

      return r ;
   } ;

   //--------------------------------------------------
   //   : EvalEngine
   // evalFunc()

   Sexp *
   EvalEngine::
   evalFunc(Sexp * form) {
      Sexp * op = form->first() ;

      FuncTable::iterator fidx = functable().find(op->symVal()) ;

      if(fidx == functable().end()) {
	 ITK_ERR("Unknown function in: " << form) ;
	 throw 0 ;
      }
      else {
	 return (*(fidx->second))(form,this) ;
      }
   } ;

   //--------------------------------------------------
   //   : EvalEngine
   // isVar()

   Bool
   EvalEngine::
   isVar(Sexp * form) {
      if(form->isSymbol()) {
	 return form->str.first() == VarPrefixChar ;
      } else {
	 return False ;
      }
   } ;

   //======================================================================
   // system functions

   //----------------------------------------
   // sysfunc: and

   Sexp *
   sysfunc_and(Sexp * form, EvalEngine * engine) {

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;
      
      Sexp * r = Sexp::TrueValue ;

      for(Sexp * f = args ; f->isCons() ; f = f->rest()) {
	 r = engine->eval(f->first()) ;
	 if(r->isNil()) {
	    return Sexp::Nil ;
	 } 
      }
      return r ;
   } ;

   //----------------------------------------
   // sysfunc: or

   Sexp *
   sysfunc_or(Sexp * form, EvalEngine * engine) {

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;
      
      Sexp * r = Sexp::Nil ;

      for(Sexp * f = args ; f->isCons() ; f = f->rest()) {
	 r = engine->eval(f->first()) ;
	 if(!r->isNil()) {
	    return r ;
	 } 
      }
      return Sexp::Nil ;
   } ;

   //----------------------------------------
   // sysfunc: not

   Sexp *
   sysfunc_not(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,1) ;

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * r = engine->eval(args->first()) ;
      
      if(r->isNil()) {
	 return Sexp::TrueValue ;
      } else {
	 return Sexp::Nil ;
      }
   } ;

   //----------------------------------------
   // sysfunc: progn

   Sexp *
   sysfunc_progn(Sexp * form, EvalEngine * engine) {

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;
      
      Sexp * r = Sexp::Nil ;

      for(Sexp * f = args ; f->isCons() ; f = f->rest()) {
	 r = engine->eval(f->first()) ;
      }
      return r ;
   } ;

   //----------------------------------------
   // sysfunc: quote

   Sexp *
   sysfunc_quote(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,1) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      return args->nth(0) ;
   }

   //----------------------------------------
   // sysfunc: setq_local

   /* [2001.12.26:I.Noda]
    * change semantics.  Now, setq_local push new bind to local bindtable
    * rather than rewrite the old one.
    */

   Sexp *
   sysfunc_setq_local(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * var = args->nth(0) ;
      Sexp * val = engine->eval(args->nth(1)) ;

      if(!engine->isVar(var)) {
	 ITK_ERR("variable should be a symbol with prefix * :" << form) ;
	 throw 0 ;
      } 

      //engine->context().letLocal(var->symVal(),val) ;
      engine->context().newLetLocal(var->symVal(),val) ;
      
      return val ;
   }

   //----------------------------------------
   // sysfunc: setq_static

   Sexp *
   sysfunc_setq_static(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * var = args->nth(0) ;
      Sexp * val = engine->eval(args->nth(1)) ;

      if(!engine->isVar(var)) {
	 ITK_ERR("variable should be a symbol with prefix * :" << form) ;
	 throw 0 ;
      } 

      engine->context().letStatic(var->symVal(),val) ;
      
      return val ;
   }

   //----------------------------------------
   // sysfunc: setq_global

   Sexp *
   sysfunc_setq_global(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * var = args->nth(0) ;
      Sexp * val = engine->eval(args->nth(1)) ;

      if(!engine->isVar(var)) {
	 ITK_ERR("variable should be a symbol with prefix * :" << form) ;
	 throw 0 ;
      } 

      engine->context().letGlobal(var->symVal(),val) ;
      
      return val ;
   }

   //----------------------------------------
   // sysfunc: add

   Sexp *
   sysfunc_add(Sexp * form, EvalEngine * engine) {

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Flt v = 0 ;

      for(Sexp * f = args ; f->isCons() ; f = f->rest()) {
	 Sexp * r = engine->eval(f->first()) ;
	 v += r->fltVal() ;
      }
      return engine->sheap().newFlt(v) ;
   } ;

   //----------------------------------------
   // sysfunc: sub

   Sexp *
   sysfunc_sub(Sexp * form, EvalEngine * engine) {

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      if(args->length() < 1) {
	 ITK_ERR("wrong arity (>1) : " << form) ;
	 throw 0 ;
      } ;

      Sexp * r0 = engine->eval(args->first()) ;

      Flt v = 0 ;
      if(args->length() == 1) {
	 v = -(r0->fltVal()) ;
      } else {
	 v = r0->fltVal() ;
	 for(Sexp * f = args->rest() ; f->isCons() ; f = f->rest()) {
	    Sexp * r = engine->eval(f->first()) ;
	    v -= r->fltVal() ;
	 }
      }
      return engine->sheap().newFlt(v) ;
   }

   //----------------------------------------
   // sysfunc: mul

   Sexp *
   sysfunc_mul(Sexp * form, EvalEngine * engine) {

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Flt v = 1.0 ;

      for(Sexp * f = args ; f->isCons() ; f = f->rest()) {
	 Sexp * r = engine->eval(f->first()) ;
	 v *= r->fltVal() ;
      }
      return engine->sheap().newFlt(v) ;
   } ;

   //----------------------------------------
   // sysfunc: div

   Sexp *
   sysfunc_div(Sexp * form, EvalEngine * engine) {

      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      if(args->length() < 2) {
	 ITK_ERR("wrong arity (>2) : " << form) ;
	 throw 0 ;
      } ;

      Sexp * r0 = engine->eval(args->first()) ;

      Flt v = r0->fltVal() ;

      for(Sexp * f = args->rest() ; f->isCons() ; f = f->rest()) {
	 Flt n = engine->eval(f->first())->fltVal() ;
	 if(n == 0.0){
	   ITK_ERR("wrong arity (div by zero) : " << form);
	   n = 0;
	   throw 0;
	 }
	 v /= n ;
      }
      return engine->sheap().newFlt(v) ;
   } ;

  //----------------------------------------
  // sysfunc : mod
#if 0
  SimpleSexp2* sysfunc_mod(SimpleSexp2* form, EvalEngine* engine){
    SimpleSexp2* args = form->rest();
    if(args->length() < 2){
      ITK_ERR("wrong arity (>2) : " << form);
      throw 0;
    }
    SimpleSexp2* r = engine->eval(args->first());
    Flt v = r->FltVal();
    for(SimpleSexp2* f = args->rest(); f->isCons(); f = f->rest()){
      Int n = engine->eval(f->first())->intVal();
      v = v % n;
    }
    return engine->sheap().newFlt(v);
  };
#endif
   //----------------------------------------
   // sysfunc: varunify
   //    (?= *var value)

   Sexp *
   sysfunc_varunify(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * var = args->nth(0) ;
      Sexp * val = engine->eval(args->nth(1)) ;

      if(engine->context().unifyLocal(var,val)) return Sexp::TrueValue ;
      else                                      return Sexp::Nil ;
   }

   //----------------------------------------
   // sysfunc: equal
   //    (== value0 value1)

   Sexp *
   sysfunc_equal(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * val0 = engine->eval(args->nth(0)) ;
      Sexp * val1 = engine->eval(args->nth(1)) ;

      if(val0->equal(val1)) return Sexp::TrueValue ;
      else                  return Sexp::Nil ;
   }

   //----------------------------------------
   // sysfunc: lt

   Sexp *
   sysfunc_lt(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * val0 = engine->eval(args->nth(0)) ;
      Sexp * val1 = engine->eval(args->nth(1)) ;

      if(val0->fltVal() < val1->fltVal()) return Sexp::TrueValue ;
      else                                return Sexp::Nil ;
   }

   //----------------------------------------
   // sysfunc: le

   Sexp *
   sysfunc_le(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * val0 = engine->eval(args->nth(0)) ;
      Sexp * val1 = engine->eval(args->nth(1)) ;

      if(val0->fltVal() <= val1->fltVal()) return Sexp::TrueValue ;
      else                                 return Sexp::Nil ;
   }

   //----------------------------------------
   // sysfunc: gt

   Sexp *
   sysfunc_gt(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * val0 = engine->eval(args->nth(0)) ;
      Sexp * val1 = engine->eval(args->nth(1)) ;

      if(val0->fltVal() > val1->fltVal()) return Sexp::TrueValue ;
      else                                return Sexp::Nil ;
   }

   //----------------------------------------
   // sysfunc: ge

   Sexp *
   sysfunc_ge(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,2) ;
      
      //Sexp * op = form->first() ;
      Sexp * args = form->rest() ;

      Sexp * val0 = engine->eval(args->nth(0)) ;
      Sexp * val1 = engine->eval(args->nth(1)) ;

      if(val0->fltVal() >= val1->fltVal()) return Sexp::TrueValue ;
      else                                 return Sexp::Nil ;
   }

   //----------------------------------------
   // sysfunc: print

   Sexp *
   sysfunc_print(Sexp * form, EvalEngine * engine) {

      EVAL_CHK_ARITY(form,1) ;

      Sexp * args = form->rest() ;

      Sexp * val = engine->eval(args->nth(0)) ;

      cout << val << endl;

      return val ;
   } ;

   //----------------------------------------
   // sysfunc: list

   Sexp *
   sysfunc_list(Sexp * form, EvalEngine * engine) {

      Sexp * args = form->rest() ;

      Sexp::Heap & sheap = engine->context().localsheap() ;


      Sexp * r = sheap.list(Sexp::Nil) ;
      Sexp * tail = r ;

      for(Sexp * a = args ; a->isCons() ; a = a->cdr()) {
	 Sexp * val = engine->eval(a->car()) ;
	 Sexp * newcell = sheap.list(val) ;
	 tail->rplacd(newcell) ;
	 tail = newcell ;
      }

      return r->cdr() ;
   } ;

   //------------------------------------------------------------
   //   : EvalEngine
   // init func: table

   void
   EvalEngine::
   initFuncTable(Bool forcep) {
      static Bool initp = False ;

      if(!initp || forcep) {
	 initp = True ;

	 registerFunc("and",	&sysfunc_and) ;
	 registerFunc("or",	&sysfunc_or) ;
	 registerFunc("not",	&sysfunc_not) ;

	 registerFunc("progn",     &sysfunc_progn) ;
	 registerFunc("quote",	&sysfunc_quote) ;
	 registerFunc("setq",	&sysfunc_setq_local) ;
	 registerFunc("ssetq",	&sysfunc_setq_static) ;
	 registerFunc("gsetq",	&sysfunc_setq_global) ;

	 registerFunc("+",	&sysfunc_add) ;
	 registerFunc("-",	&sysfunc_sub) ;
	 registerFunc("*",	&sysfunc_mul) ;
	 registerFunc("/",	&sysfunc_div) ;
	 //registerFunc("%",      &sysfunc_mod) ;

	 registerFunc("?=",	&sysfunc_varunify) ;

	 registerFunc("==",	&sysfunc_equal) ;
	 registerFunc("<",	&sysfunc_lt) ;
	 registerFunc("<=",	&sysfunc_le) ;
	 registerFunc("=<",	&sysfunc_le) ;
	 registerFunc(">",	&sysfunc_gt) ;
	 registerFunc(">=",	&sysfunc_ge) ;
	 registerFunc("=>",	&sysfunc_ge) ;

	 registerFunc("print",	&sysfunc_print) ;

	 registerFunc("list",	&sysfunc_list) ;
      }
   } ;


} ;



