// -*- Mode:C++ -*-
//Header:
//File: FrameSexp.cc
//Author: NODA, Itsuki
//Date: 2001/09/05
//

//ModifyHistory:
// 2001/09/05: Start to create this file
//EndModifyHistory:

/*
 * Copyright (C) 2001 NODA, Itsuki, CARC, AIST, JAPAN
 */

#ifndef _itk_FrameSexp_h_
#define _itk_FrameSexp_h_
//////////////////////////////////////////////////////////////////////

#include "itk/SimpleSexp2.h"

namespace Itk {

   //======================================================================
   // class FrameSexp

#define ITK_FRAME_SLOTNAME_PREFIX	":"

   /*--------------------*/
   /** 
    * A Flexible Frame Expression
    *
    */

   class FrameSexp : public SimpleSexp2 {
	 
	 //------------------------------------------------------------
	 // FrameSexp:: local types
      public:
	 class Heap ;
	 class Scanner ;

	 //------------------------------------------------------------
	 // FrameSexp:: slotname prefix string (static const)
      public:
	 static const char slotnamePrefix[] = ITK_FRAME_SLOTNAME_PREFIX ;

	 //------------------------------------------------------------
	 // FrameSexp:: Special Keyword Symbols (static const) 
	 // defines 
	 //	KeySym_xxxx
	 // usage :
	 //   to define KeySum_Foo as symbol ":foo",
	 //     place ITK_FRAME_DEF_KEYSYM_HEAD(Foo,":foo") in class def.
	 //     place ITK_FRAME_DEF_KEYSYM_BODY(Foo) outside of class def.

#define ITK_FRAME_DEF_KEYSYM_HEAD(name,str) \
	public: static const char _KeySymName_##name [] = str ; \
	public: static SimpleSexp2  _KeySym_##name ; \
	public: static SimpleSexp2 * KeySym_##name ;

#define ITK_FRAME_DEF_KEYSYM_BODY(name) \
	SimpleSexp2 FrameSexp::_KeySym_##name = \
	   SimpleSexp2(SimpleSexp2::T_Symbol,\
		     FrameSexp::_KeySymName_##name) ;\
	SimpleSexp2 * FrameSexp::KeySym_##name = &FrameSexp::_KeySym_##name ;

	 ITK_FRAME_DEF_KEYSYM_HEAD(Set, ":set") ;
	 ITK_FRAME_DEF_KEYSYM_HEAD(List,":list") ;
	 ITK_FRAME_DEF_KEYSYM_HEAD(Not, ":not") ;
	 ITK_FRAME_DEF_KEYSYM_HEAD(Wh,  ":wh") ;

	 //------------------------------------------------------------
	 // FrameSexp:: special type frame check
	 /*--------------------*/
	 /**@name Check Special Forms */
	 //@{

      public:
	 /// check the data is a frame (not check in detail)
	 Bool isFrame() const {
	    return isCons() ;
	 } ;
	 
      public:
	 /// check the data is a {\em set} frame or not 
	 Bool isSetFrame() const {
	    return isCons() && car()->equal(KeySym_Set) ;
	 } ;

      public:
	 /// check the data is a {\em list} frame or not
	 Bool isListFrame() const {
	    return isCons() && car()->equal(KeySym_List) ;
	 } ;

      public:
	 /// check the data is a {\em not} frame or not 
	 Bool isNotFrame() const {
	    return isCons() && car()->equal(KeySym_Not) ;
	 } ;
	 
      public:
	 /// check the data is a {\em wh} frame or not 
	 Bool isWhFrame() const {
	    return isCons() && car()->equal(KeySym_Wh) ;
	 } ;

      public:
	 /// check the data is a normal frame
	 Bool isNormalFrame() const {
	    return isFrame() && !isSetFrame() && !isListFrame() 
	       && !isNotFrame() && !isWhFrame() ;
	 } ;

	 //@}

	 //------------------------------------------------------------
	 // FrameSexp:: syntax check
	 /** 
	  * Syntax Check 
	  *
	  * If depth > 0, then check upper than the depth.
	  * If depth < 0, it checks whole frames.
	  * If depth < 0 and too deep frame (more than 2^32), then it stops.
	  */
      public:
	 Bool chkSyntax(Int depth = -1) const ;

	 //------------------------------------------------------------
	 // FrameSexp:: scan wrapper
	 /**@name Scan 
	  * 
	  * scan() facilities scans a frame from a Scanner, istrstream,
	  * Buffer, or c-string.
	  * scan() with istrstream, Buffer, or c-string use a shard scanner
	  * of SimpleSexp2 class. 
	  * This means that users need to take care of using them
	  * under multi-threading. 
	  */
	 //@{

      public:
	 /// scan a frame from a scanner 
	 static FrameSexp * scan(Scanner * scanner, Bool resetp = True,
				 Heap * heap = ITK_NULLPTR) {
	    return 
	       static_cast<FrameSexp*>(SimpleSexp2::scan(scanner,resetp,heap));
	 } ;
      public:
	 /// scan a frame from a scanner
	 static FrameSexp * scan(Scanner & scanner, Bool resetp = True,
				 Heap * heap = ITK_NULLPTR) {
	    return 
	       static_cast<FrameSexp*>(SimpleSexp2::scan(scanner,resetp,heap));
	 } ;
      public:
	 /// scan a frame from an istream
	 static FrameSexp * scan(istream * istr, Bool resetp = True,
				 Heap * heap = ITK_NULLPTR) {
	    return 
	       static_cast<FrameSexp *>(SimpleSexp2::scan(istr,resetp,heap)) ;
	 } ;
      public:
	 /// scan a frame from an istream
	 static FrameSexp * scan(istream & istr, Bool resetp = True,
				 Heap * heap = ITK_NULLPTR) {
	    return 
	       static_cast<FrameSexp *>(SimpleSexp2::scan(istr,resetp,heap)) ;
	 } ;
      public:
	 /// scan a frame from a buffer (string)
	 static FrameSexp * scan(Buffer & buf, Bool resetp = True,
				 Heap * heap = ITK_NULLPTR) {
	    return 
	       static_cast<FrameSexp *>(SimpleSexp2::scan(buf,resetp,heap)) ;
	 } ;
      public:
	 /// scan a frame from a buffer (string)
	 static FrameSexp * scan(Buffer * buf, Bool resetp = True,
				 Heap * heap = ITK_NULLPTR) {
	    return 
	       static_cast<FrameSexp *>(SimpleSexp2::scan(buf,resetp,heap)) ;
	 } ;
      public:
	 /// scan a frame from a c-string
	 static FrameSexp * scan(const char * str, Bool resetp = True,
				 Heap * heap = ITK_NULLPTR) {
	    return 
	       static_cast<FrameSexp *>(SimpleSexp2::scan(str,resetp,heap)) ;
	 } ;
	 
	 //@}

	 //------------------------------------------------------------
	 // FrameSexp:: access to slot value
	 /** @name Access to Slot and Value */
	 //@{
	     
	 //----------------------------------------
	 // FrameSexp:: access to head
      public:
	 /// get a frame head
	 FrameSexp * head() {
	    return static_cast<FrameSexp*>(first()) ;
	 } ;

      public:
	 /// set a frame head
	 FrameSexp * setHead(SimpleSexp2 * head) {
	    setCar(head) ;
	    return static_cast<FrameSexp*>(head) ;
	 } ;

	 //----------------------------------------
	 // FrameSexp:: get slot value
      public:
	 /// get a value of a slot
	 FrameSexp * slotvalue(const SubString & slotname) {
	    if(!isCons()) return static_cast<FrameSexp*>(Nil) ;
	    return static_cast<FrameSexp*>(cdr()->passoc(slotname)) ;
	 }

      public:
	 /// get a value of a slot
	 FrameSexp * slotvalue(const SimpleSexp2 * slotname) {
	    if(!isCons()) return static_cast<FrameSexp*>(Nil) ;
	    return static_cast<FrameSexp*>(cdr()->passoc(slotname)) ;
	 }

	 //----------------------------------------
	 // FrameSexp:: set slot value
      public:
	 /// set a value to a slot
	 FrameSexp * setSlot(const SubString & slotname,
			     SimpleSexp2 * value,
			     Heap * heap) {
	    SimpleSexp2 * r = cdr()->passocPos(slotname) ;
	    if(r->isCons()) {
	       r->cdr()->setCar(value) ;
	    } else {
	       heap->addNewSlotValue(this,
				     static_cast<FrameSexp*>(
					heap->newSymbol(slotname)),
				     value) ;
	    }
	    return this ;
	 }

     public:
	 /// set a value to a slot
	 FrameSexp * setSlot(SimpleSexp2 * slotname,
			     SimpleSexp2 * value,
			     Heap * heap) {
	    SimpleSexp2 * r = cdr()->passocPos(slotname) ;
	    if(r->isCons()) {
	       r->cdr()->setCar(value) ;
	    } else {
	       heap->addNewSlotValue(this,slotname,value) ;
	    }
	    return this ;
	 }

	 //----------------------------------------
	 // FrameSexp:: access to slot-value pairs
      public:
	 /// return number of slot-value pairs
	 Int slotN() const {
	    if(isNormalFrame()) {
	       return (length() - 1) / 2 ;
	    } else {
	       return -1 ;
	    }
	 }

      public:
	 /// access to nth slot-value pair
	 FrameSexp * nthSlot(Int n) {
	    if(isNormalFrame()) {
	       return static_cast<FrameSexp*>(nthcdr(n * 2 + 1)) ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    } ;
	 }

      public:
	 /// access to nth slot name
	 FrameSexp * nthSlotName(Int n) { 
	    return static_cast<FrameSexp*>(nthSlot(n)->first()) ; 
	 } ;

      public:
	 /// access to nth slot value
	 FrameSexp * nthSlotValue(Int n) { 
	    return static_cast<FrameSexp*>(nthSlot(n)->second()) ; 
	 } ;

      public:
	 Int confirmSlotN(Int n, Heap * heap) {
	    Int c = 0 ;
	    while(slotN() <= n) {
	       heap->concat(this,heap->list(Nil,Nil)) ;
	       c++ ;
	    }
	    return c ;
	 } ;

      public:
	 /// set nth slot
	 FrameSexp * setNthSlotName(Int n, SimpleSexp2 * slot, Heap * heap) {
	    if(isNormalFrame()) {
	       confirmSlotN(n,heap) ;
	       FrameSexp * pair = nthSlot(n) ;
	       pair->setCar(slot) ;
	       return pair ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    }
	 } ;

      public:
	 /// set nth value
	 FrameSexp * setNthSlotValue(Int n, SimpleSexp2 * value, Heap * heap) {
	    if(isNormalFrame()) {
	       confirmSlotN(n, heap) ;
	       FrameSexp * pair = nthSlot(n) ;
	       pair->cdr()->setCar(value) ;
	       return pair ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    }
	 } ;

      public:
	 /// set nth value
	 FrameSexp * setNthSlot(Int n, SimpleSexp2 * slot, 
				SimpleSexp2 * value, Heap * heap) {
	    if(isNormalFrame()) {
	       confirmSlotN(n,heap) ;
	       FrameSexp * pair = nthSlot(n) ;
	       pair->setCar(slot) ;
	       pair->cdr()->setCar(value) ;
	       return pair ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    }
	 } ;

	 //----------------------------------------
	 // FrameSexp:: access to nth item of list and set
      public:
	 /// return number of args of list and set frame
	 Int argN() const {
	    if(isListFrame() || isSetFrame()) {
	       return length() - 1 ;
	    } else {
	       return - 1 ;
	    }
	 } ;
	       
      public:
	 /// access to nth arg of list and set frame
	 FrameSexp * nthArg(Int n) {
	    if(isListFrame() || isSetFrame()) {
	       return static_cast<FrameSexp*>(nth(n+1)) ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    }
	 } ;

      public:
	 /// set nth arg of list and set frame
	 FrameSexp * setNthArg(Int n, SimpleSexp2 *arg, Heap * heap) {
	    if(isListFrame() || isSetFrame()) {
	       while(argN() <= n) {
		  heap->addLast(this,Nil) ;
	       }
	       nthcdr(n+1)->setCar(arg) ;
	       return static_cast<FrameSexp*>(arg) ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    }
	 } ;

	 //----------------------------------------
	 // FrameSexp:: access to content of not and wh frame
      public:
	 /// access to content of not and wh frame
	 FrameSexp * content() {
	    if(isNotFrame() || isWhFrame()) {
	       return static_cast<FrameSexp*>(nth(1)) ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    }
	 } ;

      public:
	 /// set content of not and wh frame
	 FrameSexp * setContent(SimpleSexp2 * content, Heap * heap) {
	    if(isNotFrame() || isWhFrame()) {
	       if(length() < 2) {
		  heap->addLast(this,Nil) ;
	       }
	       cdr()->setCar(content) ;
	       return static_cast<FrameSexp*>(content) ;
	    } else {
	       return static_cast<FrameSexp*>(Nil) ;
	    }
	 } ;

	 //@}

	 //------------------------------------------------------------
	 // FrameSexp:: duplicate
	 /** @name Duplicate a Frame */
	 //@{

	 /// diplicate a frame
      public:
	 FrameSexp * dup(Heap & heap, Bool strcopyp = True) const {
	    return static_cast<FrameSexp*>(SimpleSexp2::dup(heap,strcopyp)) ;
	 } ;

	 /// diplicate a frame
      public:
	 FrameSexp * dup(Heap * heap, Bool strcopyp = True) const {
	    return dup(*heap,strcopyp) ;
	 } ;

	 //@}

	 //------------------------------------------------------------
	 // FrameSexp:: formated output
	 /** @name Formated Output */
	 //@{

      public:
	 /// output in LaTeX format
	 void outputLaTeX(ostream & ostr, Int indent = 0) ;

      public:
	 /// output in LaTeX format
	 void outputLaTeX(ostream * ostr, Int indent = 0) { 
	    return outputLaTeX(*ostr,indent) ; 
	 } ;

      public:
	 /// output in HTML format
	 void outputHTML(ostream & ostr, Int indent = 0) ;

      public:
	 /// output in HTML format
	 void outputHTMl(ostream * ostr, Int indent = 0)  { 
	    return outputHTML(*ostr,indent) ; 
	 } ;

      public:
	 void outputWithMark(ostream & ostr, Int indent,
			     char * beginframe,
			     char * endframe,
			     char * headprefix,
			     char * headsuffix,
			     char * slotprefix,
			     char * slotvaluesep,
			     char * valuesuffix) ;

      public:
	 void outputWithMark(ostream * ostr, Int indent,
			     char * beginframe,
			     char * endframe,
			     char * headprefix,
			     char * headsuffix,
			     char * slotprefix,
			     char * slotvaluesep,
			     char * valuesuffix) {
	    return outputWithMark(*ostr,indent,
				  beginframe, endframe,
				  headprefix, headsuffix,
				  slotprefix, slotvaluesep, valuesuffix) ;
	 } ;

	 //@}

	 //============================================================
	 // FrameSexp:: class Heap
	 /** 
	  * A Heap of FrameSexp
	  */
      public:
	 class Heap : public SimpleSexp2::Heap {
	       
	       //--------------------------------------------------
	       // FrameSexp::Heap:: newFrame
	       /** @name New Frame */
	       //@{

	    public:
	       /// get a new frame with a head
	       FrameSexp * newFrame(SimpleSexp2 * head) {
		  return static_cast<FrameSexp*>(list(head)) ;
	       } ;

	    public:
	       /// get a new frame with a head
	       FrameSexp * newFrame(const SubString & s) {
		  return newFrame(const_cast<SubString&>(s),True) ;
	       } ;
	    public:
	       /// get a new frame with a head
	       FrameSexp * newFrame(SubString & s, Bool copyp = True) {
		  return newFrame(static_cast<FrameSexp*>(newSymbol(s,copyp)));
	       } ;
	    public:
	       /// get a new frame with a head
	       FrameSexp * newFrame(const char * s, Bool copyp = True) {
		  return newFrame(static_cast<FrameSexp*>(newSymbol(s,copyp)));
	       } ;

	       //@}

	       //--------------------------------------------------
	       // FrameSexp::Heap:: addNewSlotValue
	    public:
	       /// add new slot to a frame
	       FrameSexp * addNewSlotValue(SimpleSexp2 * frame,
					   SimpleSexp2 * slot,
					   SimpleSexp2 * value,
					   Bool lastp = True) {
		  if(lastp) {
		     concat(frame,list(slot,value)) ;
		  } else {
		     SimpleSexp2 * cell = list(slot,value) ;
		     concat(cell,frame->cdr()) ;
		     frame->setCdr(cell) ;
		  }
		  return static_cast<FrameSexp*>(frame) ;
	       } ;

	       //--------------------------------------------------
	       // FrameSexp::Heap:: dup
	       /** 
		* Duplicate a Frame 
		*/
	    public:
	       FrameSexp * dup(const FrameSexp * original,
			       Bool strcopyp = True) {
		  return original->dup(this,strcopyp) ;
	       } ;

	       //--------------------------------------------------
	       // FrameSexp::Heap:: scan
	       /** @name Scan 
		*
		* scan() facilities scans a frame from istrstream,
		* Buffer, or c-string.
		* These scan() uses heap's own scanner.
		* This means that users need to take care of using
		* the same heap under multi-threading.
		*/
	       //@{
	       
	       /// scan a frame from a stream
	    public:
	       FrameSexp * scan(istream * istr, Bool clearheapp = False) {
		  return 
		     static_cast<FrameSexp*>(
			SimpleSexp2::Heap::scan(istr,clearheapp)) ;
	       } ;

	       /// scan a frame from a stream
	    public:
	       FrameSexp * scan(istream & istr, Bool clearheapp = False) {
		  return static_cast<FrameSexp*>(
		     SimpleSexp2::Heap::scan(istr,clearheapp)) ;
	       } ;

	       /// scan a frame from a buffer (string)
	    public:
	       FrameSexp * scan(Buffer * b, Bool clearheapp = False) {
		  return static_cast<FrameSexp*>(
		     SimpleSexp2::Heap::scan(b,clearheapp)) ;
	       } ;

	       /// scan a frame from a buffer (string)
	    public:
	       FrameSexp * scan(Buffer & b, Bool clearheapp = False) {
		  return static_cast<FrameSexp*>(
		     SimpleSexp2::Heap::scan(b,clearheapp)) ;
	       } ;

	       /// scan a frame from a c-string
	    public:
	       FrameSexp * scan(const char * str, Bool clearheapp = False) {
		  return static_cast<FrameSexp*>(
		     SimpleSexp2::Heap::scan(str,clearheapp)) ;
	       } ;

	       //@}

	 } ;

	 //============================================================
	 // FrameSexp:: class Scanner
	 /**
	  *  A Scanner of FrameSexp
	  */
      public:
	 class Scanner : public SimpleSexp2::Scanner {
	       
	       //------------------------------
	       // SimpleSexp2::Scanner:: constructor
	       /** @name constructors */
	       //@{

	    public:
	       /// construct with nothing
	       Scanner() {
		  init((istream*)ITK_NULLPTR,False,ITK_NULLPTR) ;
	       } ;

	    public:
	       /// construct from heap
	       Scanner(Heap * h) {
		  init((istream*)ITK_NULLPTR,False,h) ;
	       } ;

	    public:
	       /// construct from istream
	       Scanner(istream * istr,
		       Bool deletablep = True,
		       Heap * h = ITK_NULLPTR) {
		  init(istr,deletablep,h) ;
	       } ;

	    public:
	       /// construct from istream
	       Scanner(istream & istr, 
		       Bool deletablep = True,
		       Heap  * h = ITK_NULLPTR) {
		  init(istr,deletablep,h) ;
	       } ;

	    public:
	       /// construct from buffer
	       Scanner(Buffer * b, 
		       Bool deletablep = True,
		       Heap * h = ITK_NULLPTR) {
		  init(b,deletablep,h) ;
	       } ;

	    public:
	       /// construct from buffer
	       Scanner(Buffer & b, 
		       Bool deletablep = True,
		       Heap * h = ITK_NULLPTR) {
		  init(b,deletablep,h) ;
	       } ;

	       //@}

	       //--------------------------------------------------
	       // FrameSexp::Scanner scan wrapper
	    public:
	       /// scan a frame
	       FrameSexp * scan(Bool resetp = True) {
		  return 
		     static_cast<FrameSexp *>(
			SimpleSexp2::Scanner::scan(resetp)) ;
	       } ;
	       
	 } ;

	 //============================================================
	 //============================================================
	 //============================================================
	 // Usage:
	 //============================================================
	 //============================================================
	 //============================================================

	 /**@name Usage */ 
	 //@{

	 //------------------------------
	 /** to scan a frame from a string.
	  *
	  * \begin{verbatim}
    FrameSexp::Heap heap ;
    FrameSexp* frame = 
       heap.scan("(foo :bar (foo1 :bar () :baz 7) :boo 3
                       :a (:list a b c d e)),True) ;
    if(!frame->chkSymtax()) { error(...) ; }
 	  \end{verbatim}
	  * New cells and strings are allocated in {\tt heap}.
          * If the second argument of scan() is True, then the heap is
	  * cleared before scanning.
	  * If the second argument if False, the new cells and strings
	  * are allocated the rest area of the heap.
	  * All scan() does not check the syntax.
	  */

	 //------------------------------
	 /** to scan a frame from an istream
	  *
	  * \begin{verbatim}
    istream istr ;
    ...
    FrameSexp::Heap heap ;
    FrameSexp* frame = 
       heap.scan(istr,True) ;
    if(!frame->chkSymtax()) { error(...) ; }
 	  \end{verbatim}
	  */

	 //------------------------------
	 /** to scan a frame from a stream repeatedly
	  *
	  * \begin{verbatim}
    istream istr ;
    ...
    FrameSexp::Heap heap ;
    FrameSexp::Scanner scanner(istr,False,&heap) ;
    FrameSexp * data ;
    for(data = scanner.scan() ; !data->isEof() ; data = scanner.scan()) {
       ... 
    }
 	  \end{verbatim}
	  * this does not check the syntax.
	  */

	 //------------------------------
	 /** to scan a frame from a string.
	  *
	  * \begin{verbatim}
    FrameSexp* frame = 
       FrameSexp::scan("(foo :bar (foo1 :bar () :baz 7) :boo 3
                        :a (:list a b c d e))") ;
    if(!frame->chkSymtax()) { error(...) ; }
 	  \end{verbatim}
	  * New cells and strings are allocated in SimpleSexp2::sharedheap
	  * (SimpleSexp2 is a parent class of FrameSexp.)
	  */

	 //------------------------------
	 /** to get a value of a slot
	  *
	  * \begin{verbatim}
    FrameSexp * value = frame->slotValue(":bar") ;

    FrameSexp::Heap heap ;
    FrameSexp * symbol = heap.newSymbol(":bar") ;
    FrameSexp * value2 = frame->slotValue(symbol) ; 
	  \end{verbatim}
	  */

	 //------------------------------
	 /** to create new atoms
	  *
	  * \begin{verbatim}
    FrameSexp::Heap heap ;
    FrameSexp * intval = heap.newInt(3) ;
    FrameSexp * fltval = heap.newFlt(3) ;
    FrameSexp * symval = heap.newSymbol("baz") ;
	  \end{verbatim}
	  */

	 //------------------------------
	 /** to construct new frame
	  *
	  * \begin{verbatim}
    FrameSexp::Heap heap ;
    FrameSexp frame = heap.newFrame("foo") ;
	  \end{verbatim}
	  */

	 //------------------------------
	 /** to set a slot to the frame
	  *
	  * \begin{verbatim}
    FrameSexp * slotname = heap.newSymbol(":bar") ;
    FrameSexp * slotvalue = heap.newInt(3) ;
    frame->setSlot(slotname, slotvalue,&heap) ;
	  \end{verbatim}
	  * setSlotValue() replaces the value of the slot.
	  * If the slot is not exists, the method add a new slot to
	  * the frame.
	  */

	 //------------------------------
	 /** duplicate a new frame using a heap
	  *
	  * \begin{verbatim}
    FrameSexp::heap1 ;
    FrameSexp * original = heap1.scan("(foo :bar baz)") ;

    FrameSexp::heap2 ;
    FrameSexp * copied = heap2.dup(original) ;
	  \end{verbatim}
	  * While all data of the original frame is allocated in heap1,
	  * all data of copied frame is allocated in heap2.
	  * This is useful when a program use temporal and permanent data.
	  * In this case, users can prepare heaps for temporal and permanent
	  * use, and copy data useing dup() facilities between these heaps.
	  */

	 //@}



   } ;

} ;

//////////////////////////////////////////////////////////////////////
#endif
