// System.cxx
//
/////////////////////////////////////////////////////////////////////////////

#include "common.hxx"
#include "System.hxx"

#ifndef NDEBUG
#  include "System.inl"
#endif

#include "objects.hxx"

namespace Rescue
{
    /////////////////////////////////////////////////////////////////////////
    // System
    
    System::~System()
    {
        delete[] m_pFreeAgents;
        delete[] m_pConnectedAgents;

        if(m_logFile != 0)
            fclose(m_logFile);

        delete m_publicSocket;
        delete m_agentSocket;
        delete m_simulatorSocket;
        delete m_gisSocket;
    }

    void System::main(int argc, char const* const* argv)
    {
        // TODO: server.conf ɤ߹
        
        //m_outputBuffer.initialize(config().buffer_size());
        //m_inputBuffer.initialize(config().buffer_size());
    
        const char* host = "localhost";
        int number = -1;
        
        int i;
        int count = 0;
        for(i=1; i<argc; i++) {
            if(argv[i][0] == '-') {
                if(strcmp(argv[i], "-file") == 0) {
                    if(++i < argc) {
                        m_config.readConfigFile(argv[i]);
                    } else {
                        fprintf(stderr, "error: '-file'\n");
                    }
                } else if(strcmp(argv[i], "-n") == 0) {
                    if(++i < argc) {
                        number = atoi(argv[i]);
                    } else {
                        fprintf(stderr, "error: '-file'\n");
                    }
                } else if(strcmp(argv[i], "-nogis") == 0) {
                    host = 0;
                } else {
                    if(++i < argc) {
                        if(!m_config.setValue(argv[i-1]+1, argv[i]))
                            fprintf(stderr, "error: '%s'\n", argv[i]);
                    } else {
                        fprintf(stderr, "error: '%s'\n", argv[i-1]);
                    }
                }
            } else {
                switch(count++) {
                case 0:
                    host = argv[i];
                    break;
                default:
                    fprintf(stderr, "error: '%s'\n", argv[i]);
                    break;
                }
            }
        }
        

        m_outputBuffer.resetFrameSize(config().send_udp_size());
        m_outputBuffer.setWait(config().send_udp_wait());

        m_publicSocket = new LongUDPSocket(Address(INADDR_ANY, (S16)config().port(), true), config().textlogname_kernel_publicsock());
        m_agentSocket = new LongUDPSocket(config().textlogname_kernel_agentsock());
        m_simulatorSocket = new LongUDPSocket(config().textlogname_kernel_simulatorsock());

        if(!config().logname().empty()) {
            m_logFile = fopen(config().logname().c_str(), "w"FOPEN_MODE_BINARY);
            if(m_logFile == 0) {
                fprintf(stderr, "error: can not open file '%s'\n", config().logname().c_str());
            } else {
                const char header[] = "RoboCup-Rescue Prototype Log 00";
                fwrite(header, sizeof(header[0]), COUNT_OF(header), m_logFile);
                fflush(m_logFile);
            }
        }

        // GIS ³
        if(number<0)
            number = INT_MAX;
        m_gisSocket = new LongUDPSocket(config().textlogname_kernel_gissock());
        m_gis = Address(host, config().gis_port());
        // ³å
        printf("connecting...\n");
        m_outputBuffer.clear();
        m_outputBuffer.put(KG_CONNECT);
        Output::Cursor base = m_outputBuffer.put(~(S32)0);
        m_outputBuffer.put(0);
        m_outputBuffer.setSize(base);
        m_outputBuffer.put(HEADER_NULL);
        gisSocket().send(m_outputBuffer, m_gis, m_time);

        m_gisConnected = false;
        while(!m_gisConnected)
            receiveMessages(gisSocket());

        Objects::const_iterator it = m_objectPool.objects().begin();
        for(; it != m_objectPool.objects().end() && number>0; it++) {
            Object* o = *it;
            Agent* a = dynamic_cast<Agent*>(o);
            if(a) {
                addFreeAgent(a);
                number--;
            }
        }
        
        setUp();
        printf("setUp() done.\n");
        if(config().wait_before_start()) {
            char buffer[102];
            printf("push the enter key to start.\n");
            fgets(buffer, 100, stdin);
        }
        loop();
    }
    
#if OS_TYPE != OS_TYPE_WINDOWS
    static void dummy(int) {
        // do nothing
    }
#endif
    
    bool System::startOk()
    {
        if(m_addressToAckWaiter.empty() && m_idToAckWaiter.empty()) {
            unsigned int i;
            for(i=0; i<m_agentTypes.size(); i++) {
                if(!m_pFreeAgents[i].empty())
                    return false;
            }
            return true;
        }
        return false;

    }

    void System::setUp() {
        typedef std::vector<const LongUDPSocket*> Sockets;
        Sockets sockets;
        sockets.push_back(&publicSocket());
        sockets.push_back(&simulatorSocket());
        sockets.push_back(&agentSocket());

        while(!startOk()) {
            bool timeout = !LongUDPSocket::wait(sockets, 5 * config().step());
            if(!timeout) {
                // 
                receiveMessages(publicSocket());
                receiveMessages(simulatorSocket());
                receiveMessages(agentSocket());
            } else {
                // Ack ֤äƤʤΤǡ롣
                if(!m_addressToAckWaiter.empty()) {
                    fprintf(stderr, "addressAck\n");
                    AckWaiter* aw = m_addressToAckWaiter.begin()->second.get();
                    aw->send();
                } else if(!m_idToAckWaiter.empty()) {
                    fprintf(stderr, "idAck\n");
                    AckWaiter* aw = m_idToAckWaiter.begin()->second.get();
                    aw->send();
                }
            }
            // Ȥ³֤ɽ
            printf("free agents = %ld", (long)m_pFreeAgents[0].size());
            for(int i=1; i<(int)m_agentTypes.size(); i++)
                printf(", %ld", (long)m_pFreeAgents[i].size());
            printf("  no ack=%ld\n", (long)(m_addressToAckWaiter.size() + m_idToAckWaiter.size()));
        }

        if(m_logFile != 0) {
            m_outputBuffer.clear();
            m_outputBuffer.put(KS_CONNECT_OK);
            Output::Cursor base = m_outputBuffer.put(~(S32)0);
            objectPool().output(m_outputBuffer);
            m_outputBuffer.setSize(base);
            m_outputBuffer.put(HEADER_NULL);

            m_outputBuffer.log(m_logFile);
        }
    }

    void System::addFreeAgent(Agent* agent)
    {
        TypeId type = agent->asObject()->type();
        int i=-1;
        do {
            i++;
            ASSERT(i < (int)m_agentTypes.size());
        } while(type != m_agentTypes[i]);
        m_pFreeAgents[i].push_back(agent);
    }

#if OS_TYPE == OS_TYPE_UNIX
    bool kbhit() {
        fd_set fds;
        FD_ZERO(&fds);
        const int STDIN = 0;
        FD_SET(STDIN, &fds);
        timeval timeout = { 0 };
        int result = select(STDIN + 1, &fds, NULL, NULL, &timeout);
        ASSERT(result != SOCKET_ERROR);
        return (bool)result;
    }
#endif
    void System::loop() {
        const int division = 2;
        long count = 0;

        m_time = 1;
        const int timeDelta = config().step() / division;
        
#if OS_TYPE == OS_TYPE_WINDOWS
        DWORD nextTime = ::timeGetTime() + timeDelta;
        int lostContinually = 0;
        while(!_kbhit() && m_time <= config().period()) {
            // 濫碌
            DWORD now = ::timeGetTime();
            if(now < nextTime) {
                Sleep(nextTime - now);
                nextTime += timeDelta;
                lostContinually = 0;
            } else {
                if(lostContinually++ < 5) {
                    nextTime += timeDelta;
                } else {
                    nextTime = now + timeDelta;
                    fprintf(stderr, "busy. %d %ld\n", (int)lostContinually, (long)now);
                }
            }
            m_systemTime = (long)now;
            // 
            loopCore(count, division);
        }
#else
        class AutoCast {
        public:
            typedef void (*type1)(int);
            typedef void (*type2)(...);
            type1 m_fp;
            AutoCast(type1 fp) {
                m_fp = fp;
            }
            operator type1 () {
                return m_fp;
            }
            operator type2 () {
                return (type2)m_fp;
            }
        } autoCast(dummy);
        struct sigaction sa = { 0 };
        sa.sa_handler = autoCast;
        //sa.sa_flags &= (~SA_RESETHAND);
        sigaction(SIGALRM, &sa, NULL);
        struct itimerval itv = { { 0 } };
        itv.it_interval.tv_sec = timeDelta / 1000;
        itv.it_interval.tv_usec = (timeDelta % 1000) * 1000;
        itv.it_value.tv_sec = timeDelta / 1000;
        itv.it_value.tv_usec = (timeDelta % 1000) * 1000;
        setitimer(ITIMER_REAL, &itv, NULL);
        while(!(config().interactive_stop() && kbhit()) && m_time <= config().period()) {
            // 濫碌
            sigpause(SIGUSR1);
            m_systemTime += timeDelta;
            // 
            loopCore(count, division);
        }
#endif
    }
    void System::loopCore(long& count, int division)
    {
        //printf("system time %ld\n", (long)m_systemTime);

        ASSERT(division == 2);
        switch(count++ % division) {
        default:
            ASSERT(false);
        case 0:
        {
            receiveMessages(publicSocket());
            resetAgents();
            receiveMessages(agentSocket());
            sendToSimulators();
            if(m_logFile != 0)
                fflush(m_logFile);
            resetSimulators();
        }
            break;
        case 1:
            receiveMessages(simulatorSocket());
            step();
            m_time++;
            sendToAgents();
            sendUpdate(m_time - 1);
            printf("time=%ld\n", (long)m_time);
            break;
        }
    }
    void System::resetAgents()
    {
    }
    void System::resetSimulators()
    {
    }
    
    void System::receiveMessages(LongUDPSocket& from)
    {
        while(!from.empty()) {
            if(from.receive(m_inputBuffer, m_time)) {
                //printf("LongUDPPacket size = %ld\n", (long)m_inputBuffer.size());
                try {
                    while(processMessage(from));
                } catch(Input::Overrun) {
                    fprintf(stderr, "illegal message\n");
                }
            }
        }
    }
    bool System::processMessage(const LongUDPSocket& from)
    {
        // TODO: from μˤäƤϼݤ

        Header header = m_inputBuffer.get();
        if(header == HEADER_NULL)
            return false;

        S32 size = m_inputBuffer.get();
        Input::Cursor start = m_inputBuffer.cursor();

        //fprintf(stderr, "Received header=0x%lX.\n", (long)header);
        if(header >= HEADER_COMMAND_MIN) {
            Id sender = m_inputBuffer.get();
            Agent* agent = dynamic_cast<Agent*>(m_objectPool.get(sender));
            if(agent == 0) {
                fprintf(stderr, "Illegal Sender ID=%ld\n", (long)sender);
            } else {
                if(m_time > config().steps_agents_freezed())
                    processCommand(agent, header, size - sizeof(sender), m_inputBuffer, from);
            }
        } else {
            switch(header) {
            default:
                fprintf(stderr, "Unknown header.\n");
                break;
            case GK_CONNECT_OK:
                printf("GK_CONNECT_OK\n");
                gisConnectOk(from);
                break;
            case AK_CONNECT:
                printf("AK_CONNECT\n");
                agentConnect(from);
                break;
            case AK_ACKNOWLEDGE:
                printf("AK_ACKNOWLEDGE\n");
                agentAcknowledge(from);
                break;
            case SK_CONNECT:
                printf("SK_CONNECT\n");
                simulatorConnect(from);
                break;
            case VK_CONNECT:
                printf("VK_CONNECT\n");
                simulatorConnect(from);
                break;
            case SK_ACKNOWLEDGE:
                printf("SK_ACKNOWLEDGE\n");
                simulatorAcknowledge(from);
            case SK_UPDATE:
                m_objectPool.restructure(m_time, m_inputBuffer);
                break;
            }
        }

        m_inputBuffer.setCursor(start);
        if(size == ~(S32)0)
            return false;
        m_inputBuffer.skip(size);
        return true;
    }
    void System::processCommand(Agent* agent, Header header, S32 size, Input& input, const LongUDPSocket& /*from*/)
    {
        agent->inputCommand(header, size, input);
        m_commandToAngents[header].push_back(agent);
    }
    
    void System::gisConnectOk(const LongUDPSocket& from)
    {
        printf("Ok.\n");
        
        // ack ֤
        m_outputBuffer.clear();
        m_outputBuffer.put(KG_ACKNOWLEDGE);
        Output::Cursor base = m_outputBuffer.put(~(S32)0);
        m_outputBuffer.setSize(base);
        m_outputBuffer.put(HEADER_NULL);
        gisSocket().send(m_outputBuffer, m_gis, m_time);
        
        
        printf("Initializing...\n");
        m_gisConnected = true;
        m_gis = from.addressRecievedFrom();
        m_objectPool.restructure(0, m_inputBuffer);
        m_objectPool.setUpMesh(config().mesh_size(), config().mesh_size());
        printf("Ok.\n");
    }
    void System::agentConnectError(const LongUDPSocket& from, S32 temporaryId, const char* message)
    {
        m_outputBuffer.clear();
        m_outputBuffer.put(KA_CONNECT_ERROR);
        Output::Cursor base = m_outputBuffer.put(~(S32)0);
        m_outputBuffer.put(temporaryId);
        //m_outputBuffer.put(~(S32)0);
        m_outputBuffer.putString(message);
        m_outputBuffer.setSize(base);
        m_outputBuffer.put(HEADER_NULL);
        agentSocket().send(m_outputBuffer, from.addressRecievedFrom(), m_time);
        return;
    }
    
    class AgentAckWaiter : public System::AckWaiter {
    public:
        System* system;
        Address address;
        Id temporaryId;
        Agent* agent;
        bool sendMap;
        virtual void send() {
            Output outputBuffer;
            outputBuffer.setWait(system->config().send_udp_wait());
            outputBuffer.resetFrameSize(system->config().send_udp_size());
            //outputBuffer.initialize(system->config().buffer_size());
            
            outputBuffer.clear();
            outputBuffer.put(KA_CONNECT_OK);
            Output::Cursor base = outputBuffer.put(~(S32)0);
            outputBuffer.put(temporaryId);
            outputBuffer.put(agent->asObject()->id());
            {
                // self
                // TODO: ʬȽǤΤĤ롣
                agent->asObject()->outputObject(outputBuffer);
            }
            if(sendMap) {
                // map
                // TODO: ʤΤꤨ롣
                // ȤꤢڹäƤ롣
                const Objects& objects = system->objectPool().objects();
                Objects::const_iterator it = objects.begin();
                for(; it != objects.end(); it++) {
                    const Object& object = **it;
                    system->outputObjectForAgentsAtStart(&object, outputBuffer);
                }
                outputBuffer.put(TYPE_NULL);
            }
            outputBuffer.setSize(base);
            outputBuffer.put(HEADER_NULL);
            system->agentSocket().send(outputBuffer, address);
        }
    };
    void System::agentConnect(const LongUDPSocket& from)
    {
        S32 temporaryId = m_inputBuffer.get();
        S32 version = m_inputBuffer.get();
        if(version < 0 || 2 < version) {
            agentConnectError(from, temporaryId, "unknown verison");
            return;
        }
        S32 agentType = m_inputBuffer.get();
        unsigned int i=0;
        Agent* agent = NULL;
        for(; i<m_agentTypes.size(); i++, agentType>>=1) {
            if((agentType & 1) && !m_pFreeAgents[i].empty()) {
                agent = m_pFreeAgents[i].front();
                m_pFreeAgents[i].erase(m_pFreeAgents[i].begin());
                break;
            }
        }
        if(!agent) {
            agentConnectError(from, temporaryId, "no more agent");
            return;
        }

        agent->setClientAddress(from.addressRecievedFrom());
        agent->setNeedsSensoryInformation(version != 2);
        m_pConnectedAgents[i].push_back(agent);

        AgentAckWaiter* aaw = new AgentAckWaiter();
        aaw->system = this;
        aaw->address = from.addressRecievedFrom();
        aaw->temporaryId = temporaryId;
        aaw->agent = agent;
        aaw->sendMap = (version == 0);
        aaw->send();
        m_idToAckWaiter[agent->asObject()->id()] = AutoPtr<AckWaiter>(aaw);
    }
    void System::agentAcknowledge(const LongUDPSocket& /*from*/)
    {
        Id id = m_inputBuffer.get();
        IdToAckWaiter::iterator it = m_idToAckWaiter.find(id);
        if(it == m_idToAckWaiter.end()) {
            fprintf(stderr, "illegal id\n");
            return;
        }
        m_idToAckWaiter.erase(it);
    }
    class SimulatorAckWaiter : public System::AckWaiter {
    public:
        System* system;
        Address address;
        virtual void send() {
            Output outputBuffer;
            outputBuffer.setWait(system->config().send_udp_wait());
            outputBuffer.resetFrameSize(system->config().send_udp_size());
            //outputBuffer.initialize(system->config().buffer_size());

            outputBuffer.clear();
            outputBuffer.put(KS_CONNECT_OK);
            Output::Cursor base = outputBuffer.put(~(S32)0);
            system->objectPool().output(outputBuffer);
            outputBuffer.setSize(base);
            outputBuffer.put(HEADER_NULL);
            system->simulatorSocket().send(outputBuffer, address);
            //printf("send KS_CONNECT_OK size=%ld\n", (long)outputBuffer.size());
        }
    };
    void System::simulatorConnect(const LongUDPSocket& from)
    {
        m_simulators.push_back(from.addressRecievedFrom());
        
        SimulatorAckWaiter* saw = new SimulatorAckWaiter();
        saw->system = this;
        saw->address = from.addressRecievedFrom();
        //usleep(2000 * 1000);
        saw->send();
        m_addressToAckWaiter[saw->address] = AutoPtr<AckWaiter>(saw);
    }
    void System::simulatorAcknowledge(const LongUDPSocket& from)
    {
        AddressToAckWaiter::iterator it = m_addressToAckWaiter.find(from.addressRecievedFrom());
        if(it == m_addressToAckWaiter.end()) {
            fprintf(stderr, "illegal sender\n");
            return;
        }
        m_addressToAckWaiter.erase(it);
    }
    
    
    void System::outputObjectForAgentsAtStart(const Object* o, Output& buffer)
    {
        o->outputObject(buffer);
    }
    void System::outputObjectForAgents(const Object* o, Output& buffer)
    {
        o->outputObject(buffer);
    }
    
    void System::sendToSimulators() {
        m_outputBuffer.clear();
        m_outputBuffer.put(KS_COMMANDS);
        Output::Cursor base = m_outputBuffer.put(~(S32)0);
        m_outputBuffer.put(m_time);
        HeaderToAgents::iterator it = m_commandToAngents.begin();
        for(; it != m_commandToAngents.end(); it++) {
            m_outputBuffer.put(it->first);
            Output::Cursor base2 = m_outputBuffer.put(~(S32)0);
            // TODO: ֤ϥब˾ޤ
            Agents::iterator it2 = it->second.begin();
            for(; it2 != it->second.end(); it2++) {
                Agent* agent = *it2;
                if(agent->getCommandType() == it->first) {
                    m_outputBuffer.put(dynamic_cast<Object*>(agent)->id());
                    agent->outputCommand(m_outputBuffer);
                    agent->resetCommand();
                }
            }
            m_outputBuffer.put((Id)0);
            m_outputBuffer.setSize(base2);
        }
        m_outputBuffer.put(HEADER_NULL);
        m_outputBuffer.setSize(base);
        m_outputBuffer.put(HEADER_NULL);
        m_commandToAngents.clear();

        // TODO: ֥ɥ㥹Ȥ
        Addresses::const_iterator it2 = m_simulators.begin();
        for(; it2 != m_simulators.end(); it2++) {
            const Address& simulator = *it2;
            publicSocket().send(m_outputBuffer, simulator, m_time);
        }
    }
    void System::sendUpdate(S32 time)
    {
        // TODO: ʬ
        m_outputBuffer.clear();
        m_outputBuffer.put(KS_UPDATE);
        Output::Cursor base = m_outputBuffer.put(~(S32)0);
        m_outputBuffer.put(time);
        objectPool().output(m_outputBuffer, time);
        m_outputBuffer.setSize(base);
        m_outputBuffer.put(HEADER_NULL);
        
        // TODO: ֥ɥ㥹Ȥ
        if(m_gisConnected) {
            ASSERT(KS_UPDATE == KG_UPDATE);
            gisSocket().send(m_outputBuffer, m_gis, m_time);
        }
        Addresses::const_iterator it2 = m_simulators.begin();
        for(; it2 != m_simulators.end(); it2++) {
            const Address& simulator = *it2;
            simulatorSocket().send(m_outputBuffer, simulator, m_time);
        }

        if(m_logFile != 0)
            m_outputBuffer.log(m_logFile);
    }
    
    void System::step()
    {
    }


    /////////////////////////////////////////////////////////////////////////
} // namespace Rescue
