Load "openssllib.ring"

// Blowfish test vectors: https://www.schneier.com/wp-content/uploads/2015/12/vectors-2.txt
// 
bf_ecb_vectors = [
	[:key = hex2str("FEDCBA9876543210"),:data = hex2str("0123456789ABCDEF"),:iv = NULL, :result = hex2str("0ACEAB0FC6A0A28D559CE0648C2EDADC")],
	[:key = hex2str("7CA110454A1A6E57"),:data = hex2str("01A1D6D039776742"),:iv = NULL, :result = hex2str("59C68245EB05282BB58B6ADD67F69788")]]

bf_cbc_vectors = [
	[:key = hex2str("0123456789ABCDEFF0E1D2C3B4A59687"),:data = hex2str("37363534333231204E6F77206973207468652074696D6520666F722000"),:iv = hex2str("FEDCBA9876543210"), :result = hex2str("6B77B4D63006DEE605B156E27403979358DEB9E7154616D9749DECBEC05D264B")]]

bf_cfb_vectors = [
	[:key = hex2str("0123456789ABCDEFF0E1D2C3B4A59687"),:data = hex2str("37363534333231204E6F77206973207468652074696D6520666F722000"),:iv = hex2str("FEDCBA9876543210"), :result = hex2str("E73214A2822139CAF26ECF6D2EB9E76E3DA3DE04D1517200519D57A6C3")]]

bf_ofb_vectors = [
	[:key = hex2str("0123456789ABCDEFF0E1D2C3B4A59687"),:data = hex2str("37363534333231204E6F77206973207468652074696D6520666F722000"),:iv = hex2str("FEDCBA9876543210"), :result = hex2str("E73214A2822139CA62B343CC5B65587310DD908D0C241B2263C2CF80DA")]]

// DES and TripleDES test vectors based on those of Botan library:
// 	- https://github.com/randombit/botan/tree/master/src/tests/data/block
//	- https://github.com/randombit/botan/tree/master/src/tests/data/modes
des_ecb_vectors = [
	[:key = hex2str("0123456789ABCDEF"),:data = hex2str("666F7220616C6C2068652074696D6520"),:iv = NULL, :result = hex2str("893D51EC4B563B536A271787AB8883F9086F9A1D74C94D4E")],
	[:key = hex2str("BEBAFBEABAFFEAAF"),:data = hex2str("9D4A44AEFCE79965"),:iv = NULL, :result = hex2str("77B2ECC9278E97140A1E1A829EB57391")]]

des3_ecb_vectors = [
	[:key = hex2str("0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF"),:data = hex2str("0123456789ABCDE768652074696D65204E6F772069732074"),:iv = NULL, :result = hex2str("C95744256A5ED31D6A271787AB8883F93FA40E8A984D4815086F9A1D74C94D4E")],
	[:key = hex2str("EFACE1D2899E742C6A65EB523447B6001E2FE3DB1D6BD904"),:data = hex2str("2E499BC53CFEB666"),:iv = NULL, :result = hex2str("87B59F9BD6ADE17CEA341DF5E0590DE1")]]

des_cbc_vectors = [
	[:key = hex2str("0123456789ABCDEF"),:data = hex2str("4E6F77206973207468652074696D6520666F7220616C6C20"),:iv = hex2str("1234567890ABCDEF"), :result = hex2str("E5C7CDDE872BF27C43E934008C389C0F683788499A7C05F662C16A27E4FCF277")],
	[:key = hex2str("D581A1D0C70F94A1"),:data = hex2str("69"),:iv = hex2str("C1DDD7447249EF80"), :result = hex2str("60FA7B46523AA51F")]]

des3_cbc_vectors = [
	[:key = hex2str("08763da862ad16ef5815408f5d3b705415ab1543a42c3efb"),:data = hex2str("109a3d3d745d65b38edbc73d1de8b2807f7820221a6c3937faab19fcbb75d3c8aaf4b63f2714cfc94e95ae43d65f6df43815efc214ec66a5d1be185d855a6260141ffd179bc980490f8a26d8215dd2ab"),:iv = hex2str("0634d69eaff3ae17"), :result = hex2str("e9513e8892a09085bee29c358014afd60d7578d21e00a31e5d61b965c18778ebe18469170794e5ddf24aa777c8ab0a2c62474109e617978bcc5ce3456ddd9622833420443c2a26b1b6e20a05c189da6c46D5F74FA507C4B3")]]
	
des3_cfb_vectors = [
	[:key = hex2str("4f5226019db36b40464392c8e5c42ad95e2032263dce8f20"),:data = hex2str("6a9aafb34b4dd8ad400721b5c2ee625015c89a92cb6d92358b7cbde8104e63e97428fed6afb5e5c15bb32782baecba51f968d594b6e19af5c177f8b81fa5d04b5574561fe3eebbb70ed609ffe2964ede"),:iv = hex2str("ee24f2e7d6d1f6f9"), :result = hex2str("349ddf37bfa125304f91035aa02b5fd5c49c6fd244379c43cf7ebecad489e38aedeb7490a77bb1777d628f018c86222ebf4d12483f979ddf136f35151c565c54f86f2459adc7b1668066e69fc2661701")]]
	
des3_cfb8_vectors = [
	[:key = hex2str("c4d345fddc9df7023b04e676528a2f57e051ba866b987aba"),:data = hex2str("1d1e42cd"),:iv = hex2str("d76fbd434d8b2ed9"), :result = hex2str("44deeee3")],
	[:key = hex2str("e3347a6b0bc1152c642a25cbd3bc31abfba162a81f197c15"),:data = hex2str("dbe915fcb33bca18ef14"),:iv = hex2str("b740cc21e925e3c8"), :result = hex2str("f4801a8d039db4ca8ff6")]]

// AES test vectors based on those of Botan library:
// 	- https://github.com/randombit/botan/tree/master/src/tests/data/block
//	- https://github.com/randombit/botan/tree/master/src/tests/data/modes

aes128_ecb_vectors = [
	[:key = hex2str("000102030405060708090A0B0C0D0E0F"),:data = hex2str("00112233445566778899AABBCCDDEEFF"),:iv = NULL, :result = hex2str("69C4E0D86A7B0430D8CDB78070B4C55A954F64F2E4E86E9EEE82D20216684899")],
	[:key = hex2str("A0A1A2A3A5A6A7A8AAABACADAFB0B1B2"),:data = hex2str("41321EE10E21BD907227C4450FF42324"),:iv = NULL, :result = hex2str("D2AB7662DF9B8C740210E5EEB61C199DF5E54D0575DD48D906B8724355B91AC1")],
	[:key = hex2str("F8F9FAFBFDFEFE00020304050708090A"),:data = hex2str("32333031C2C1C13F0D0C0B0A050A0B08"),:iv = NULL, :result = hex2str("2C3164C1CC7D0064816BDC0FAA362C52B9F96CEEDFD04D2E5002C3B649EF022A")],
	[:key = hex2str("00000000000000000000000000000000"),:data = hex2str("80000000000000000000000000000000400000000000000000000000000000002000000000000000000000000000000010000000000000000000000000000000"),:iv = NULL, :result = hex2str("3AD78E726C1EC02B7EBFE92B23D9EC3445BC707D29E8204D88DFBA2F0B0CAD9B161556838018F52805CDBD6202002E3FF5569B3AB6A6D11EFDE1BF0A64C6854A0143DB63EE66B0CDFF9F69917680151E")]]

aes128_cbc_vectors = [
	[:key = hex2str("10d6f8e78c0ccf8736e4307aaf5b07ef"),:data = hex2str("3a"),:iv = hex2str("3eb182d95bbd5a609aecfb59a0ca898b"), :result = hex2str("a7d290687ae325054a8691014d6821d7")],
	[:key = hex2str("10d6f8e78c0ccf8736e4307aaf5b07ef"),:data = hex2str("3a513eb569a503b4413b31"),:iv = hex2str("3eb182d95bbd5a609aecfb59a0ca898b"), :result = hex2str("c25342f688b2a45d7499ba54eebea246")],
	[:key = hex2str("10d6f8e78c0ccf8736e4307aaf5b07ef"),:data = hex2str("3a513eb569a503b4413b31fa883ddc88bcd6c62226c44c546b5e79d197159337a74ee109dc6e79b82e7fec3eff3bef26d1d50672ef5529a3c18c3e9522fc2097"),:iv = hex2str("3eb182d95bbd5a609aecfb59a0ca898b"), :result = hex2str("0cbaf4fa94df265fe264633a994bc25fc13b613502b81c32d7e4b94d4c6026fb25939e898ad7a66b8dd0c5191b8d1fa246c8cbdb4a582d44cd20c2c768aa6984573a891bd4e64cb33fcf1f0cf69dd78f")],
	[:key = hex2str("2B7E151628AED2A6ABF7158809CF4F3C"),:data = hex2str("6BC1BEE22E409F96E93D7E117393172AAE2D8A571E03AC9C9EB76FAC45AF8E5130C81C46A35CE411E5FBC1191A0A52EFF69F2445DF4F9B17AD2B417BE66C3710"),:iv = hex2str("000102030405060708090A0B0C0D0E0F"), :result = hex2str("7649ABAC8119B246CEE98E9B12E9197D5086CB9B507219EE95DB113A917678B273BED6B8E3C1743B7116E69E222295163FF1CAA1681FAC09120ECA307586E1A78CB82807230E1321D3FAE00D18CC2012")],
	[:key = hex2str("2B7E151628AED2A6ABF7158809CF4F3C"),:data = hex2str("6BC1BEE22E409F96E93D7E117393172AAE2D8A571E03AC9C9EB76FAC45AF8E5130C81C46A35CE411E5FBC1191A0A52EFF69F2445DF4F9B17AD2B417BE66C3710"),:iv = hex2str("00000000000000000000000000000000"), :result = hex2str("3AD77BB40D7A3660A89ECAF32466EF97B148C17F309EE692287AE57CF12ADD49C93D11BFAF08C5DC4D90B37B4DEE002BA7356E1207BB406639E5E5CEB9A9ED93DA089D0F8CA5D2D60E505EB6A4F6BF07")]]
	
aes128_cfb_vectors = [
	[:key = hex2str("2B7E151628AED2A6ABF7158809CF4F3C"),:data = hex2str("6BC1BEE22E409F96E93D7E117393172AAE2D8A571E03AC9C9EB76FAC45AF8E5130C81C46A35CE411E5FBC1191A0A52EFF69F2445DF4F9B17AD2B417BE66C3710"),:iv = hex2str("000102030405060708090A0B0C0D0E0F"), :result = hex2str("3B3FD92EB72DAD20333449F8E83CFB4AC8A64537A0B3A93FCDE3CDAD9F1CE58B26751F67A3CBB140B1808CF187A4F4DFC04B05357C5D1C0EEAC4C66F9FF7F2E6")]]

// OpenSSL 1.1.1 rejects duplicated XTS keys, so we ensure that our test vectors follow this
// https://github.com/openssl/openssl/blob/c39352/crypto/evp/e_aes.c#L3119
aes128_xts_vectors = [
	[:key = hex2str("1111111111111111111111111111111122222222222222222222222222222222"),:data = hex2str("4444444444444444444444444444444444444444444444444444444444444444"),:iv = hex2str("33333333330000000000000000000000"), :result = hex2str("C454185E6A16936E39334038ACEF838BFB186FFF7480ADC4289382ECD6D394F0")],
	[:key = hex2str("FFFEFDFCFBFAF9F8F7F6F5F4F3F2F1F022222222222222222222222222222222"),:data = hex2str("4444444444444444444444444444444444444444444444444444444444444444"),:iv = hex2str("33333333330000000000000000000000"), :result = hex2str("AF85336B597AFC1A900B2EB21EC949D292DF4C047E0B21532186A5971A227A89")],
	[:key = hex2str("a1b90cba3f06ac353b2c343876081762090923026e91771815f29dab01932f2f"),:data = hex2str("ebabce95b14d3c8d6fb350390790311c"),:iv = hex2str("4faef7117cda59c66e4b92013e768ad5"), :result = hex2str("778ae8b43cb98d5a825081d5be471c63")]]

aes192_ecb_vectors = [
	[:key = hex2str("00010203050607080A0B0C0D0F10111214151617191A1B1C"),:data = hex2str("2D33EEF2C0430A8A9EBF45E809C40BB6"),:iv = NULL, :result = hex2str("DFF4945E0336DF4C1C56BC700EFF837F31A4570A0ED8B275B0C0F739489CE19B")],
	[:key = hex2str("3C3D3E3F41424344464748494B4C4D4E5051525355565758"),:data = hex2str("BC3736518B9490DCB8ED60EB26758ED4"),:iv = NULL, :result = hex2str("D23684E3D963B3AFCF1A114ACA90CBD6D4B73763A9B3EC4B94D2DD677520CDB3")],
	[:key = hex2str("969798999B9C9D9EA0A1A2A3A5A6A7A8AAABACADAFB0B1B2"),:data = hex2str("E2AEF6ACC33B965C4FA1F91C75FF6F36"),:iv = NULL, :result = hex2str("D17D073B01E71502E28B47AB551168B3458FF75CD8C54E11F644C6184AAB3C0A")],
	[:key = hex2str("000000000000000000000000000000000000000000000000"),:data = hex2str("00000000000000000000000000000040000000000000000000000000000000200000000000000000000000000000001000000000000000000000000000000008000000000000000000000000000000040000000000000000000000000000000200000000000000000000000000000001"),:iv = NULL, :result = hex2str("53995DE0009CA18BECAFB8307C54C14C2006BF99F4C58B6CC2627856593FAEEA2DA697D2737CB30B744A4644FA1CBC6E47A22ACDB60C3A986A8F76ECD0EA3433FDAA17C2CDE20268FE36E164EA53215198E7247C07F0FE411C267E4384B0F600CD33B28AC773F74BA00ED1F31257243502BB292527E726FD51EB29894D6F0AAD")]]

aes192_cbc_vectors = [
	[:key = hex2str("8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B"),:data = hex2str("6BC1BEE22E409F96E93D7E117393172AAE2D8A571E03AC9C9EB76FAC45AF8E5130C81C46A35CE411E5FBC1191A0A52EFF69F2445DF4F9B17AD2B417BE66C3710"),:iv = hex2str("000102030405060708090A0B0C0D0E0F"), :result = hex2str("4F021DB243BC633D7178183A9FA071E8B4D9ADA9AD7DEDF4E5E738763F69145A571B242012FB7AE07FA9BAAC3DF102E008B0E27988598881D920A9E64F5615CD612CCD79224B350935D45DD6A98F8176")]]
	
aes192_cfb_vectors = [
	[:key = hex2str("8E73B0F7DA0E6452C810F32B809079E562F8EAD2522C6B7B"),:data = hex2str("6BC1BEE22E409F96E93D7E117393172AAE2D8A571E03AC9C9EB76FAC45AF8E5130C81C46A35CE411E5FBC1191A0A52EFF69F2445DF4F9B17AD2B417BE66C3710"),:iv = hex2str("000102030405060708090A0B0C0D0E0F"), :result = hex2str("CDC80D6FDDF18CAB34C25909C99A417467CE7F7F81173621961A2B70171D3D7A2E1E8A1DD59B88B1C8E60FED1EFAC4C9C05F9F9CA9834FA042AE8FBA584B09FF")]]
	
aes256_ecb_vectors = [
	[:key = hex2str("000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F"),:data = hex2str("00112233445566778899AABBCCDDEEFF"),:iv = NULL, :result = hex2str("8EA2B7CA516745BFEAFC49904B4960899F3B7504926F8BD36E3118E903A4CD4A")],
	[:key = hex2str("50515253555657585A5B5C5D5F60616264656667696A6B6C6E6F707173747576"),:data = hex2str("A291D86301A4A739F7392173AA3C604C"),:iv = NULL, :result = hex2str("6585C8F43D13A6BEAB6419FC5935B9D0CD3131FD2F2B5A0581B2A6D3494F431A")],
	[:key = hex2str("90919293959697989A9B9C9D9FA0A1A2A4A5A6A7A9AAABACAEAFB0B1B3B4B5B6"),:data = hex2str("D8CEAAF8976E5FBE1012D8C84F323799"),:iv = NULL, :result = hex2str("09614206D15CBACE63227D06DB6BEEBBB53A5220D42C9C1C03AE4C1182185F86")],
	[:key = hex2str("0000000000000000000000000000000000000000000000000000000000000000"),:data = hex2str("0000000000000000000000000000001000000000000000000000000000000008000000000000000000000000000000040000000000000000000000000000000200000000000000000000000000000001"),:iv = NULL, :result = hex2str("1490A05A7CEE43BDE98B56E309DC0126ABFA77CD6E85DA245FB0BDC5E52CFC29DD4AB1284D4AE17B41E85924470C36F7CEA7403D4D606B6E074EC5D3BAF39D18530F8AFBC74536B9A963B4F1C4CB738B1F788FE6D86C317549697FBF0C07FA43")]]

aes256_cbc_vectors = [
	[:key = hex2str("603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4"),:data = hex2str("6BC1BEE22E409F96E93D7E117393172AAE2D8A571E03AC9C9EB76FAC45AF8E5130C81C46A35CE411E5FBC1191A0A52EFF69F2445DF4F9B17AD2B417BE66C3710"),:iv = hex2str("000102030405060708090A0B0C0D0E0F"), :result = hex2str("F58C4C04D6E5F1BA779EABFB5F7BFBD69CFC4E967EDB808D679F777BC6702C7D39F23369A9D9BACFA530E26304231461B2EB05E2C39BE9FCDA6C19078C6A9D1B3F461796D6B0D6B2E0C2A72B4D80E644")]]

aes256_cfb_vectors = [
	[:key = hex2str("603DEB1015CA71BE2B73AEF0857D77811F352C073B6108D72D9810A30914DFF4"),:data = hex2str("6BC1BEE22E409F96E93D7E117393172AAE2D8A571E03AC9C9EB76FAC45AF8E5130C81C46A35CE411E5FBC1191A0A52EFF69F2445DF4F9B17AD2B417BE66C3710"),:iv = hex2str("000102030405060708090A0B0C0D0E0F"), :result = hex2str("DC7E84BFDA79164B7ECD8486985D386039FFED143B28B1C832113C6331E5407BDF10132415E54B92A13ED0A8267AE2F975A385741AB9CEF82031623D55B1E471")]]

aes256_xts_vectors = [
	[:key = hex2str("27182818284590452353602874713526624977572470936999595749669676273141592653589793238462643383279502884197169399375105820974944592"),:data = hex2str("000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF"),:iv = hex2str("FF000000000000000000000000000000"), :result = hex2str("1C3B3A102F770386E4836C99E370CF9BEA00803F5E482357A4AE12D414A3E63B5D31E276F8FE4A8D66B317F9AC683F44680A86AC35ADFC3345BEFECB4BB188FD5776926C49A3095EB108FD1098BAEC70AAA66999A72A82F27D848B21D4A741B0C5CD4D5FFF9DAC89AEBA122961D03A757123E9870F8ACF1000020887891429CA2A3E7A7D7DF7B10355165C8B9A6D0A7DE8B062C4500DC4CD120C0F7418DAE3D0B5781C34803FA75421C790DFE1DE1834F280D7667B327F6C8CD7557E12AC3A0F93EC05C52E0493EF31A12D3D9260F79A289D6A379BC70C50841473D1A8CC81EC583E9645E07B8D9670655BA5BBCFECC6DC3966380AD8FECB17B6BA02469A020A84E18E8F84252070C13E9F1F289BE54FBC481457778F616015E1327A02B140F1505EB309326D68378F8374595C849D84F4C333EC4423885143CB47BD71C5EDAE9BE69A2FFECEB1BEC9DE244FBE15992B11B77C040F12BD8F6A975A44A0F90C29A9ABC3D4D893927284C58754CCE294529F8614DCD2ABA991925FEDC4AE74FFAC6E333B93EB4AFF0479DA9A410E4450E0DD7AE4C6E2910900575DA401FC07059F645E8B7E9BFDEF33943054FF84011493C27B3429EAEDB4ED5376441A77ED43851AD77F16F541DFD269D50D6A5F14FB0AAB1CBB4C1550BE97F7AB4066193C4CAA773DAD38014BD2092FA755C824BB5E54C4F36FFDA9FCEA70B9C6E693E148C151")],
	[:key = hex2str("27182818284590452353602874713526624977572470936999595749669676273141592653589793238462643383279502884197169399375105820974944592"),:data = hex2str("000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF"),:iv = hex2str("FFFFFFFFFF0000000000000000000000"), :result = hex2str("64497E5A831E4A932C09BE3E5393376DAA599548B816031D224BBF50A818ED2350EAE7E96087C8A0DB51AD290BD00C1AC1620857635BF246C176AB463BE30B808DA548081AC847B158E1264BE25BB0910BBC92647108089415D45FAB1B3D2604E8A8EFF1AE4020CFA39936B66827B23F371B92200BE90251E6D73C5F86DE5FD4A950781933D79A28272B782A2EC313EFDFCC0628F43D744C2DC2FF3DCB66999B50C7CA895B0C64791EEAA5F29499FB1C026F84CE5B5C72BA1083CDDB5CE45434631665C333B60B11593FB253C5179A2C8DB813782A004856A1653011E93FB6D876C18366DD8683F53412C0C180F9C848592D593F8609CA736317D356E13E2BFF3A9F59CD9AEB19CD482593D8C46128BB32423B37A9ADFB482B99453FBE25A41BF6FEB4AA0BEF5ED24BF73C762978025482C13115E4015AAC992E5613A3B5C2F685B84795CB6E9B2656D8C88157E52C42F978D8634C43D06FEA928F2822E465AA6576E9BF419384506CC3CE3C54AC1A6F67DC66F3B30191E698380BC999B05ABCE19DC0C6DCC2DD001EC535BA18DEB2DF1A101023108318C75DC98611A09DC48A0ACDEC676FABDF222F07E026F059B672B56E5CBC8E1D21BBD867DD927212054681D70EA737134CDFCE93B6F82AE22423274E58A0821CC5502E2D0AB4585E94DE6975BE5E0B4EFCE51CD3E70C25A1FBBBD609D273AD5B0D59631C531F6A0A57B9")],
	[:key = hex2str("FFFEFDFCFBFAF9F8F7F6F5F4F3F2F1F0EFEEEDECEBEAE9E8E7E6E5E4E3E2E1E0BFBEBDBCBBBAB9B8B7B6B5B4B3B2B1B0AFAEADACABAAA9A8A7A6A5A4A3A2A1A0"),:data = hex2str("000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E"),:iv = hex2str("9A785634120000000000000000000000"), :result = hex2str("581EA1FEE5516AD432DDEBE75FD27C6FC30CA8F2ED57307EDC87E544867AC8")]]


/* Start of the test program */

// list all ciphers that are supported by the underlying openssl library
$SupportedCiphersList = supportedCiphers()

See String(Len($SupportedCiphersList)) + " supported ciphers = "
PrintList($SupportedCiphersList)
See nl

ValidateDefaultAlgorithm()
ValidateBlowfishECB()
ValidateBlowfishCBC()
ValidateBlowfishCFB()
ValidateBlowfishOFB()

ValidateDesECB()
ValidateDesCBC()
ValidateDes3ECB()
ValidateDes3CBC()
ValidateDes3CFB()
ValidateDes3CFB8()

ValidateAES128ECB()
ValidateAES128CBC()
ValidateAES128CFB()
ValidateAES128XTS()

ValidateAES192ECB()
ValidateAES192CBC()
ValidateAES192CFB()

ValidateAES256ECB()
ValidateAES256CBC()
ValidateAES256CFB()
ValidateAES256XTS()


/********************************************************************/

func PrintList(aList)
	count = Len(aList)
	See "["
	if count >= 1
		See alist[1]
		for i=2 to count
			See ", " + aList[i]
		next
	ok
	See "]"
	See nl

func ValidateTestVectors (cipherName, testDescription, testVectors)
	/* skip test of algorithm not supported */
	found = 0
	if isnull(cipherName)
		/* default cipher is "bf" for compatibility with older Ring versions */
		found = Find ($SupportedCiphersList, "bf-cbc")
	else
		found = Find ($SupportedCiphersList, cipherName)
	ok
	testResult = "OK"
	if found >= 1
		count = Len(testVectors)
		for i=1 to count
			vect = testVectors[i]
			/* catch any internal failure */
			try
				encryptResult = encrypt(vect[:data], vect[:key], vect[:iv], cipherName)
				if isnull(cipherName)
					/* we check that not specifying cipherName is equivalent to passing NULL */
					encryptResult2 = encrypt(vect[:data], vect[:key], vect[:iv])

					if encryptResult != encryptResult2
						See testDescription + " failing encryption test with NULL cipherName" + nl
						testResult="FAILED"
						exit
					ok
				ok
			catch
				encryptResult = ""
			done

			if encryptResult != vect[:result]
				See testDescription + " failing encryption test vector: " + nl
				See "   key = " + str2hex(vect[:key]) + nl
				See "   IV = " + str2hex(vect[:iv]) + nl
				See "   data = " + str2hex(vect[:data]) + nl
				See "   result = " + Upper(str2hex(encryptResult)) + nl
				See "   expected result = " + Upper(str2hex(vect[:result])) + nl
				testResult="FAILED" 
				exit 
			ok
			
			/* catch any internal failure */
			try
				/* here we test passing cipherName as NULL */
				decryptResult = decrypt(vect[:result], vect[:key], vect[:iv], cipherName)
				if isnull(cipherName)
					/* we check that not specifying cipherName is equivalent to passing NULL */
					decryptResult2 = decrypt(vect[:result], vect[:key], vect[:iv])

					if decryptResult != decryptResult2
						See testDescription + " failing decryption test with NULL cipherName" + nl
						testResult="FAILED"
						exit
					ok
				ok
			catch
				decryptResult = ""
			done

			if decryptResult != vect[:data]
				See testDescription + " failing decryption test vector: " + nl
				See "   key = " + str2hex(vect[:key]) + nl
				See "   IV = " + str2hex(vect[:iv]) + nl
				See "   data = " + str2hex(vect[:data]) + nl
				See "   result = " + Upper(str2hex(encryptResult)) + nl
				See "   expected result = " + Upper(str2hex(vect[:result])) + nl
				testResult="FAILED" 
				exit 
			ok
		next
	else
		/* skip test */
		testResult="SKIPPED"
	ok
	See testDescription + " = " + testResult + nl
	
func ValidateDefaultAlgorithm
	ValidateTestVectors("", "Default Algo", bf_cbc_vectors)

func ValidateBlowfishECB
	ValidateTestVectors("bf-ecb", "Blowfish ECB", bf_ecb_vectors)

func ValidateBlowfishCBC
	ValidateTestVectors("bf-cbc", "Blowfish CBC", bf_cbc_vectors)

func ValidateBlowfishCFB
	ValidateTestVectors("bf-cfb", "Blowfish CFB", bf_cfb_vectors)

func ValidateBlowfishOFB
	ValidateTestVectors("bf-ofb", "Blowfish OFB", bf_ofb_vectors)
	
func ValidateDes3ECB
	ValidateTestVectors("des-ede3", "TripleDES ECB", des3_ecb_vectors)
	
func ValidateDes3CBC
	ValidateTestVectors("des-ede3-cbc", "TripleDES CBC", des3_cbc_vectors)
	
func ValidateDes3CFB
	ValidateTestVectors("des-ede3-cfb", "TripleDES CFB", des3_cfb_vectors)
	
func ValidateDes3CFB8
	ValidateTestVectors("des-ede3-cfb8", "TripleDES CFB8", des3_cfb8_vectors)
	
func ValidateDesECB
	ValidateTestVectors("des-ecb", "DES ECB", des_ecb_vectors)
	
func ValidateDesCBC
	ValidateTestVectors("des-cbc", "DES CBC", des_cbc_vectors)

func ValidateAES128ECB
	ValidateTestVectors("aes-128-ecb", "AES-128 ECB", aes128_ecb_vectors)

func ValidateAES128CBC
	ValidateTestVectors("aes-128-cbc", "AES-128 CBC", aes128_cbc_vectors)
	
func ValidateAES128CFB
	ValidateTestVectors("aes-128-cfb", "AES-128 CFB", aes128_cfb_vectors)
	
func ValidateAES128XTS
	ValidateTestVectors("aes-128-xts", "AES-128 XTS", aes128_xts_vectors)
	
func ValidateAES192ECB
	ValidateTestVectors("aes-192-ecb", "AES-192 ECB", aes192_ecb_vectors)

func ValidateAES192CBC
	ValidateTestVectors("aes-192-cbc", "AES-192 CBC", aes192_cbc_vectors)
	
func ValidateAES192CFB
	ValidateTestVectors("aes-192-cfb", "AES-192 CFB", aes192_cfb_vectors)

func ValidateAES256ECB
	ValidateTestVectors("aes-256-ecb", "AES-256 ECB", aes256_ecb_vectors)

func ValidateAES256CBC
	ValidateTestVectors("aes-256-cbc", "AES-256 CBC", aes256_cbc_vectors)

func ValidateAES256CFB
	ValidateTestVectors("aes-256-cfb", "AES-256 CFB", aes256_cfb_vectors)
	
func ValidateAES256XTS
	ValidateTestVectors("aes-256-xts", "AES-256 XTS", aes256_xts_vectors)
