/**
 * @file nes_posix_signal.c
 * @brief core signal interface implement
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <errno.h>
#include <stdlib.h>
#include <time.h>
#include <btron/errcode.h>
#include <btron/proctask.h>

#include "nes_posix_signal.h"
#include "sigmanager.h"		/* signal manager */
#include "manager.h"		/* pthread manager */
#include "memory_check.h"

extern sigset_t __g_regist_sigmask;			/* sigmanager.c */
extern sigset_t __g_block_sigmask;			/* sigmanager.c */
extern struct sigaction __g_all_actions[_NSIG+1];	/* sigmanager.c */
extern struct sigpending __g_sigpending;		/* sigmanager.c  */
extern struct __alarm_time __g_alarm_time;		/* sigmanager.c  */
extern struct __sigwait_queue __g_sigwait_queue;	/* sigmanager.c */

/**********************************************************************/
/* Function name: nes_posix_getpid                                              */
/* Description: returns  the process ID of the current process        */
/* Return type - pid_t:  Current process's ID                         */
/* Argument - void:                                                   */
/**********************************************************************/
pid_t nes_posix_getpid(void) 
{
	return b_prc_sts(0, NULL, NULL); 
}

/**********************************************************************/
/* Function name: nes_posix_sigfillset                                          */
/* Description:nes_posix_sigfillset() initializes set to full,including all     */
/*             signals.                                               */
/* Return type - int : return 0 on success and -1 on error.           */
/* Argument - sigset_t *set: sigset_t pointer                         */
/**********************************************************************/
int nes_posix_sigfillset(sigset_t *set)
{
	memset(set, 0xFF, sizeof(sigset_t));
	return 0;	
}

/**********************************************************************/
/* Function name: nes_posix_sigemptyset                                         */
/* Description: nes_posix_sigemptyset() initializes the signal set given by set */
/*              to empty, with allsignals excluded from the set.      */
/* Return type - int : return 0 on success and -1 on error.           */
/* Argument - sigset_t *set: sigset_t pointer                         */
/**********************************************************************/
int nes_posix_sigemptyset(sigset_t *set)
{
	memset(set, 0, sizeof(sigset_t));
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_sigaddset                                           */
/* Description:  add respectively signal signum from set.             */
/* Return type - int :                                                */
/* Argument - sigset_t *set: sigset_t pointer                         */
/* Argument - int _sig:                                               */
/**********************************************************************/
int nes_posix_sigaddset(sigset_t *set, int _sig)
{
	/* Signal mask : 1 << ((sig-1) % _NSIG_BPW) */
	unsigned int s = _sig - 1;
	if( _sig < 1 || _sig > 64 ) 
	{
		errno = EINVAL;
		return -1;
	}
	set->sig[s / _NSIG_BPW] |= (1 << (s % _NSIG_BPW)); 
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_sigdelset                                           */
/* Description: delete respectively signal signum from set.           */
/* Return type - int :                                                */
/* Argument - sigset_t *set: sigset_t pointer                         */
/* Argument - int _sig:                                               */
/**********************************************************************/
int nes_posix_sigdelset(sigset_t *set, int _sig)
{
	unsigned int s = _sig - 1;
	if( _sig < 1 || _sig > 64 ) 
	{
		errno = EINVAL;
		return -1;
	}
	set->sig[s / _NSIG_BPW] &= ~(1 << (s % _NSIG_BPW)); 
	return 0;	
}

/**********************************************************************/
/* Function name: nes_posix_sigismember                                         */
/* Description: nes_posix_sigismember() tests whether signum is a member of set.*/
/* Return type - int:returns 1 if signum is a member of set, 0 if     */
/*                   signum is not a member, and -1 on error.         */
/* Argument - sigset_t *set: sigset_t pointer                         */
/* Argument - int _sig:  respectively signal                          */
/**********************************************************************/
int nes_posix_sigismember(const sigset_t *set, int _sig)
{
	unsigned int s = _sig - 1;
	if( _sig < 1 || _sig > 64 ) 
	{
		errno = EINVAL;
		return -1;
	}	
	return 1 & (set->sig[s / _NSIG_BPW] >> (s % _NSIG_BPW));
}


/**********************************************************************/
/* Function name: nes_posix_sigorsets                                           */
/* Description:                                                       */
/* Return type - int :                                                */
/* Argument - const sigset_t *in1:                                    */
/* Argument - const sigset_t *in2:                                    */
/* Argument - sigset_t *out:                                          */
/**********************************************************************/
int nes_posix_sigorsets(const sigset_t *in1, const sigset_t *in2, sigset_t *out)
{
	int i = 0;
	for(i = 0; i < _NSIG_WORDS; i++ ) 
	{
		out->sig[i] = in1->sig[i] | in1->sig[i];
	}
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_sigandsets                                */
/* Description:                                                       */
/* Return type - int :                                                */
/* Argument - const sigset_t *in1:                                    */
/* Argument - const sigset_t *in2:                                    */
/* Argument - sigset_t *out:                                          */
/**********************************************************************/
int nes_posix_sigandsets(const sigset_t *in1, const sigset_t *in2, sigset_t *out)
{
	int i = 0;
	for(i = 0; i < _NSIG_WORDS; i++ ) 
	{
		out->sig[i] = in1->sig[i] & in1->sig[i];
	}
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_sigaction                                           */
/* Description:The  nes_posix_sigaction() system  call is used to change the    */
/*             action taken by a process on receipt of a specific     */
/*             signal.                                                */
/* Return type - int:nes_posix_sigaction() returns 0 on success and -1 on error */
/* Argument - int signum: signum specifies the signal and can be any  */
/*            valid signal except SIGKILL and SIGSTOP.                */
/* Argument - const struct sigaction* act: If act is non-null, the    */
/*            new action for signal signum is installed from act.     */
/* Argument - struct sigaction* oldact:   If oldact is non-null,      */
/*            theprevious action is saved in oldact.                  */
/**********************************************************************/
int nes_posix_sigaction(int signum, const struct sigaction* act,  struct sigaction* oldact)
{
	/* _sa_sigaction_handler_t default_handler; */

	/* Can't catch SIGKILL and SIGSTOP */
	if( !__signal_is_valid(signum) || signum == SIGKILL || signum == SIGSTOP ) 
	{
		errno = EINVAL;
		return -1;
	}

	if( act && !act->sa_sigaction ) 		/* Invalid new handler */
	{
		errno = EINVAL;
		return -1;
	}

	GLOBAL_SIG_LOCK();
	if( oldact ) 
	{
		memcpy(oldact, &__g_all_actions[signum], sizeof(struct sigaction));
	}

	if( act )		
	{	
		/* Save new handler */	
		memcpy(&__g_all_actions[signum], act, sizeof(struct sigaction));
		nes_posix_sigaddset(&(__g_all_actions[signum].sa_mask), signum);		/*  Block self */
	} 
	else			
	{	
		/* Reset handler */	
		__g_all_actions[signum].sa_handler = SIG_DFL;
	}
	
	GLOBAL_SIG_UNLOCK();
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_signal                                              */
/* Description:The  nes_posix_signal()  call installs a new signal handler for  */
/*             the signal with number signum.  The signal handler is  */
/*             set to sighandler which may be a user specified        */
/*             function, or either SIG_IGN or SIG_DFL.                */
/* Return type - sighandler_t:The nes_posix_signal() function returns the       */
/*         previous value of the signal handler, or SIG_ERR on error. */
/* Argument - int signum: dest signal                                 */
/* Argument - sighandler_t handler: new signal handler.can be SIG_IGN */
/*            or SIG_DFL                                              */
/**********************************************************************/
sighandler_t nes_posix_signal(int signum, sighandler_t handler)
{
	int ret = 0;
	struct sigaction act, old;

	if( !handler ) 
	{
		return SIG_ERR;
	}
	
	/* Init new action */
	act.sa_handler = handler;
	act.sa_flags = 0;
	act.sa_flags |= SA_NOMASK;		/* Not set SA_ONESHOT flag */ 
	// act.sa_flags &= ~SA_SIGINFO;	/* Set signal param flag */

	nes_posix_sigemptyset(&act.sa_mask);
	nes_posix_sigaddset(&act.sa_mask, signum);

        ret = nes_posix_sigaction(signum, &act, &old);
	if( ret == 0 ) 
	{
		return old.sa_handler;
	}

	return SIG_ERR;
}

/**********************************************************************/
/* Function name: nes_posix_sigprocmask                                         */
/* Description:  change the signal mask                               */
/* Return type - int : returns 0 on success and -1 on error.          */
/*                                                                    */
/* Argument - int how::                                               */
/* SIG_BLOCK                                                          */
/* The set of blocked signals is the union of the current set and the */
/* set argument.                                                      */
/*                                                                    */
/* SIG_UNBLOCK                                                        */
/* The signals in set are removed from the current set of blocked     */
/* signals.It is o attempt to unblock a signal which is not blocked.  */
/*                                                                    */
/* SIG_SETMASK                                                        */
/* The set of blocked signals is set to the argument set.             */
/* Argument - const sigset_t* newset:New mask, If set  is NULL,then   */
/*                                   the signal mask is unchanged     */
/* Argument - sigset_t* oldset: If oldset is non-null, the previous   */
/*                       value of the signal mask is stored in oldset */
/**********************************************************************/
int nes_posix_sigprocmask(int how, const sigset_t* newset, sigset_t* oldset)
{
	int i = 0;
	sigset_t set;

	/* Invalid operation */
	if( how != SIG_BLOCK && how != SIG_UNBLOCK && how != SIG_SETMASK ) 
	{
		errno = EINVAL;
		return -1;
	}

	GLOBAL_SIG_LOCK();
	if( oldset ) 
	{
		memcpy(oldset, &__g_block_sigmask, sizeof(sigset_t));
	}

	if( newset ) 
	{
		set = *newset;
		nes_posix_sigdelset(&set, SIGKILL);
		nes_posix_sigdelset(&set, SIGSTOP);
		switch(how)
		{
		case SIG_BLOCK:		/* Add block signal */
			for(i = 0; i < _NSIG_WORDS; i++ )  __g_block_sigmask.sig[i] |= set.sig[i];
			break;
		case SIG_UNBLOCK:	/* Remove block signal */
			for(i = 0; i < _NSIG_WORDS; i++ )  __g_block_sigmask.sig[i] &= ~set.sig[i];
			break;
		case SIG_SETMASK:
			memcpy(&__g_block_sigmask, &set, sizeof(sigset_t));
			break;
		default:
			break;
		}

	}
	GLOBAL_SIG_UNLOCK();

	/* Notify helper thread to check block update */
	__send_block_notify();
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_sigpending                                          */
/* Description:nes_posix_sigpending() returns the set of signals that are pending*/
/*             for delivery to the calling thread (i.e., the signals  */
/*             which have been raised while blocked).The mask of      */
/*             pending signals is returned in set.                    */
/* Return type - int:nes_posix_sigpending() returns 0 on success and -1 on error*/
/* Argument - sigset_t* set:                                          */
/**********************************************************************/
int nes_posix_sigpending(sigset_t* set)
{
	GLOBAL_SIG_LOCK();
	memcpy(set, &(__g_sigpending.mask), sizeof(sigset_t));
	GLOBAL_SIG_UNLOCK();
	return 0;
}

/**********************************************************************/
/* Function name: __send_signal                                       */
/* Description: Build special MESSAGE for require and send.           */
/* Return type - int :  returns 0 on success and -1 on error          */
/* Argument - pid_t pid: Dest process id                              */
/* Argument - int signum:  signal wait to send                        */
/* Argument - int data:  signal data (if have else zero)              */
/* Argument - int sender: sender type                                 */
/* Argument - int thread:  send to thread? then threadid or zero      */
/**********************************************************************/
int __send_signal(pid_t pid, int signum, int data, int sender, int thread)
{
	ERR ret = 0;
	MESSAGE msg;

	if( pid <= 0 ) 
	{
		errno = EINVAL;
		return -1;
	}

	if( !__signal_is_valid(signum) )
	{
		errno = EINVAL;
		return -1;
	}

	/*
	 * msg->msg_body.ANYMSG.msg_str[32]
	 *
	 * 0       4         8       12       16       20       24        28       32  bytes
	 * +--------+--------+--------+--------+--------+--------+--------+--------+
	 * |signum  |sigdata | sender | thread |        |        |        |        |     
	 * +--------+--------+--------+--------+--------+--------+--------+--------+
	 */	
	msg.msg_type	= signum >= SIGRTMIN ? __MSG_REALTIME : __MSG_UNREALTIME;
	msg.msg_size	= 16;
	*(int*)(msg.msg_body.ANYMSG.msg_str)		= signum;		/* signum */
	*(int*)(msg.msg_body.ANYMSG.msg_str + 4)	= data;			/* sigdata */
	*(int*)(msg.msg_body.ANYMSG.msg_str + 8)	= sender;		/* sender */
	*(int*)(msg.msg_body.ANYMSG.msg_str + 12)	= thread;		/* thread */

	ret = b_snd_msg(pid, &msg, NOWAIT);

	/* Error mapping */
	switch(ret)
	{
	case ER_OK:
		ret = 0;
		break;
	case ER_MINTR:
		errno = EINTR;
		ret = -1;
		break;
	case ER_NOPRC:
		errno = ESRCH;
		ret = -1;
		break;
	case ER_NOSPC:
		errno = ENOMEM;
		ret = -1;
		break;
	case ER_ADR:
	case ER_SELF:
	case ER_PAR:
	case ER_SZOVR:
	default:
		errno = EINVAL;
		ret = -1;
		break;
	}

	return ret;
}

/**********************************************************************/
/* Function name: nes_posix_kill                                                */
/* Description: Send signal to process                                */
/* Return type - int :On success (at least one signal was sent),zero  */
/*                    is returned.                                    */
/* On error, -1 is returned, and errno is set appropriately.          */
/* Argument - pid_t pid:  Dest process id                             */
/* Argument - int signum:  Signal wait to send                        */
/**********************************************************************/
int nes_posix_kill(pid_t pid, int signum)
{
	/* 
	 * 1. Not support send signal to 'Process Group' 
	 * 2. Not support send signal to all process(except 1 and self) 
	 * 3. Only suppport send signal to extra process
	 */
	if( pid <= 0 ) 
	{
		errno = EINVAL;
		return -1;
	}

	if( !__signal_is_valid(signum) ) 
	{
		errno = EINVAL;
		return -1;
	}

	return __send_signal(pid, signum, 0, SI_USER, 0);
}

/**********************************************************************/
/* Function name: nes_posix_raise                                               */
/* Description: Send signal to current process                        */
/* Return type - int :On success (at least one signal was sent),zero  */
/*                    is returned.                                    */
/* On error, -1 is returned, and errno is set appropriately.          */
/* Argument - int signum:  Signal wait to send                        */
/**********************************************************************/
int nes_posix_raise(int signum)
{
	return nes_posix_kill(nes_posix_getpid(), signum);
}

/**********************************************************************/
/* Function name: nes_posix_sigqueue                                            */
/* Description: Send signal (can take append data) to process         */
/* Return type - int : On success (at least one signal was sent),zero */
/* is returned.On error, -1 is returned, and errno is set appropriately*/
/* Argument - pid_t pid: Dest process                                 */
/* Argument - int signum: Signal wait to send                         */
/* Argument - const union sigval value: add on signal data            */
/**********************************************************************/
int nes_posix_sigqueue(pid_t pid, int signum, const union sigval value)
{
	/* 
	 * 1. Not support send signal to 'Process Group' 
	 * 2. Not support send signal to all process(except 1 and self) 
	 * 3. Only suppport send signal to extra process
	 */
	if( pid <= 0 ) 
	{
		errno = EINVAL;
		return -1;	
	}

	if( !__signal_is_valid(signum) ) 
	{
		errno = EINVAL;
		return -1;
	}

	return __send_signal(pid, signum, value.sival_int, SI_QUEUE, 0);
}

/**********************************************************************/
/* Function name: nes_posix_alarm                                               */
/* Description:nes_posix_alarm() arranges for a SIGALRM signal to be delivered  */
/*             to the process in seconds seconds.If seconds is zero,  */
/*             no new alarm() is scheduled.In any event any previously*/
/*             set alarm() is cancelled.                              */
/* Return type - unsigned int:alarm() returns the number of seconds   */
/*                            remaining                               */
/* until any previously scheduled alarm was due to be                 */
/* delivered, or zero if there was no previously scheduled alarm.     */
/* Argument - unsigned int seconds: next SIGALRM send time            */
/**********************************************************************/
unsigned int nes_posix_alarm(unsigned int seconds)
{
	ERR call = 0;
	unsigned int alive_time = 0;
	sigqueue_t* node = NULL;

	if( seconds < 0 ) 
	{
		return seconds;
	} 


	/* Cancel timeout MESSAGE in queue */
	b_can_tmg();

	/* Remove SIGALAM from pending queue */
	GLOBAL_SIG_LOCK();
	if( nes_posix_sigismember(&__g_block_sigmask, SIGALRM) )
	{
		node = __find_and_remove_pending_signal(SIGALRM);
		if( node ) {
			MC_FREE(node);
			node = NULL;
		}
	}

	/* Get alive time */
	if( __g_alarm_time.__send_time > 0 ) 
	{
		alive_time = __g_alarm_time.__alive_time - (time(NULL) - __g_alarm_time.__send_time);
		alive_time = alive_time > 0 ? alive_time : 0;
	} 
	else 
	{
		alive_time = 0;
	}

	if( seconds == 0 ) 
	{
		__g_alarm_time.__alive_time = 0;
		__g_alarm_time.__send_time = 0;
		GLOBAL_SIG_UNLOCK();
		return alive_time;

	} 
	else 
	{
		__g_alarm_time.__alive_time = seconds;
		__g_alarm_time.__send_time = time(NULL);
	}

	GLOBAL_SIG_UNLOCK();	

	call = b_req_tmg(seconds*1000, 0);

	switch(call)
	{
	case ER_OK:
		return alive_time;
	case ER_NOSPC:
	case ER_PAR:
		return -1;
	}

	return alive_time;
}

/**********************************************************************/
/* Function name: nes_posix_pause                                               */
/* Description:The nes_posix_pause() causes the invoking process (or thread) to */
/*             sleep until a signal is received that either terminates*/
/*             it or causes it to call a signal-catching function.    */
/* Return type - int :The nes_posix_pause() function only returns when a signal */
/*                    was caught and the signal-catching function     */
/*                    returned. In this case nes_posix_pause() returns -1 and   */
/*                    errno is set to EINTR.                          */
/* Argument - void:                                                   */
/**********************************************************************/
int nes_posix_pause(void)
{
	/*
	 * Every signal's handler is excuted, main thread's block call should be interrupt
	 * So, when b_slp_tsk() is interrrupted, a signal must has reached.
	 */
	b_slp_tsk(-1);
	errno = EINTR;
	return -1;
}

/**********************************************************************/
/* Function name: nes_posix_sigsuspend                                          */
/* Description: nes_posix_sigsuspend() temporarily replaces the signal mask of  */
/*              the calling process with the mask given by mask and   */
/*              then suspends the process until delivery of a signal  */
/*              whose action is to invoke a signal  handler           */
/* Return type - int : nes_posix_sigsuspend() always returns -1, normally with  */
/*                     the error EINTR.                               */
/* Argument - const sigset_t* set: tempory signal mask                */
/**********************************************************************/
int nes_posix_sigsuspend(const sigset_t* set)
{
	sigset_t oldset;
	__pthread_desr* desr = NULL;

	if( nes_posix_pthread_self() == __MAIN_THREAD ) 
	{

		/* Replace block mask */
		nes_posix_sigprocmask(SIG_SETMASK, set, &oldset);
	
		/* Wait signal reach process */
		b_slp_tsk(-1);

		/* Recover block sigset */
		nes_posix_sigprocmask(SIG_SETMASK, &oldset, NULL);
	
	} 
	else 
	{
		GLOBAL_THREADMGR_LOCK();
		desr = __pthread_find_desr(nes_posix_pthread_self());
		if( desr ) 
		{
			memcpy(&oldset, &(desr->__block_sigmask), sizeof(sigset_t));  
			memcpy(&(desr->__block_sigmask), set, sizeof(sigset_t));
			nes_posix_sigdelset(&(desr->__block_sigmask), SIGKILL);
			nes_posix_sigdelset(&(desr->__block_sigmask), SIGSTOP);
		}
		GLOBAL_THREADMGR_UNLOCK();

		/* Wait signal reach thread */
		nes_posix_pause();

		GLOBAL_THREADMGR_LOCK();
		desr = __pthread_find_desr(nes_posix_pthread_self());
		if( desr ) 
		{
			memcpy(&(desr->__block_sigmask), &oldset, sizeof(sigset_t));  
		}
		GLOBAL_THREADMGR_UNLOCK();
	}

	/* 
	 * The function sigsuspend always returns -1, normally with the error EINTR. 
	 */
	errno = EINTR;	/* Set global error code */
	return -1;
}
