/**
 * @file clean.c
 * @define implement of cancel operation 
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tkse/extension/proctask.h>

#include "nes_posix_pthread.h" 
#include "manager.h"
#include "clean.h"
#include "common.h"
  
/**********************************************************************/ 
/* Function name: nes_posix_pthread_setcancelstate                              */ 
/* Description:   This function atomically sets the calling thread's  */
/*                cancelability state to 'state' and returns the      */
/*                previous cancelability state at the location        */
/*                referenced by 'oldstate'.                           */ 
/* Return type:   0: successfully set cancelability type              */
/*                EINVAL :'state' is invalid                          */ 
/* Argument -     state,                                              */
/*                oldstate                                            */ 
/*                           PTHREAD_CANCEL_ENABLE                    */
/*                           cancellation is enabled,                 */
/*                                                                    */
/*                           PTHREAD_CANCEL_DISABLE                   */
/*                           cancellation is disabled                 */ 
/**********************************************************************/ 
int nes_posix_pthread_setcancelstate(int state, int* oldstate)
{ 
	int iRet = 0;
	__pthread_desr* desr;
	pthread_t self;
	self = pthread_self();
	
	DP(("[%d] pthread_setcancelstate IN ---state = [%d]\n", self, state));
 
	//state is invalid 
	if (state < PTHREAD_CANCEL_ENABLE || state > PTHREAD_CANCEL_DISABLE)
	{ 
		iRet = EINVAL; 
		goto exit; 
	}
	//get old state 
	GLOBAL_THREADMGR_LOCK(); 
	desr = __pthread_find_desr(self);
	//desr is not existing
	if(!desr)
	{
		iRet = ESRCH;
		DP(("[%d] __pthread_find_desr() desr is not existing\n", self)); 
		GLOBAL_THREADMGR_UNLOCK(); 
		goto exit; 
	} 
	/*
	 *if oldstate is not null,store the state to the oldstate
	 */
	if(desr->cancelState)
	{
		*oldstate = desr->cancelState;
	}
	desr->cancelState = state;
	GLOBAL_THREADMGR_UNLOCK(); 	
exit:	
	DP(("[%d] pthread_setcancelstate OUT\n", self));	 
	return iRet; 
} 

/**********************************************************************/ 
/* Function name: nes_posix_pthread_setcanceltype                               */ 
/* Description:   This function atomically sets the calling thread's  */
/*                cancelability type to 'type' and returns the        */
/*                previous cancelability type at the location         */
/*                referenced by 'oldtype'.                            */
/*                the type valuse is ineffective,because the default  */
/*                execution is asynchronous in PMC T-Engine           */ 
/* Return type:   0: successfully set cancelability type              */
/*                EINVAL :'type' is invalid                           */ 
/* Argument -     type,                                               */
/*                oldtype                                             */ 
/*                                                                    */
/*                           PTHREAD_CANCEL_ASYNCHRONOUS              */
/*                           Asynchronous cancellation is allowed     */ 
/**********************************************************************/

int nes_posix_pthread_setcanceltype (int type, int* oldtype)
{
	int iRet = 0;
	__pthread_desr* desr;
	
	pthread_t self;
	self = pthread_self();
	DP(("[%d] pthread_setcanceltype IN\n", self)); 
	//state is invalid 
	if (type != PTHREAD_CANCEL_ASYNCHRONOUS)
	{ 
		iRet = EINVAL; 
		goto exit;
	}	 
	//get old state 
	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(self); 
	//desr is not existing
	if(!desr)
	{
		iRet = ESRCH;
		DP(("[%d] __pthread_find_desr() desr is not existing\n", self)); 
		GLOBAL_THREADMGR_UNLOCK(); 
		goto exit; 
	} 

	desr->cancelType = type; 
	iRet = 0;

	GLOBAL_THREADMGR_UNLOCK(); 
			 
exit:	
	DP(("[%d] pthread_setcanceltype OUT\n", self));
	return iRet; 
} 
 
/**********************************************************************/ 
/* Function name: nes_posix_pthread_cancel                                      */ 
/* Description:   This function requests cancellation of 'thread',when*/
/*                this function is executed, the thread is canceled on*/
/*                time.that is different with Linux .except that,     */
/*                can't use the function cancel itself                */ 
/* Return type: 0: cancel successfully                                */ 
/*              ESRCH: cancel failed                                  */ 
/* Argument - pthread_t thread: thread ID                             */  
/**********************************************************************/ 
int nes_posix_pthread_cancel(pthread_t thread)
{
	__pthread_desr* desr = NULL; 
	int iRet = 0; 
	
	DP1( D_MODU_CANC,("[%d] pthread_cancel [%d] IN\n", pthread_self(), thread));
	GLOBAL_THREADMGR_LOCK(); 
	desr = __pthread_find_desr(thread);
	//this thread is not existing or finished
	if(!desr)
	{
		iRet = ESRCH;
		DP1( D_MODU_CANC,("[%d] pthread_cancel() desr is not existing\n", thread)); 
		goto exit; 
	}
	//thread exist 
	if(desr->cancelState == PTHREAD_CANCEL_DISABLE)
	{
		//cancelation failed
		iRet = ESRCH; 
		goto exit;
	}

	iRet = tkse_ter_tsk(thread); 
	//return code mapping 
	if(iRet != E_OK) 
	{ 
		iRet = EINVAL; 
		DP1( D_MODU_CANC,("[%d] tkse_ter_tsk [%d] Failed!\n", pthread_self(), thread));	
		goto exit;		      
	}

	//set thread status
	desr->__status |= STATUS_THREAD_CANCEL_ALREADY;
	
	//if thread is joinable, don't delete the desr
	if( desr->__attr.__detachstate == PTHREAD_CREATE_JOINABLE ) 
	{
		__pthread_destory_all_cleanup(desr);
		__pthread_destory_all_tsd(desr);
		__internal_unlock(&(desr->__lock));
	}
	else
	{
		__pthread_destory_all_cleanup(desr);
		__pthread_destory_all_tsd(desr);
		__pthread_del_desr(thread);   
        	__pthread_desr_destroy(desr);
       		free(desr);
	}

exit:	
	GLOBAL_THREADMGR_UNLOCK();
	DP1( D_MODU_CANC,("[%d] pthread_cancel [%d] OUT\n", pthread_self(), thread));
	return iRet;
} 
/**********************************************************************/ 
/* Function name: __pthread_destory_all_cleanup                       */ 
/* Description: removes the most recently installed cleanup handler,  */
/*              and execute the handler                               */ 
/* Return type:     int                                               */ 
/* Argument - void* __desr                                            */
/**********************************************************************/  
int __pthread_destory_all_cleanup(void* __desr)
{
	__pthread_desr* desr = NULL;
	desr = (__pthread_desr*)__desr;
	struct _pthread_cleanup_buffer *tempbuf;
		
	//recovery the source when push is executed
	while(desr->p_cleanup)
	{
		
//		DP1( D_MODU_CANC,("[%d] pop is execute \n", pthread_self()));
		//__routine will be execute when it is poped
		desr->p_cleanup->__routine(desr->p_cleanup->__arg);
		//pop the last node	
		tempbuf = desr->p_cleanup;
		desr->p_cleanup = desr->p_cleanup->__prev;
		free(tempbuf);
	}
	desr->p_cleanup = NULL;
	return 0;
	
} 
 
/**********************************************************************/ 
/* Function name: nes_posix_pthread_testcancel                                  */ 
/* Description:    do nothing                                         */ 
/* Return type:     null                                              */ 
/* Argument - Void                                                    */
/**********************************************************************/ 
 
void nes_posix_pthread_testcancel(void)
{
	
}

/**********************************************************************/ 
/* Function name: pthread_cleanup_push                                */ 
/* Description:   installs the routine function with argument arg as  */
/*                a cleanup handler                                   */ 
/* Return type:   null                                                */
/* Argument -     void (*routine) (void *):                           */
/*                void *arg:                                          */
/**********************************************************************/

void _pthread_cleanup_push(void (*routine)(void *), void *arg)
{
	int iRet = 0;
	__pthread_desr* desr = NULL;	
	pthread_t self;
	struct _pthread_cleanup_buffer *pbuf = NULL;	
	self = pthread_self();
	DP1( D_MODU_CANC,("[%d] pthread_cleanup_push IN\n", self));

	GLOBAL_THREADMGR_LOCK();
	
	desr = __pthread_find_desr(self);
	//desr is not existing 
	if(!desr) 
	{
		DP1( D_MODU_CANC,("[%d] pthread_find_desr() desr is not existing\n", self)); 
		iRet = ESRCH; 
		goto exit; 
	}
	//desr exist	
	pbuf = (struct _pthread_cleanup_buffer*)malloc(sizeof(struct _pthread_cleanup_buffer));
	if( !pbuf )
	{
		iRet = ENOMEM;
		goto exit;
	}
	pbuf->__routine = routine;
	pbuf->__arg = arg;	

	if(desr->p_cleanup)
	{
		pbuf->__prev = desr->p_cleanup->__prev;
		desr->p_cleanup->__prev = pbuf;
	}
	else
	{
		pbuf->__prev = NULL;
		desr->p_cleanup = pbuf;		
	}

exit:	
	GLOBAL_THREADMGR_UNLOCK();
	DP1( D_MODU_CANC,("[%d] pthread_cleanup_push OUT [%d]\n", self, iRet));
}

/**********************************************************************/ 
/* Function name: pthread_cleanup_pop                                 */ 
/* Description:   removes the most recently installed cleanup handler */ 
/* Return type:   null                                                */
/* Argument - int execute:                                            */
/**********************************************************************/
void _pthread_cleanup_pop(int execute)
{
	__pthread_desr* desr = NULL; 
	int iRet = 0; 	
	struct _pthread_cleanup_buffer *tempbuf, *headbuf;
	tempbuf = NULL;
	headbuf = NULL;
	
	DP1( D_MODU_CANC,("[%d] pthread_cleanup_pop IN\n", pthread_self()));
	
	GLOBAL_THREADMGR_LOCK(); 
	desr = __pthread_find_desr(pthread_self());
	//this thread is not existing or finished
	if(!desr)
	{
		iRet = ESRCH;
		DP(("[%d] pthread_cleanup_pop() desr is not existing\n", pthread_self())); 
		goto exit; 
	}
	
	//restore the queue of push
	headbuf = desr->p_cleanup;
	desr->p_cleanup = NULL;
	GLOBAL_THREADMGR_UNLOCK();
	
	//thread exist.recovery the source when push is executed
	while(headbuf)
	{
		DP1( D_MODU_CANC,("[%d] pthread_cleanup_pop is executed \n", pthread_self()));
		/* 
		 * when execute is not 0, __routine will be execute when it is poped
		 * if execute is 0,__routine is only removed but not executed.
		 */
		if(execute)
		{
			headbuf->__routine(headbuf->__arg);
		}		
		//pop the last node	
		tempbuf = headbuf;
		headbuf = headbuf->__prev;
		free(tempbuf);
	}
	headbuf = NULL;
	
exit:	
	DP1( D_MODU_CANC,("[%d] pthread_cleanup_pop OUT\n", pthread_self()));
}

/**********************************************************************/ 
/* Function name: nes_posix_pthread_cleanup_push_defer_np                          */ 
/* Description:   same with _pthread_cleanup_push                     */ 
/* Return type:   null                                              */ 
/* Argument -     void (*routine) (void *):                           */
/*                void *arg:                                          */
/**********************************************************************/
void nes_posix_pthread_cleanup_push_defer_np(void (*routine)(void*), void* arg)
{
	/*for PMC T-Engine don't support it.
	 *only offer the interface,its function is not implemented
	 */
}
	
/**********************************************************************/ 
/* Function name: nes_posix_pthread_cleanup_pop_restore_np                         */ 
/* Description:   removes the most recently installed cleanup handler */ 
/* Return type:   null                                                */
/* Argument - int execute:                                            */
/*            buffer: temptation memory                               */ 
/**********************************************************************/
void nes_posix_pthread_cleanup_pop_restore_np(int execute)
{
	/*for PMC T-Engine don't support it.
	 *only offer the interface,its function is not implemented
	 */
}
