/**
 * @file pthread_signal.c
 * @brief implement thread signal operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <tkse/errno.h>
#include <tkse/extension/proctask.h>

#include "nes_posix_signal.h"
#include "nes_posix_pthread.h"
#include "manager.h"
#include "clean.h"
#include "sigmanager.h"
#include "memory_check.h"

extern struct __sigwait_queue __g_sigwait_queue;		/* sigmanager.c */

/**********************************************************************/
/* Function name: nes_posix_pthread_kill                                        */
/* Description:                                                       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*             ESRCH  the thread thread does not exist (e.g. it has   */
/*             already termi-nated)                                   */
/* Argument - pthread_t thread: thread need to kill                   */
/* Argument - int signo: Not used here                                */
/**********************************************************************/
extern int __send_signal(pid_t pid, int signum, int data, int sender, int thread);
int nes_posix_pthread_kill(pthread_t thread, int signo)
{
	int ret = 0;
	W wCall = 0;		/* typedef int W */
	__pthread_desr* desr = NULL;

	if( thread <= 0 ) 
	{
		ret = ESRCH;
		goto exit;
	}

	if( !__signal_is_valid(signo) ) 
	{
		ret = EINVAL;
		goto exit;
	}

	/* Not permit to kill self thread */
	if( thread == pthread_self() && (signo == SIGKILL || signo == SIGSTOP) )	
	{
		ret = EINVAL;
		goto exit;
	}

	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(thread);
	if( !desr )	/* Invalid thread id */
	{
		GLOBAL_THREADMGR_UNLOCK();
		ret = ESRCH;
		goto exit;
	}

	/*
	* Modified by wang-yg @ NECSoft (JiNan) 
	* For support signal
	*/
	if( signo != SIGKILL && signo != SIGSTOP ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		__send_signal(getpid(), signo, 0, SI_USER, thread);	/* Not use signal global lock */
		ret = 0;
		goto exit;
	}
	/* modified ended */

	/* Can't kill main thread */
	if( desr->__status & STATUS_THREAD_MAIN_THREAD ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		ret = EINVAL;
		goto exit;
	}

	/* Real kill */
	wCall = tkse_ter_tsk(thread);
	ret = (wCall == E_OK ? 0 : EINVAL);

	/*
	* Cleanup thread's resource
	*/
	__pthread_destory_all_tsd(desr);		/* Destrory TSD */
	__pthread_destory_all_cleanup(desr);	/* Do cleanup */

	__internal_unlock(&(desr->__lock));	
	__pthread_del_desr(thread);
	__pthread_desr_destroy(desr);
	free(desr);
	GLOBAL_THREADMGR_UNLOCK();

exit:
	return ret;
}

/**********************************************************************/ 
/* Function name: nes_posix_pthread_sigmask                                     */ 
/* Description:   examine and change the signal mask for the calling  */
/*                thread                                              */ 
/* Return type:   0: always return 0                                  */
/* Argument - how: Specifies how the new signal mask parameter is to  */
/*                 be combined with the current signal mask; one of   */
/*                 the values specified above.                        */
/*            newmask: the new signal mask to be combined with the    */
/*                     current signal mask                            */ 
/*            oldmask: the location where the current signal mask     */
/*                     is stored                                      */
/**********************************************************************/
int nes_posix_pthread_sigmask(int how, const sigset_t *newmask, sigset_t *oldmask) 
{
	int i = 0, ret = 0;
	__pthread_desr* desr = NULL;

	/* Invalid operation */
	if( how != SIG_BLOCK && how != SIG_UNBLOCK && how != SIG_SETMASK ) 
	{
		return  EINVAL;
	}

	/* Unpermited signals */
	if( newmask ) 
	{
		if( sigismember(newmask, SIGKILL) || sigismember(newmask, SIGSTOP) ) 
		{
			GLOBAL_THREADMGR_UNLOCK();
			return EINVAL;
		}
	}

	/* Main thread ?*/
	if( pthread_self() == __MAIN_THREAD ) {
		if( sigprocmask(how, newmask, oldmask) != 0 ) 
		{
			ret = errno;
		}
		return ret;
	}

	GLOBAL_THREADMGR_LOCK();
	if( newmask ) 
	{
		desr = __pthread_find_desr(pthread_self());
		if( !desr )	/* Invalid thread id */
		{
			GLOBAL_THREADMGR_UNLOCK();
			ret = ESRCH;
			goto exit;
		}

		switch(how)
		{
		case SIG_BLOCK:		/* Add block signal */
			for(i = 0; i < _NSIG_WORDS; i++ )  desr->__block_sigmask.sig[i] |= newmask->sig[i];
			break;
		case SIG_UNBLOCK:	/* Remove block signal */
			for(i = 0; i < _NSIG_WORDS; i++ )  desr->__block_sigmask.sig[i] &= ~newmask->sig[i];
			break;
		case SIG_SETMASK:
			memcpy(&(desr->__block_sigmask), newmask, sizeof(sigset_t));
			break;
		default:
			break;
		}
	}

	__send_block_notify();

	if( oldmask ) 
	{
		memcpy(oldmask, &(desr->__block_sigmask), sizeof(sigset_t));
	}	
	GLOBAL_THREADMGR_UNLOCK();

exit:
	return ret;	
}


/**********************************************************************/ 
/* Function name: nes_posix_sigwait                                             */ 
/* Description:   selects a pending signal from set, atomically clears*/
/*                it from the system's set of pending signals, and    */
/*                returns that signal number in the location          */
/*                referenced by sig. If no signal in set is pending at*/
/*                the time of the call, the thread is suspended until */
/*                one or more becomes pending.                        */ 
/* Return type:   0: always return 0                                  */
/* Argument -     set: Specifies the set of signals to wait for.      */
/*                sig: the address into which the signal number of the*/
/*                     received signal is stored.                     */ 
/**********************************************************************/
int nes_posix_sigwait(const sigset_t*set, int *sig)
{
	int ret = 0;
	__pthread_desr* desr = NULL;
	struct __sigwait_queue* entry = NULL;

	if( !set || !sig ) 
	{
		return EINVAL;
	}

	entry = (struct __sigwait_queue*)MC_MALLOC(sizeof(struct __sigwait_queue));
	if( !entry ) 
	{
		return ENOMEM;
	}

	entry->thread = pthread_self();
	entry->type = SIGWAIT_WAIT_TYPE;
	entry->waitset = *set;

	GLOBAL_SIG_LOCK();
	QueInsert((QUEUE*)entry, (QUEUE*)(&__g_sigwait_queue));
	GLOBAL_SIG_UNLOCK();

	/* Suspend current thread */
	ret = tkse_slp_tsk(-1);

	/* If sleep is interrupted, we must be has received the signal */
	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(pthread_self());
	if( !desr ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		return ESRCH;
	}
	*sig = desr->__sigwait_signum;
	GLOBAL_THREADMGR_UNLOCK();

	return 0;
}
