/**
 * @file thread_attr.c
 * @brief implement of thread attribute get/set operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "nes_posix_pthread.h"
#include "nes_posix_error.h"
#include "manager.h"

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_init                                   */
/* Description:Initialize thread attribute *ATTR with default         */
/*attributes (detachstate is PTHREAD_JOINABLE, scheduling policy      */
/* is SCHED_OTHER,no user-provided stack).                            */
/* Return type: int   Always return 0                                 */
/* Argument - pthread_attr_t* attr:  the dest pthread_attr_t poniter  */
/**********************************************************************/
int nes_posix_pthread_attr_init(pthread_attr_t* attr)
{
	memset(attr, 0, sizeof(pthread_attr_t));

	attr->__detachstate			= PTHREAD_CREATE_JOINABLE;
	attr->__schedpolicy			= SCHED_OTHER;				/* Not use */
	attr->__schedparam.sched_priority = PRIORITY_NORMAL;				/* -1 */	
	attr->__inheritsched			= PTHREAD_EXPLICIT_SCHED;		/* Not use */
	attr->__scope				= PTHREAD_SCOPE_SYSTEM;			/* Not use */
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_destroy                                */
/* Description:Destroy thread attribute *ATTR, but NOT FREE param attr*/
/*                   In fact, we do nothing here.                     */
/* Return type: int   Always return 0                                 */
/* Argument - pthread_attr_t* attr:  Dest pthread_attr_t poniter      */
/**********************************************************************/
int nes_posix_pthread_attr_destroy(pthread_attr_t* attr)
{
	/* Here, we needn't do anything */
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setdetachstate                         */
/* Description: Set the `detachstate' attribute in *ATTR according    */
/*		to DETACHSTATE.If detachstate is unknow return EINVAL */
/* Return type: int   Return 0 if success else EINVAL                 */
/* Argument - pthread_attr_t* attr: Dest pthread_attr_t poniter       */
/* Argument - int detachstate: Thread state to set                    */
/**********************************************************************/
int nes_posix_pthread_attr_setdetachstate(pthread_attr_t* attr, int detachstate)
{
	if( detachstate != PTHREAD_CREATE_JOINABLE && 
		detachstate != PTHREAD_CREATE_DETACHED )
	{
		return EINVAL;
	}

	attr->__detachstate = detachstate;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getdetachstate                         */
/* Description:Return in *DETACHSTATE the `detachstate' attribute in  */
/*               *ATTR.                                               */
/* Return type: int    Always return 0                                */
/* Argument - const pthread_attr_t* attr:  Dest pthread_attr_t poniter*/
/* Argument - int* detachstate:  Store buffer                         */
/**********************************************************************/
int nes_posix_pthread_attr_getdetachstate(const pthread_attr_t* attr, int* detachstate)
{
	*detachstate = attr->__detachstate;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setschedparam                          */
/* Description: Set scheduling parameters(priority, etc) in *ATTR     */
/*                according to PARAM.                                 */
/* Return type: int  Return 0 if success else EINVAL                  */
/* Argument - pthread_attr_t* attr: Dest pthread_attr_t poniter       */
/* Argument - const struct sched_param* param:  sched params, now only*/
/*            support priority                                        */
/**********************************************************************/
int nes_posix_pthread_attr_setschedparam(pthread_attr_t* attr, const struct sched_param* param)
{
	/* it only matters for the realtime policies !SCHED_RR! and !SCHED_FIFO! */
	/* At linux, it can be changed on thread running, but at PMC ext, can't do this */
	if( attr->__schedpolicy == SCHED_RR || attr->__schedpolicy == SCHED_FIFO )
	{
		attr->__schedparam.sched_priority = param->sched_priority;
		return 0;
	}

	return EINVAL;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getschedparam                          */
/* Description:   Return in *PARAM the scheduling parameters of *ATTR.*/
/* Return type: int   Always return 0                                 */
/* Argument - const pthread_attr_t* attr:  Dest pthread_attr_t poniter*/
/* Argument - struct sched_param* param:  Store buffer                */
/**********************************************************************/
int nes_posix_pthread_attr_getschedparam(const pthread_attr_t* attr, struct sched_param* param)
{
	param->sched_priority = attr->__schedparam.sched_priority;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setschedpolicy                         */
/* Description: Set scheduling policy in *ATTR according to POLICY.   */
/* Return type: int    Return 0 if success else EINVAL                */
/* Argument - pthread_attr_t* attr: Dest pthread_attr_t poniter       */
/* Argument - int policy: Policy to be set                            */
/**********************************************************************/
int nes_posix_pthread_attr_setschedpolicy(pthread_attr_t* attr, int policy)
{
	if( policy != SCHED_RR && policy != SCHED_FIFO && policy != SCHED_OTHER )
	{
		return EINVAL;
	}

	attr->__schedpolicy = policy;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getschedpolicy                         */
/* Description: Return in *POLICY the scheduling policy of *ATTR.     */
/* Return type: int   Always return 0                                 */
/* Argument - const pthread_attr_t* attr:Dest pthread_attr_t poniter  */
/* Argument - int* policy: Store buffer                               */
/**********************************************************************/
int nes_posix_pthread_attr_getschedpolicy(const pthread_attr_t* attr, int* policy)
{
	*policy = attr->__schedpolicy;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setinheritsched                        */
/* Description: Set scheduling inheritance mode in *ATTR according to */
/*                INHERIT.                                            */
/* Return type: int  Return 0 if success else EINVAL                  */
/* Argument - pthread_attr_t* attr:Dest pthread_attr_t poniter        */
/* Argument - int inherit: Inherit to be set                          */
/**********************************************************************/
int nes_posix_pthread_attr_setinheritsched(pthread_attr_t* attr, int inherit) 
{
	/* PMC extention can't really do that */
	if( inherit != PTHREAD_INHERIT_SCHED && inherit != PTHREAD_EXPLICIT_SCHED ) 
	{
		return EINVAL;
	}

	attr->__inheritsched = inherit;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getinheritsched                        */
/* Description:  Return in *INHERIT the scheduling inheritance mode of*/ 
/* *ATTR.                                                             */
/* Return type: int    Always return 0                                */
/* Argument - const pthread_attr_t* attr:Dest pthread_attr_t poniter  */
/* Argument - int* inherit:  Store buffer                             */
/**********************************************************************/
int nes_posix_pthread_attr_getinheritsched(const pthread_attr_t* attr, int* inherit)
{
	*inherit = attr->__inheritsched;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setscope                               */
/* Description: Set scheduling contention scope in *ATTR according to */
/* SCOPE. And now not support PTHREAD_SCOPE_PROCESS type              */
/* Return type: On success, 0 is returned. On error, a non-zero error code is returned. */
/*                   EPERM: Not support PTHREAD_SCOPE_PROCESS */
/*                   EINVAL: invalid type */
/* Argument - pthread_attr_t* attr:Dest pthread_attr_t poniter        */
/* Argument - int scope: Scope to be set                              */
/**********************************************************************/
int nes_posix_pthread_attr_setscope(pthread_attr_t* attr, int scope)
{
	switch (scope) 
	{
	case PTHREAD_SCOPE_SYSTEM:
		attr->__scope = scope;
		return 0;
	case PTHREAD_SCOPE_PROCESS:
		return EPERM;
	default:
		return EINVAL;
	}
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getscope                               */
/* Description:  Return in *SCOPE the scheduling contention scope of  */
/*  *ATTR.                                                            */
/* Return type: int   Always return 0                                 */
/* Argument - const pthread_attr_t* attr:Dest pthread_attr_t poniter  */
/* Argument - int* scope: Store buffer                                */
/**********************************************************************/
int nes_posix_pthread_attr_getscope(const pthread_attr_t* attr, int* scope)
{
	*scope = attr->__scope;
	return 0;
}


/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setstack                               */
/* Description: The following two interfaces are intended to replace  */
/*              the last two.                                         */
/* Return type: int  Always return 0                                  */
/* Argument - pthread_attr_t* attr:Dest pthread_attr_t poniter        */
/* Argument - void* stackaddr: Stackaddr to be set                    */
/* Argument - size_t stacksize: Stack szie to be set                  */
/**********************************************************************/
int nes_posix_pthread_attr_setstack(pthread_attr_t* attr, void* stackaddr, size_t stacksize)
{
	attr->__stackaddr = stackaddr;
	attr->__stacksize = stacksize;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getstack                               */
/* Description:  Return the previously set address for the stack.     */
/* Return type: int  Always return 0                                  */
/* Argument - const pthread_attr_t* attr:Dest pthread_attr_t poniter  */
/* Argument - void** stackaddr: Store buffer                          */
/* Argument - size_t* stacksize:  Store buffer                        */
/**********************************************************************/
int nes_posix_pthread_attr_getstack(const pthread_attr_t* attr, void** stackaddr, size_t* stacksize)
{
	*stackaddr	= attr->__stackaddr;
	*stacksize	= attr->__stacksize;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setstackaddr                           */
/* Description:  Set the starting address of the stack of the thread  */
/*               to be created.                                       */
/* Return type: int   Always return 0                                 */
/* Argument - pthread_attr_t* attr:Dest pthread_attr_t poniter        */
/* Argument - void* stackaddr: stackaddr to be set                    */
/**********************************************************************/
int nes_posix_pthread_attr_setstackaddr(pthread_attr_t* attr, void* stackaddr) 
{
	attr->__stackaddr = stackaddr;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getstackaddr                           */
/* Description:    Return the previously set address for the stack.   */
/* Return type: int  Always return 0                                  */
/* Argument - const pthread_attr_t* attr:Dest pthread_attr_t poniter  */
/* Argument - void** stackaddr: Store buffer                          */
/**********************************************************************/
int nes_posix_pthread_attr_getstackaddr(const pthread_attr_t* attr, void** stackaddr)
{
	*stackaddr = attr->__stackaddr;
	return 0;
}


/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setstacksize                           */
/* Description:  Add information about the minimum stack size needed  */
/* for the thread to be started.                                      */
/* Return type: int                                                   */
/* Argument - pthread_attr_t* attr: Dest pthread_attr_t poniter       */
/* Argument - size_t stacksize: Stack size to be set                  */
/**********************************************************************/
int nes_posix_pthread_attr_setstacksize(pthread_attr_t* attr, size_t stacksize) 
{
	attr->__stacksize = stacksize;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getstacksize                           */
/* Description:   Return the currently used minimal stack size.       */
/* Return type: int    Always return 0                                */
/* Argument - const pthread_attr_t* attr:Dest pthread_attr_t poniter  */
/* Argument - size_t* stacksize: Store buffer                         */
/**********************************************************************/
int nes_posix_pthread_attr_getstacksize(const pthread_attr_t* attr, size_t* stacksize)
{
	*stacksize = attr->__stacksize;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_setguardsize                           */
/* Description: Set the size of the guard area at the bottom of the   */
/*                thread.                                             */
/* Return type: int  Always return 0                                  */
/* Argument - pthread_attr_t* attr: Dest pthread_attr_t poniter       */
/* Argument - size_t guardsize: guard size to be set                  */
/**********************************************************************/
int nes_posix_pthread_attr_setguardsize(pthread_attr_t* attr, size_t guardsize)
{
	attr->__guardsize = guardsize;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_attr_getguardsize                 */
/* Description: Get the size of the guard area at the bottom of the   */
/*              thread.                                               */
/* Return type: int Always return 0                                   */
/* Argument - const pthread_attr_t* attr: Dest pthread_attr_t poniter */
/* Argument - size_t* guardsize: Store buffer                         */
/**********************************************************************/
int nes_posix_pthread_attr_getguardsize(const pthread_attr_t* attr, size_t* guardsize)
{
	*guardsize = attr->__guardsize;
	return 0;
}

/**********************************************************************/
/* Function name: nes_posix_pthread_getattr_np                                  */
/* Description: Initialize thread attribute *ATTR with attributes     */
/*              corresponding to the already running thread TH.       */
/*              It shall be called on unitialized ATTR                */
/* and destroyed with pthread_attr_destroy when no longer needed.     */
/* Return type: int Return 0 if success else return EINVAL            */
/* Argument - pthread_t thread:  dest thread id                       */
/* Argument - pthread_attr_t *attr:  Dest pthread_attr_t poniter      */
/**********************************************************************/
int nes_posix_pthread_getattr_np(pthread_t thread, pthread_attr_t* attr)
{
	__pthread_desr* desr = NULL;

	if( !attr ) 
	{
		return EINVAL;
	}

	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(thread);

	if( !desr ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		return EINVAL;		/* Incorrect thread id */
	}

	/* Just copy all data */
	memcpy(attr, &desr->__attr, sizeof(pthread_attr_t));
	GLOBAL_THREADMGR_UNLOCK();

	return 0;
}

