/**
 * @file tsd.c
 * @brief implement of TSD function 
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "tsd.h"
#include "nes_posix_pthread.h"
#include "manager.h"

typedef void (*destr_function)(void *);

struct __pthread_key_info {
	int use;			 /* already allocated? */
	destr_function destr;		 /* destruction routine */
};

/* Global lock use to tsd protected */
pthread_mutex_t __g_tsdlock = {0};
struct __pthread_key_info __g_pthread_keys[PTHREAD_KEYS_MAX];

/**********************************************************************/
/* Function name: __tsd_init                                          */
/* Description: Init TSD module                                       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __tsd_init(void)
{
	if( __g_tsdlock.__m_lock != 0 ) { /* Init already */
		return 0;
	}

	int i = 0;
	for( i = 0; i < PTHREAD_KEYS_MAX; i++ ) 
	{
		__g_pthread_keys[i].use	= 0;
		__g_pthread_keys[i].destr	= NULL;
	}

	return nes_posix_pthread_mutex_init(&__g_tsdlock, NULL);;
}

/**********************************************************************/
/* Function name: __tsd_finish                                        */
/* Description: Clean TSD module                                      */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __tsd_finish(void)
{
	return nes_posix_pthread_mutex_destroy(&__g_tsdlock);;
}

/**********************************************************************/
/* Function name: __tsd_lock                                          */
/* Description: Lock TSD module                                       */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __tsd_lock(void)
{
	return nes_posix_pthread_mutex_lock(&__g_tsdlock);
}

/**********************************************************************/
/* Function name: __tsd_unlock                                        */
/* Description: Unlock TSD module                                     */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/* Argument - void:                                                   */
/**********************************************************************/
int __tsd_unlock(void)
{
	return nes_posix_pthread_mutex_unlock(&__g_tsdlock);
}

/**********************************************************************/
/* Function name: pthread_key_create                                  */
/* Description: Create a key value identifying a location in the      */
/* thread-specific data area.  Each thread maintains a distinct       */
/* thread-specific data area.  DESTR_FUNCTION, if non-NULL, is called */
/* with the value associated to that key when the key is destroyed.   */
/* DESTR_FUNCTION is not called if the value associated is NULL when  */
/* the key is destroyed.                                              */
/*                                                                    */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*             EAGAIN: no key now, retry please                       */
/* Argument - pthread_key_t*__key:    Data buffer                     */
/* Argument - void (*__destr_function: Release function               */
/**********************************************************************/
int nes_posix_pthread_key_create(pthread_key_t*__key, void (*__destr_function) (void*))
{
	int i = 0;
	GLOBAL_TSD_LOCK();
	for( i = 0; i < PTHREAD_KEYS_MAX; i++ ) 
	{
		/* Found one unused */
		if( !__g_pthread_keys[i].use ) 
		{	
			__g_pthread_keys[i].use	= 1;
			__g_pthread_keys[i].destr	= __destr_function;
			*__key = i;
			GLOBAL_TSD_UNLOCK();
			return 0;
		}
	}

	GLOBAL_TSD_UNLOCK();
	
	/* Retry please */
	return EAGAIN;	
}


/**********************************************************************/
/* Function name: pthread_key_delete                                  */
/* Description: Destroy KEY                                           */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*              EINVAL: key is invalid                                 */
/* Argument - pthread_key_t __key:                                    */
/**********************************************************************/
int nes_posix_pthread_key_delete(pthread_key_t __key)
{
	/* Check TSD key */
	if( __key < 0 || __key >= PTHREAD_KEYS_MAX ) 
	{
		return EINVAL;
	}
	
	GLOBAL_TSD_LOCK();

	if( !__g_pthread_keys[__key].use ) 	/* Key not allocte now, must be invalid */
	{
		GLOBAL_TSD_UNLOCK();
		return EINVAL;
	} 

	__g_pthread_keys[__key].use = 0;
	__g_pthread_keys[__key].destr = NULL;
	GLOBAL_TSD_UNLOCK();

	return 0;
}

/**********************************************************************/
/* Function name: pthread_setspecific                                 */
/* Description: Store POINTER in the thread-specific data slot        */
/*              identified by KEY.                                    */
/* Return type: On success, 0 is returned. On error, a non-zero error */
/*              code is returned.                                     */
/*              EINVAL: key is invalid                                */
/*              ENOMEM: memory is not enough                          */
/* Argument - const void* __pointer:                                  */
/**********************************************************************/
int nes_posix_pthread_setspecific(pthread_key_t __key, const void* __pointer)
{
	int ret = 0;
	int x, y;
	void* newrow = NULL;
	__pthread_desr* desr = NULL;

	/* Check TSD key */
	if( __key < 0 || __key >= PTHREAD_KEYS_MAX ) 
	{
		ret = EINVAL;
		goto exit;
	}

	/* Key not allocte now, must be invalid */
/*	GLOBAL_TSD_LOCK();
	if( !__g_pthread_keys[__key].use ) {	
		GLOBAL_TSD_UNLOCK();
		ret = EINVAL;
		goto exit;
	} 
	GLOBAL_TSD_UNLOCK();
*/
	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(nes_posix_pthread_self());
	if( !desr ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		ret = EINVAL;
		goto exit;
	}
	
	x = __key / PTHREAD_KEY_2NDLEVEL_SIZE;	/* Which row ? */
	y = __key % PTHREAD_KEY_2NDLEVEL_SIZE;	/* Which colum ? */

	/* 
	 * The space is a vector, PTHREAD_KEY_1STLEVEL_SIZE * PTHREAD_KEY_2NDLEVEL_SIZE
	 * If no space to store key data, means the total key must be 16*n
	 * So, mallc a new row (PTHREAD_KEY_2NDLEVEL_SIZE keys) to store new key date.
	 *
	 * 0	1	2	3	4	...	PTHREAD_KEY_2NDLEVEL_SIZE
	 * 1	x	x	x	x		
	 * 2	x	x	x	x	
	 * 3	x	x	x	x
	 * 4	x	x	x	x
	 * .
	 * .
	 * PTHREAD_KEY_1STLEVEL_SIZE
	 */

	if( !desr->__tsd_buf[x] ) 
	{
		newrow = (void*)calloc(PTHREAD_KEY_2NDLEVEL_SIZE, sizeof(void*));
		if( !newrow ) 
		{
			GLOBAL_THREADMGR_UNLOCK();
			ret = ENOMEM;
			goto exit;
		}
		memset(newrow, 0, PTHREAD_KEY_2NDLEVEL_SIZE*sizeof(void*));
		desr->__tsd_buf[x] = newrow;
	}

	/* We find the space */
	desr->__tsd_buf[x][y] = (void*)__pointer;
	ret = 0;

	GLOBAL_THREADMGR_UNLOCK();

exit:
	return ret;
}

/**********************************************************************/
/* Function name: *pthread_getspecific                                */
/* Description: Return current value of the thread-specific data slot */
/*             identified by KEY.                                     */
/* Return type: void*                                                 */
/*              if not found or not set NULL                          */
/*              if success return Not zero value                      */
/* Argument - pthread_key_t __key:                                    */
/**********************************************************************/
void* nes_posix_pthread_getspecific(pthread_key_t __key)
{
	int x, y;
	void* return_value = NULL;
	__pthread_desr* desr = NULL;

	/* Check TSD key */
	if( __key < 0 || __key >= PTHREAD_KEYS_MAX ) 
	{
		return_value = NULL;
		goto exit;
	}

	GLOBAL_THREADMGR_LOCK();
	desr = __pthread_find_desr(nes_posix_pthread_self());
	if( !desr ) 
	{
		GLOBAL_THREADMGR_UNLOCK();
		return_value = NULL;
		goto exit;
	}

	x = __key / PTHREAD_KEY_2NDLEVEL_SIZE;	/* Which row ? */
	y = __key % PTHREAD_KEY_2NDLEVEL_SIZE;	/* Which colum ? */
	if( !desr->__tsd_buf[x]  )				/* Not setspecific */
	{
		GLOBAL_THREADMGR_UNLOCK();
		return_value = NULL;
		goto exit;
	}
	
	return_value = desr->__tsd_buf[x][y];		/* Get the value */
	GLOBAL_THREADMGR_UNLOCK();
	
exit:
	return return_value;
}

/**********************************************************************/
/* Function name: __pthread_destory_all_tsd                           */
/* Description: Destory all tsd data and call the release function    */
/* Return type: int Always return 0                                   */
/* Argument - void* __desr:                                           */
/**********************************************************************/
int __pthread_destory_all_tsd(void* __desr /*__pthread_desr */)
{
	/*
	 * Work without GLOBAL_THREADMGR_LOCK
	 */

	int i, j;
	void * data;
	destr_function destr;
	__pthread_desr* desr = NULL;
	desr = (__pthread_desr*)__desr;

	 /* Access every row */
	for( i = 0; i < PTHREAD_KEY_1STLEVEL_SIZE; i++ ) 
	{
		if( !desr->__tsd_buf[i] ) 
		{
			continue;
		}

		/* Access every column */
		for( j = 0; j < PTHREAD_KEY_2NDLEVEL_SIZE; j++ ) 
		{
			destr = __g_pthread_keys[i * PTHREAD_KEY_2NDLEVEL_SIZE + j].destr;
			data = desr->__tsd_buf[i][j];
			if( destr && data ) 
			{
				destr(data);		/* Excute release function at the end */
			}
		} 

		/* Release resource */
		free(desr->__tsd_buf[i]);
		desr->__tsd_buf[i] = NULL;
	}

	return 0;
}

