/**
 * @file queue.c
 * @brief This file is part of the posix library
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "queue.h"

/**********************************************************************/
/* Function name: queue_init                                          */
/* Description:   This function init a queue                          */
/* Return type:   no return value                                     */
/* Argument - myroot: pointer to an instance of queue                 */
/**********************************************************************/
void queue_init(struct queue *myroot) 
{
	if( !myroot )
	{
		return;
	}
	myroot->head = NULL;
	myroot->tail = NULL;	
}
/**********************************************************************/
/* Function name: queue_push                                          */
/* Description:   This function push a node into the queue            */
/* Return type:   no return value                                     */
/* Argument - myroot: pointer to an instance of queue                 */
/*            mynode: pointer to an instance of node                  */
/**********************************************************************/
void queue_push(struct queue *myroot,node *mynode) 
{
	mynode->next=NULL;
	
	if (myroot->tail != NULL)
		myroot->tail->next=mynode;
	
	myroot->tail = mynode;
	
	if (myroot->head==NULL)
		myroot->head=mynode;
}

/**********************************************************************/
/* Function name: queue_find                                          */
/* Description:   This function search the data is existing or not in */
/*                the queue                                           */
/* Return type:   -1 :  don't exist                                   */
/*                tempnode->iptn : the iptn of existing node          */
/* Argument - myroot: pointer to an instance of queue                 */
/*            tskid:  phtread ID                                      */
/**********************************************************************/

int queue_find(struct queue *myroot,int tskid) 
{
	node *tempnode = NULL ;
	
	if( !myroot ) 
	{
		return -1;
	}
	
	tempnode = myroot->head;

	while( tempnode )
	{
		if( tempnode->itid == tskid)
		{	
			return tempnode->iptn;
		}
		tempnode = tempnode->next;
	}
	
	return -1;
	
}
/**********************************************************************/
/* Function name: queue_search_del                                    */
/* Description:   This function search the data is existing or not in */
/*                the queue,if find it,then delete it                 */
/* Return type:   0 : destroy succesfully                             */
/*                -1 : destroy Failed                                 */
/* Argument - myroot: pointer to an instance of queue                 */
/*            tskid:  phtread ID                                      */
/**********************************************************************/
int queue_search_del(struct queue *myroot, int tskid)
{
	node* temp = NULL;
	node* parent = NULL; 

	if( !myroot ) 
	{
		return -1;
	}

	if (!myroot->head)
	{
		return 0;
	}

	//the node is the head node of the queue and delet it
	if( myroot->head->itid == tskid )
	{
		temp = myroot->head;
		if( myroot->tail == myroot->head ) {	/* last node */
			myroot->tail = NULL;
		}		

		myroot->head = myroot->head->next;
		free(temp);
		return 0;
	}

	parent = myroot->head;
	while(parent->next)
	{
		//find the node in the queue and delet it
		if( parent->next->itid == tskid)
		{
			temp = parent->next;
			if( parent->next == myroot->tail ) {	/* last node */
				myroot->tail = parent;
			}			
			parent->next = parent->next->next;
			free(temp);
			return 0;
		}
		parent = parent->next;
	}
	return -1;
}

/**********************************************************************/
/* Function name: queue_destroy                                       */
/* Description:   This function delte all the node of queue without   */
/*                destroy the queue                                   */
/*                the queue                                           */
/* Return type:   0 : destroy succesfully                             */
/* Argument - myroot: pointer to an instance of queue                 */
/**********************************************************************/
int queue_destroy(struct queue *myroot)
{
	node *tempnode;

	if ( !myroot )
	{
		return 0;
	}

	if (!myroot->head)
	{
		return 0;
	}
	
	while (myroot->head->next != NULL)
	{	
		tempnode = myroot->head;
		myroot->head = myroot->head->next;
		free(tempnode);
	}
	
	free(myroot->head);
	myroot->head = NULL;
	myroot->tail = NULL;
	return 0;
	
}


/**********************************************************************/
/* Function name: queue_pop                                           */
/* Description:   This function pop the head node the queue           */
/* Return type:   mynode : head node                                  */
/* Argument - myroot: pointer to an instance of queue                 */
/**********************************************************************/ 
node* queue_pop(struct queue *myroot) 
{
	node *mynode;	
        
        if(!myroot)
        {
        	return 0;	
        }
	mynode = myroot->head;

	if ( mynode != NULL) {
		myroot->head = mynode->next;
	}

	/* Last node is pop */
	if( myroot->head == NULL ) {
		myroot->tail = NULL;
	}
	
	return mynode;
}
