/**
 * @file nes_posix_sched.c
 * @brief implement of process operation
 *
 * Copyright 2011 NEC Soft, Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <btron/proctask.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include "nes_posix_pthread.h"
#include "nes_posix_proc.h"
#include "nes_posix_sched.h"


/**********************************************************************/ 
/* Function name: nes_posix_sched_get_priority_max                    */ 
/* Description:   this syscall returns the maximum rt_priority that   */  
/*                can be used by a given scheduling class.            */
/* Return value:  On sucessfully:  maximum rt_priority                */
/*                erro: -1                                            */
/*                      ENOBUFS:no buffer space avaiable              */
/*                      EINVAL: The value of the policy parameter does*/
/*                              not represent a defined scheduling    */
/*                              policy.                               */
/* Argument -     policy: scheduling class                            */
/**********************************************************************/ 
int nes_posix_sched_get_priority_max (int policy)
{
	P_STATE buff;
	WERR wRet = 0;

	//parameter valid or not
	if( (policy != SCHED_FIFO) && (policy != SCHED_RR) && (policy != SCHED_OTHER))
	{
		wRet = -1;
		errno = EINVAL;
		return wRet;	
	}

	//get the process priority
	wRet = b_prc_sts(0, &buff, NULL);
	if(wRet <= 0)
	{
		wRet = -1;
		errno = EINVAL;
	}
	if((buff.priority <= 127) &&(buff.priority >= 0))
	{
		wRet = 127;
	}
	else if((buff.priority <= 191) &&(buff.priority >= 128))
	{
		wRet = 191;
	}
	else if((buff.priority <= 255) &&(buff.priority >= 192))
	{
		wRet = 255;
	}
	else
	{
		wRet = -1;
		errno = EINVAL;
	}
	return wRet;	
}

/**********************************************************************/ 
/* Function name: nes_posix_sched_get_priority_min                    */ 
/* Description:   the function will get the appropriate minium values */
/*                for the scheduling policy specified by policy.      */
/* Return value:  On sucessfully: minium values                       */
/*                erro: -1                                            */
/*                      ENOBUFS:no buffer space avaiable              */
/*                      EINVAL: The value of the policy parameter does*/
/*                              not represent a defined scheduling    */
/*                              policy.                               */
/* Argument -     policy: scheduling policies                         */
/**********************************************************************/ 
int nes_posix_sched_get_priority_min (int policy)
{
	P_STATE buff;
	WERR wRet = 0;

	//parameter valid or not
	if( (policy != SCHED_FIFO) && (policy != SCHED_RR) && (policy != SCHED_OTHER))
	{
		wRet = -1;
		errno = EINVAL;
		return wRet;	
	}

	//get the process priority
	wRet = b_prc_sts(0, &buff, NULL);
	if(wRet <= 0)
	{
		wRet = -1;
		errno = EINVAL;
	}
	if((buff.priority <= 127) &&(buff.priority > 0))
	{
		wRet = 1;
	}
	else if((buff.priority <= 191) &&(buff.priority >= 128))
	{
		wRet = 128;
	}
	else if((buff.priority <= 255) &&(buff.priority >= 192))
	{
		wRet = 192;
	}
	else
	{
		wRet = -1;
		errno = EINVAL;					
	}

	return wRet;	
}


/**********************************************************************/ 
/* Function name: nes_posix_sched_rr_get_interval                               */ 
/* Description:   the function writes the default timeslice value of a*/
/*                given process                                       */
/* Return value:  On sucessfully: 0                                   */
/*                erro: -1                                            */
/*                      ENOBUFS: no buffer space avaiable             */
/*                      ESRCH : No process can be found corresponding */
/*                              to that specified by pid.             */
/*                      EACCES: permition denied                      */
/*                      EINVAL: other erro                            */
/* Argument -     pid: pid of the process.                            */
/*                tp : userspace pointer to the timeslice value.      */
/**********************************************************************/ 
int nes_posix_sched_rr_get_interval(pid_t pid, struct timespec *tp)
{
	ERR eRet = 0;
	P_INFO buff;
	int temptime = 0;

	//parameter jugement
	if(!tp)
	{
		eRet = -1;
		errno = EINVAL;
		goto exit;	    
	}    
	//get process' information
	eRet = b_get_inf(pid, &buff);
	if(eRet != 0)
	{
		switch(eRet)
		{
			case ER_NOPRC:
				/*
				 * no process can be found corresponding
				 * to that specified by pid.
				 */
				eRet = -1;
				errno = ESRCH;
				goto exit;
				break;
			case ER_ADR : 
				//permission denied
				eRet = -1;
				errno = EACCES;
				goto exit;
				break;
			default:
				eRet = -1;
				errno = EINVAL;
				goto exit;
				break;
		}
	}

	//get the cpu time of the process used
	temptime = buff.utime; 
	tp->tv_nsec = (temptime%1000)*1000000;
	tp->tv_sec = temptime/1000;		


exit:
	return eRet;	
}

/**********************************************************************/ 
/* Function name: nes_posix_sched_yield                               */ 
/* Description:   the function this function yields the current CPU by*/
/*                moving the calling thread to the expired array. If  */
/*                there are no other threads running on this CPU then */
/*                this function will return.                          */
/* Return value:  On sucessfully: 0                                   */
/*                erro: -1                                            */
/*                      EINTR: other erro                             */
/* Argument -     void:                                               */
/**********************************************************************/ 
int nes_posix_sched_yield(void)
{
	ERR eRet = 0;

	eRet = b_dly_tsk(0);
	if(eRet != 0)
	{
		eRet = -1;
		errno = EINTR;
	}
	return eRet;	
}
