/*
 * Copyright (c) 2008, AIST.
 * All rights reserved. This program is made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors:
 * National Institute of Advanced Industrial Science and Technology (AIST)
 */
/*
 * socket.c
 *
 *      Copyright(c) Isao Hara, 2006
 *
 * $Revision: 1.3 $
 * $Date: 2008/05/29 04:54:56 $
 * $Id: socket.c,v 1.3 2008/05/29 04:54:56 yoshi Exp $
 *
 */
#ifdef __T_KERNEL__
#include <nes_posix_time.h>
#else
#include <math.h>
#endif
#include <RtORB/corba.h>
#include <RtORB/sockport.h>

struct sockport_profile  SockProfile[FD_SETSIZE];
static  fd_set main_socket_bits;
#ifdef USE_THREAD
#ifdef __T_KERNEL__
pthread_mutex_t SOCK_MUTEX = {0};
#else
static pthread_mutex_t SOCK_MUTEX = PTHREAD_MUTEX_INITIALIZER;
#endif
#define SOCKET_LOCK() pthread_mutex_lock(&SOCK_MUTEX)
#define SOCKET_UNLOCK() pthread_mutex_unlock(&SOCK_MUTEX)
#else
#define SOCKET_LOCK() 
#define SOCKET_UNLOCK() 
#endif

/*
 *   Server Side
 */
int make_server_socket_port(int port_no) {
  int server_sock;
  struct sockaddr_in *server_addr;
  int reuse = 1;

  server_addr=(struct sockaddr_in *)RtORB_alloc(sizeof(struct sockaddr_in),
		  "make_server_socket_port");
  memset(server_addr, 0, sizeof (*server_addr));
  server_addr->sin_family      = AF_INET;
  server_addr->sin_port        = htons(port_no);
  server_addr->sin_addr.s_addr = htonl(INADDR_ANY);

  server_sock = socket (AF_INET, SOCK_STREAM, 0);

  if (server_sock < 0) {
	  fprintf(stderr, "Fail to create a socket\n");
	  free(server_addr);
	  return -1;
  }

  setsockopt(server_sock, SOL_SOCKET, SO_REUSEADDR, (char *)&reuse, sizeof(reuse));

  if (bind(server_sock, (struct sockaddr *)server_addr, sizeof(*server_addr)) < 0){
    fprintf (stderr, "Fail to bind a socket\n");
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    closesocket(server_sock);
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    close(server_sock);
#endif
    return -1;
  }

  free(server_addr);

  if (listen (server_sock, 2) < 0 ){
      fprintf (stderr, "listen failed\n");
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
      closesocket(server_sock);
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
      close(server_sock);
#endif
      return -1;
  }
  return server_sock;
}

long accept_connection(long port_no) {
  struct sockaddr_in  client_addr;
  int   client_len, snew, opt_len, stat;
  long  opt;

  client_len = sizeof(struct sockaddr_in);
  snew = accept(port_no, (struct sockaddr *)&client_addr, (socklen_t*)&client_len);
 
  if (snew>0) {
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    stat=getsockopt(snew, 6, SO_REUSEADDR, (char*)&opt, (socklen_t*)&opt_len);
    opt = 1;
    stat=setsockopt(snew, 6, SO_REUSEADDR, (char*)&opt, sizeof(opt));
    stat=getsockopt(snew, 6, SO_REUSEADDR, (char*)&opt, (socklen_t*)&opt_len);
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    stat=getsockopt(snew, 6, SO_REUSEADDR, &opt, (socklen_t*)&opt_len);
    opt = 1;
    stat=setsockopt(snew, 6, SO_REUSEADDR, &opt, sizeof(opt));
    stat=getsockopt(snew, 6, SO_REUSEADDR, &opt, (socklen_t*)&opt_len);
#endif
    }
  return(snew);
}

int copy_fds_set(fd_set *target, fd_set *src){
  int i,j,k;
  int	 nfds = 1;

  for(k= FD_SETSIZE / NFDBITS ; k >0 ;k--){
#if CYGWIN
    target->fds_bits[k-1] |= src->fds_bits[k-1];
    if (target->fds_bits[k-1]) break;
#elif __cplusplus
    target->fds_bits[k-1] |= src->fds_bits[k-1];
    if (target->fds_bits[k-1]) break;
#elif __T_KERNEL__
#ifdef DEBUG_T_KERNEL
//DEL    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    target->fds_bits[k-1] |= src->fds_bits[k-1];
    if (target->fds_bits[k-1]) break;
#ifdef DEBUG_T_KERNEL
//DEL    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    target->__fds_bits[k-1] |= src->__fds_bits[k-1];
    if (target->__fds_bits[k-1]) break;
#endif
  }

  if (k) {
#if CYGWIN
    j=target->fds_bits[k-1];
#elif __cplusplus
    j=target->fds_bits[k-1];
#elif __T_KERNEL__
#ifdef DEBUG_T_KERNEL
//DEL    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    j=target->fds_bits[k-1];
#ifdef DEBUG_T_KERNEL
//DEL    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    j=target->__fds_bits[k-1];
#endif
    for (i=0; i < NFDBITS; i++, j = j >> 1 ) { if (j & 1) nfds=i+1; }
    nfds += NFDBITS * (k-1);
  }
  for(; k >0 ;k--){
#if CYGWIN
    target->fds_bits[k-1] |= src->fds_bits[k-1];
#elif __cplusplus
    target->fds_bits[k-1] |= src->fds_bits[k-1];
#elif __T_KERNEL__
#ifdef DEBUG_T_KERNEL
//DEL    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
    target->fds_bits[k-1] |= src->fds_bits[k-1];
#ifdef DEBUG_T_KERNEL
//DEL    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
    target->__fds_bits[k-1] |= src->__fds_bits[k-1];
#endif
  }

  return nfds;
}

int select_server_sockets(fd_set *sockbits, int server_port, 
	struct timeval time_out,
	void (*connection_request)(int),
	int  (*command_request)(GIOP_ConnectionHandler *))
{
  int	 nfds;
  fd_set socks;
  int	 stat, i = 0, newsock;

  FD_ZERO(&socks);
  nfds = copy_fds_set(&socks, sockbits);

  stat=select(nfds, &socks, 0, 0 /*&exceptions*/ , &time_out);
  if (stat>0) {
    if (FD_ISSET(server_port, &socks)) { 
	    fprintf(stderr, "New Connection\n");
      newsock= accept_connection(server_port); /* new connection */
      if (newsock>0) FD_SET(newsock, sockbits);
      FD_CLR(server_port, &socks);
      if (connection_request) (*connection_request)(newsock);
      stat--;
    } else {
      i=0;
      while (stat > 0 && i < nfds) {
        if (FD_ISSET(i, &socks)) {
			GIOP_ConnectionHandler h;
          h.sock = i;
          h.type = CONNECTION_TYPE_SOCKET;
	    fprintf(stderr, "Request!! (%d)\n", i);
          if((*command_request)(&h) == -1) FD_CLR(i, sockbits);
          FD_CLR(i, &socks);
          stat--;
        }
      i++; 
      }
    }
  }
  return i; 
}

void server_socket_loop(int server_port, float time_out_float, void (*idle)(),
  void (*connection_request)(int), 
  int (*command_request)(GIOP_ConnectionHandler *))
{
  fd_set sockbits;
  struct timeval time_out;
  struct timezone tz;
  struct timeval time1,time2;
  long time_cost;

  FD_ZERO(&sockbits);
  FD_SET(server_port, &sockbits);

  time_out.tv_sec = (unsigned long)(time_out_float / 1000.0);
  time_out.tv_usec = (time_out_float - time_out.tv_sec * 1000) * 1000 ;

//  fprintf(stderr, "--time_out %d, %d\n", time_out.tv_sec, time_out.tv_usec);
  while (1) {
      gettimeofday(&time1, &tz); 
      select_server_sockets(&sockbits, server_port, time_out,
		connection_request, command_request);

      if (idle) (*idle)(); 
      gettimeofday(&time2, &tz); 

#if 1
      time_cost = (time2.tv_sec - time1.tv_sec) * 1000000 + time2.tv_usec
	      - time1.tv_usec - time_out_float * 1000 ;

//      if (time_cost > 0) usleep(time_cost);
//      else usleep(1);
#endif
  //    usleep(1);
  }
}

/**
 *
 *   find a hostname and a port number from a socket descriptor
 *
 */

short get_socket_port(int sock){
  int len;
  struct sockaddr_in addr;

  len = sizeof(struct sockaddr_in);

  if (getsockname(sock, (struct sockaddr *)&addr, (socklen_t*)&len) < 0) return -1;
  return  ntohs(addr.sin_port);
}

char *get_ip_address(int sock){
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
  struct sockaddr_in addr;
  int len = sizeof(addr);
  int ret = getsockname(sock, (SOCKADDR*)&addr, &len);
  if( ret < 0) {
    return (char *)NULL;
  }

  char *ip_addr = (char *)inet_ntoa(addr.sin_addr);
#ifdef DEBUG_T_KERNEL
    printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
  return RtORB_strdup(ip_addr, "get_ip_address");
#else
  struct sockaddr_in *addr=0;
  struct ifreq ifreqs[10];
  struct ifconf ifconf;
  int i, num;
  
  ifconf.ifc_len = sizeof(ifreqs);
  ifconf.ifc_req = ifreqs;

  if(ioctl(sock, SIOCGIFCONF, &ifconf) < 0){
    return (char *)NULL;
  }
  num = ifconf.ifc_len / sizeof(struct ifreq);

  for(i=0; i< num ; i++){
    addr = (struct sockaddr_in *)&ifreqs[i].ifr_ifru.ifru_addr;
    char *ip_addr = (char *)inet_ntoa(addr->sin_addr);

#if DEBUG1
    fprintf(stderr, "Host = %s \n", ip_addr);
#endif

    if(strcmp(ip_addr,"127.0.0.1") &&
//       strncmp(ip_addr,"192.168.0", 9) &&
//       strncmp(ip_addr,"192.168.1", 9) &&
       strcmp(ip_addr,"0.0.0.0") ) return RtORB_strdup(ip_addr, "get_ip_address");
  }
  return (char *)NULL;
#endif
}

/*
 *
 * Client side
 *
 **/
int make_client_socket_port(char *hostname, int port)
{
	struct sockaddr_in sock_addr;
	struct hostent *hp;
	int fd;

	if(!hostname) return -1;
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
        printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
//DEL	sock_addr.sin_addr.s_addr = inet_addr(hostname);

	//ͤȥɥåȵˡȽ
	int  c, n, v ;
	char *p;
	for(p = hostname, n = v = 0; (c = *p++) != 0;)
	{
		if (c >= '0' && c <= '9')
		{
			if ((v = v * 10 + (c - '0')) > 255) break;
		} else if (c == '.' && n < 3) {
			n++;
			v=0;
		} else break;
	}

	if(c == 0 && n == 3)
	{
		sock_addr.sin_addr.s_addr = inet_addr(hostname);
	} else {
		hp = gethostbyname(hostname);
		if(hp == NULL)
		{
			fprintf(stderr,"host %s not valid\r\n", hostname);
			return -1;
		}
		sock_addr.sin_addr.s_addr = **(unsigned int **)(hp->h_addr_list);
	}

#else
	if((hp = gethostbyname(hostname)) == NULL){
		fprintf(stderr,"host %s not valid\r\n", hostname);
		return -1;
	}

        memcpy((char *) &(sock_addr.sin_addr), hp->h_addr, hp->h_length); 
#endif
	sock_addr.sin_family = AF_INET;
	sock_addr.sin_port = htons(port);

	/* create socket */
	if ((fd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
//		 perror("socket");
		 return(-1);
	}

	if(connect(fd, (struct sockaddr *)&sock_addr, sizeof(sock_addr)) < 0){
//		perror("connect");
		return(-1);
	}
	
	return fd;
}

#if 0 /* no use */
int select_client_socket( int client_port, int timeout,
		void (*exception_request)(), int (*command_request)())
{
  int	 nfds;
  fd_set socks;
  int	 stat;
  struct timeval time_out;

  //time_out.tv_sec = (int)floor(timeout / 1000);
  time_out.tv_sec = timeout / 1000;
  time_out.tv_usec = (timeout - time_out.tv_sec * 1000) * 1000 ;

  FD_ZERO(&socks);
  FD_SET(client_port, &socks);

  nfds = client_port+1;
  stat=select(nfds, &socks, 0, 0 /*&exceptions*/, &time_out);

  if (stat>0) {
    if (FD_ISSET(client_port, &socks)) { 
      if(command_request){
        if((*command_request)(client_port) == -1){
          close(client_port);
	  return 0;
        }
      }
      FD_CLR(client_port, &socks);
    }
  }

  return 1; 
}
#endif /* no use */


////////////// TEST Implementation
//

void init_SockProfile(){
  int i;

  SOCKET_LOCK();

  for(i=0;i<FD_SETSIZE;i++){
    memset(&SockProfile[i], 0, sizeof(struct sockport_profile));
  }

  SOCKET_UNLOCK();

  return;
}

void clear_SockProfile(int fd){

  SOCKET_LOCK();

  memset(&SockProfile[fd], 0, sizeof(struct sockport_profile));

  SOCKET_UNLOCK();
  
  return;
}

/*
void set_SockProfile_type(int fd, int type){

  SOCKET_LOCK();
  
  SockProfile[fd].type = type;

  SOCKET_UNLOCK();
  return;
}
*/
void set_SockProfile_arg(int fd, void *arg){
  
  SOCKET_LOCK();
  
  SockProfile[fd].arg = arg;

  SOCKET_UNLOCK();
  return;
}

void *get_SockProfile_arg(int fd)
{
  return SockProfile[fd].arg;
}


/*
#ifdef USE_SHMC
void set_SockProfile_shmc(int fd, Shmc *shmc)
{
  SOCKET_LOCK();
  
  SockProfile[fd].shmc = shmc;

  SOCKET_UNLOCK();
}
#endif
*/
/*
void set_SockProfile_service(int fd,
		int (*command_request)()){

  SOCKET_LOCK();
  
  SockProfile[fd].command_proc = command_request;

  SOCKET_UNLOCK();

  return;
}
*/

static void get_Servers_from_SockProfile(int *res, int *n, int start_fd){
  int i;
  int current;

  SOCKET_LOCK();

  for(i=start_fd, current=0;i<FD_SETSIZE;i++){
    if(SockProfile[i].type == SOCK_SERVER){
      res[ current++ ]=i;
    }
  }
  *n=current;

  SOCKET_UNLOCK();

  return;
}

static void copy_sockport_profile(int dist, int src){

  SOCKET_LOCK();
  
  memcpy(&SockProfile[dist], &SockProfile[src], sizeof(struct sockport_profile));

  SOCKET_UNLOCK();

  return;
}

int set_SockProfile(int fd, int type,
		int (*connection_request)(int, void*),
		int (*disconnect_request)(int, void*),
		int (*command_request)(GIOP_ConnectionHandler*)){
  SOCKET_LOCK();
  
  SockProfile[fd].type = type;
  SockProfile[fd].command_proc = command_request;
  SockProfile[fd].connection_proc = connection_request;
  SockProfile[fd].disconnect_proc = disconnect_request;

  FD_SET(fd, &main_socket_bits);

  SOCKET_UNLOCK();

  return fd;
}

int clear_socket_profile(int fd, void (*clear_func)(int)){
  clear_SockProfile(fd);

  SOCKET_LOCK();

  FD_CLR(fd, &main_socket_bits);

  SOCKET_UNLOCK();

  if(clear_func) (*clear_func)(fd);

  return fd;
}

static int check_current_sockbits(int fd){
  if(FD_ISSET(fd, &main_socket_bits)) return 1;

  return 0;
}

static int next_active_socket(fd_set *socks, int start_fd){
  int i;

  for (i = start_fd; i < FD_SETSIZE; i++)
    if (FD_ISSET(i, socks))  return i;

  return -1;
}

int select_socket_ports( fd_set *sockbits, struct timeval time_out)
{
  int	 nfds;
  fd_set socks;
  int	 stat, res, newsock;
  int active_port;
  int (*cmnd_func)(GIOP_ConnectionHandler*);
  int (*cnct_func)(int, void*);
  int (*discn_func)(int, void*);
  GIOP_ConnectionHandler h;
//  struct timeval tv;

  FD_ZERO(&socks);
  nfds = copy_fds_set(&socks, sockbits);

//  tv.tv_sec = time_out->tv_sec;
//  tv.tv_usec = time_out->tv_usec;
  res = stat = select(nfds, &socks, 0, 0, &time_out);
  active_port = 0;

  while( stat > 0 ){
    if((active_port = next_active_socket(&socks, active_port) ) < 0 ){
      fprintf(stderr, "Invalid port is active...\n");
      return -1;
    }

    SOCKET_LOCK();

    cmnd_func = SockProfile[active_port].command_proc;

    SOCKET_UNLOCK();

    switch (SockProfile[active_port].type){
      case SOCK_SERVER:
#if DEBUG1
	fprintf(stderr, "Request Connection fd=%d\n", active_port);
#endif
        newsock = accept_connection(active_port);     /* new connection */

        if (newsock>0) {
          FD_SET(newsock, sockbits);
	  copy_sockport_profile(newsock, active_port);

	  SOCKET_LOCK();

          SockProfile[newsock].type = SOCK_SERVICE;

	  SOCKET_UNLOCK();

#if DEBUG1
	  fprintf(stderr, "New Connection fd=%d\n", newsock);
#endif
	}

        cnct_func = SockProfile[active_port].connection_proc;
	if(cnct_func) (*cnct_func)(active_port, SockProfile[active_port].arg );

        FD_CLR(active_port, &socks);
        stat--;
	break;
      case SOCK_CLIENT:
      case SOCK_SERVICE:
	h.type = CONNECTION_TYPE_SOCKET;
	h.sock = active_port;

        if(cmnd_func &&  (*cmnd_func)(&h) == -1 ){
          discn_func = SockProfile[active_port].disconnect_proc;
	  if(discn_func) (*discn_func)(active_port, SockProfile[active_port].arg );

#if DEBUG1
	  fprintf(stderr, "Request Close Connection fd=%d\n", active_port);
#endif
#ifdef __T_KERNEL__
#ifdef DEBUG_T_KERNEL
          printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
	  closesocket(active_port);
#ifdef DEBUG_T_KERNEL
          printf("T_KERNEL %s:%d\n", __FILE__, __LINE__);
#endif
#else
	  close(active_port);
#endif

          FD_CLR(active_port, sockbits);

	  SOCKET_LOCK();

          SockProfile[active_port].type = SOCK_CLOSED;

	  SOCKET_UNLOCK();
	}

        FD_CLR(active_port, &socks);
        stat--;
	break;
      default:
        fprintf(stderr, "Socket already closed...\n");
        stat--;
	break;
      }
  }

  return res; 
}

void init_socket_servers()
{
  int i;
  int sockServers[FD_SETSIZE];
  int nSock;

  get_Servers_from_SockProfile(sockServers, &nSock, 0);

  SOCKET_LOCK();

  FD_ZERO(&main_socket_bits);
  for(i=0;i<nSock;i++){ FD_SET(sockServers[i], &main_socket_bits); }

  SOCKET_UNLOCK();
}

void select_socket_servers(struct timeval time_out)
{
  select_socket_ports( &main_socket_bits, time_out);
}

