/*!
 * @file gnav.cpp
 * @brief Global Navigation Backend
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#include <exception>
#include <algorithm>
#include <cassert>
#include <iostream>
#include <cstring>
#include <fstream>
#include <iterator>
#include <queue>

#include "gnav.h"
#include "config_data.h"
#include "service_if.h"
#include "debug_utils.h"
#include "baseRTC.h"
#include "navigation_data.h"
#include "Localization.h"
#include "GridMap.h"
#include "Path.h"

using namespace std;
using RGIS::PntC;
using RGIS::BoxC;
using RGIS::Geo::GridInfoC;
using RGIS::Geo::GridC_var;
using RGIS::Geo::LineStringC_var;
using namespace NavData;

const char GNav::NAME[] = "GlobalNavigationRTC";
const char GNav::TAG_SHORTEST_PATH[] = "ShortestPath";

GNav::GNav(MapService_t& ms, RouteService_t& rs) 
	: sif_(new ServiceIF(ms, rs)),
	  id_()
{
}
void GNav::initialize()
{
	sif_->initialize();
}
template <class Serializable>
bool
GNav::writeWithHeader(const Serializable& s, const string& tag, ostream& os)
{
	baseRTC::Header hdr(NavData::CATEGORY, 
						Serializable::TYPE_ID,
						NAME,
						id_,
						tag);
	if (hdr.writeToXml(os) < 0) {
		cout << "Cannot write header for " << Serializable::TYPE_ID << "\n";
		return false;
	}
	if (s.writeToXml(os) < 0) {
		cout << "Cannot write object " << Serializable::TYPE_ID << "\n";
		return false;
	}
	return true;
}
bool
GNav::cmd_setDestination(istream& is, queue<string>& resultq, const baseRTC::Header& hdr)
{
	FUNC_TRACE;
	NavData::TaggedPoint tp;
	if (tp.readFromXml(is) < 0) {
		cout << "Parse error\n";
		return false;
	}
	if (tp.tag != "destination") {
		cout << "This point is not a destination!\n";
		return false;
	}
	if (tp.coordinate != NavData::COORD_JGD9) {
		cout << "Destination must be described in " 
			 << NavData::COORD_JGD9 << "." 
			 << "But this has " << tp.coordinate << "\n";
		return false;
	}
	return setDestination(tp.position, resultq);
}
bool
GNav::setDestination(const RGIS::PntC& dest, queue<string>& resultq)
{
	cout << "Setting destination to "
		 << "(" << dest.x << ", " << dest.y << ") .... " 
		 << endl;

	NavData::PotentialGrid grid;
	if (!sif_->setDestination(dest, grid)) {
		cout << "error while setting destination!\n";
		return false;
	} 
	cout << "done\n";
	ostringstream os;
	if (writeWithHeader(grid, grid.TYPE_ID, os)) {
		resultq.push(os.str());
		return true;
	} else 
		return false;
}
bool
GNav::cmd_setCurrentPos(istream& is, queue<string>& resultq, const baseRTC::Header& hdr)
{
	FUNC_TRACE;
	Localization loc;
	if (loc.readFromXml(is) < 0) {
		cout << "Parse error\n";
		return false;
	}

	CurPos curpos;
	curpos.pos.x = loc.pos_x;
	curpos.pos.y = loc.pos_y;
	curpos.pos.z = 0.0;
	curpos.theta = loc.pos_the;
	curpos.reset = (hdr.tag == "initialize");

	return setCurrentPos(curpos, resultq);
}
bool
GNav::setCurrentPos(const CurPos& curpos, queue<string>& resultq)
{
	FUNC_TRACE;
	cout << "Setting Current position to "
		 << "(" 
		 << curpos.pos.x << ", " << curpos.pos.y << ", " << curpos.theta
		 << ") .... " 
		 << flush;

	Path path;
	if (!sif_->setCurrentPosition(curpos, path)) {
		cout << "error while setting current position\n";
		return false;
	}

	cout << "done\n";
	if (path.path.empty())
		return true;   // Already sent path. Don't need to enqueue it.

	ostringstream os;
	if (writeWithHeader(path, TAG_SHORTEST_PATH, os)) {
		resultq.push(os.str());
		return true;
	} else 
		return false;
}
bool
GNav::cmd_saveGridMap(istream& is, queue<string>& resultq, const baseRTC::Header& hdr)
{
	FUNC_TRACE;
	
	(void) resultq;				// no output string.

	GridMap gm;
	if (gm.readFromXml(is) < 0) {
		cout << "Parse error\n";
		return false;
	}
	if (!sif_->storeProbabilityMap(gm)) {
		cout << "error while storing probability grid map\n";
		return false;
	}
	return true;
}

bool
GNav::cmd_commandString(istream& is, queue<string>& resultq, const baseRTC::Header& hdr)
{
	FUNC_TRACE;
	NavData::CommandString cs;
	if (cs.readFromXml(is) < 0) {
		cout << "Parse error\n";
		return false;
	}
	cout << "command.type == " << cs.type << "\n";
	if (cs.type == "calc_grid") {
		NavData::PotentialGrid grid;
		if (!sif_->recalcPotentialGrid(grid)) {
			cout << "error while recalc grid!\n";
			return false;
		}
		ostringstream os;
		if (writeWithHeader(grid, grid.TYPE_ID, os)) {
			resultq.push(os.str());
		} else 
			return false;

		os.str("");
		Path path;
		sif_->getCurrentPath(path); // path may be empty.
		if (writeWithHeader(path, TAG_SHORTEST_PATH, os)) {
			resultq.push(os.str());
		} else 
			return false;

	} else if (cs.type == "initialize") {
		initialize();
	} else {
		cout << "unknown command\n"; 
		return false;
	}

	return true;
}
bool
GNav::processXml(const string& xml, queue<string>& resultq)
{
	istringstream is(xml);
	baseRTC::Header ihdr;
	if (ihdr.readFromXml(is) < 0) {
		cout << "Cannot Read Header\n";
		return RTC::RTC_OK;
	}
	cout << "category:    [" << ihdr.category << "]\n";
	cout << "sender_name: " << ihdr.sender_name << "\n";
	cout << "type:        " << ihdr.type << "\n";
	cout << "time:        " << setprecision(6) << fixed << ihdr.time << "\n";
	cout << "id:          " << ihdr.id << "\n";
	cout << "tag:         " << ihdr.tag << "\n";
	cout << "version:     " << ihdr.hdr_version << "\n";

	if (ihdr.category != NavData::CATEGORY && ihdr.category != "navigation") {
		cout << "Not for me.\n";
		return RTC::RTC_OK;
	}

	const struct {
		const char* const type;
		bool (GNav::* const handler)(istream&, queue<string>& resultq, const baseRTC::Header& ihdr);
	} TBL[] = {
		{ NavData::TaggedPoint::TYPE_ID,   &GNav::cmd_setDestination },
		{ Localization::TYPE_ID,           &GNav::cmd_setCurrentPos },
		{ GridMap::TYPE_ID,                &GNav::cmd_saveGridMap },
		{ NavData::CommandString::TYPE_ID, &GNav::cmd_commandString },
	};
	static const size_t TBL_SIZE = sizeof(TBL) / sizeof (TBL[0]);

	for (size_t i = 0; i < TBL_SIZE; ++i) {
		if (ihdr.type == TBL[i].type)
			return (this->*(TBL[i].handler))(is, resultq, ihdr);
	}
	cout << "This XML is not for me ... \n";

	return false;
}
bool
GNav::getDestination(RGIS::PntC& dest) const
{
	return sif_->getDestination(dest);
}
bool
GNav::getCurrentPos(CurPos& curpos) const
{
	return sif_->getCurrentPos(curpos);
}
void
GNav::readConfigFrom(const ConfigData& cd)
{
	sif_->setPotentialGridDepth(cd.getPotentialGridDepth());
}

/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
