/*!
 * @file gnav.h
 * @brief Global Navigation Backend
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 *
 */



#if !defined(GNAV_HEADER_)
#define      GNAV_HEADER_

#include <string>
#include <queue>
#include <boost/shared_ptr.hpp>
#include "GlobalNavigationRTC.h"
#include "cur_pos.h"

namespace baseRTC {
class Header;
}

class ConfigData;
class ServiceIF;
/*! 
 * @class GNav
 * @brief GNav: Global Navigation Backend.
 *
 * 大経路計画のサービスを行う。@a GlobalNavigationRTC に対してサービスの
 * インターフェースを提供する。@a ServiceIF のインスタンスを所有し，実際
 * のRGISとの通信は @a ServiceIF のインスタンスが行う。
 * @a GNav はXMLを処理し，ServiceIFのコマンドを起動する。
 *
 *
 */
class GNav {
public:
	//! @typedef RTC::CorbaConsumer<RGIS::RTComp::MapService> MapService_t
	//! MapService_t RGISのMapServiceオブジェクトを表す型
	typedef RTC::CorbaConsumer<RGIS::RTComp::MapService> MapService_t;

	//! @typedef RTC::CorbaConsumer<RGIS::RTComp::RouteService> RouteService_t
	//! RouteService_t RGISのRouteServiceオブジェクトを表す型
	typedef RTC::CorbaConsumer<RGIS::RTComp::RouteService> RouteService_t;

	/*! constructor.
	 * 
	 * @param[in] ms MapServicePortへの参照
	 * @param[in] rs RouteServicePortへの参照
	 */
	GNav(MapService_t& ms, RouteService_t& rs);

	//! destructor.
	//!  何もしない。念のためpublic virtualとして宣言のみしておく
	virtual ~GNav() { }

	
	void initialize();			//!< 初期状態に復帰
	
	//! 設定の読み込み.
	//! @param[in] cd 設定をここから読む
	void readConfigFrom(const ConfigData& cd);

	/*! 目的地を設定する.
	 * 
	 * @param[in] dest 設定する目的地
	 * @param[out] resultq XML出力の結果が格納される。ここでは@a PotentialGrid
	 * @return 成功(true)／失敗(false)
	 */
	bool setDestination(const RGIS::PntC& dest, std::queue<std::string>& resultq);

	/*! 現在位置を設定する.
	 * 
	 * @param[in] curpos 設定する目的地
	 * @param[out] resultq XML出力の結果が格納される。ここでは@a Path
	 * @return 成功(true)／失敗(false)
	 */
	bool setCurrentPos(const CurPos& curpos, std::queue<std::string>& resultq);

	/*! 入力されたXMLを処理し，適切なコマンドを起動する。
	 * 
	 * @param[in] xml文字列。(シリアライズされたオブジェクト)
	 * @param[out] resultq XML出力の結果が格納される。
	 * @return 成功(true)／失敗(false)
	 */
	bool processXml(const std::string& xml, std::queue<std::string>& resultq);

	/*! 目的地が設定されていれば，それを返す。
	 * 
	 * @param[out] dest 目的地があればここに格納される
	 * @return 目的地があればtrue(@a dest に格納)。なければfalse(@a dest は不変)
	 */
	bool getDestination(RGIS::PntC& dest) const;

	/*! 現在地が設定されていれば，それを返す.
	 * 
	 * @param[out] curpos 現在地があればここに格納される
	 * @return 現在地があればtrue(@a curpos に格納)。なければfalse(@a curpos は不変)
	 */
	bool getCurrentPos(CurPos& curpos) const;

private:
	/*! オブジェクトを適切なヘッダとともにストリームに書き込む.
	 *
	 * @tparam Serializable シリアライズ可能クラス
	 * @param[in] s シリアライズしたいオブジェクト
	 * @param[in] tag ヘッダ中のtagフィールド
	 * @param[in] os 書込み先ストリーム
	 * @return 成功(true)／失敗(false)
	 */
	template <class Serializable> bool writeWithHeader(const Serializable& s, 
													   const std::string& tag,
													   std::ostream& os);

	/*! 目的地の設定。processXmlからコールされる.
	 *
	 * @param[in] is 入力のオブジェクトをここからデシリアライズする
	 * @param[out] resultq 結果のオブジェクトをシリアライズして書くqueue
	 * @param[in] ihdr RTCヘッダ
	 * @return 成功(true)／失敗(false)
	 */
	bool cmd_setDestination(std::istream& is, std::queue<std::string>& resultq,
							const baseRTC::Header& ihdr);

	/*! 現在位置の設定。processXmlからコールされる.
	 *
	 * @param[in] is 入力のオブジェクトをここからデシリアライズする
	 * @param[out] resultq 結果のオブジェクトをシリアライズして書くqueue
	 * @param[in] ihdr RTCヘッダ
	 * @return 成功(true)／失敗(false)
	 */
	bool cmd_setCurrentPos(std::istream& is, std::queue<std::string>& resultq, 
						   const baseRTC::Header& ihdr);

	/*! 確率グリッドマップの保存。processXmlからコールされる.
	 *
	 * @param[in] is 入力のオブジェクトをここからデシリアライズする
	 * @param[out] resultq 結果のオブジェクトをシリアライズして書くqueue
	 * @param[in] ihdr RTCヘッダ
	 * @return 成功(true)／失敗(false)
	 */
	bool cmd_saveGridMap(std::istream& is, std::queue<std::string>& resultq,
						 const baseRTC::Header& ihdr);

	/*! 文字列コマンドの処理。processXmlからコールされる.
	 *
	 * @param[in] is 入力のオブジェクトをここからデシリアライズする
	 * @param[out] resultq 結果のオブジェクトをシリアライズして書くqueue
	 * @param[in] ihdr RTCヘッダ
	 * @return 成功(true)／失敗(false)
	 */
	bool cmd_commandString(std::istream& is, std::queue<std::string>& resultq,
						   const baseRTC::Header& ihdr);

	static const char TAG_SHORTEST_PATH[];
	static const char NAME[]; //!< 出力Headerのnameフィールドに対応する文字列
	boost::shared_ptr<ServiceIF> sif_; //!< @a ServiceIF へのスマートポインタ
	std::string id_;   //!< 出力Headerのidフィールドに対応する文字列

};


#endif  /*!  GNAV_HEADER_ */
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
