/*!
 * @file service_if.cpp
 * @brief Interface to RGIS MapService and RouteService
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 *
 */

#include "cur_pos.h"
#include "debug_utils.h"
#include "baseRTC.h"
#include "navigation_data.h"
#include "Localization.h"
#include "GridMap.h"
#include "Path.h"
#include "service_if.h"
#include "merge_grid.h"


using namespace std;
using RGIS::PntC;
using RGIS::BoxC;
using RGIS::Geo::GridC;
using RGIS::Geo::GridInfoC;
using RGIS::Geo::GridC_var;
using RGIS::Geo::LineStringC_var;
using NavData::CommandString;
using NavData::PotentialGrid;

/* ================================================================ *
 *  file-local utilities
 * ================================================================ */

namespace /* unnamed */ { 
/*! 
 * @brief PntCの等号演算子
 * @param[in] lhs 左辺
 * @param[in] rhs 右辺
 *
 * 全ての成分が等しい場合に両辺は等しい
 */
bool
operator==(const PntC& lhs, const PntC& rhs)
{
	return (lhs.x == rhs.x) && (lhs.y == rhs.y) && (lhs.z == rhs.z);
}
/*! 
 * @brief PntCの不等演算子
 * @param[in] lhs 左辺
 * @param[in] rhs 右辺
 *
 * 等値の否定
 */
bool
operator!=(const PntC& lhs, const PntC& rhs)
{
	return !(lhs == rhs);
}

} /* unnamed namespace */


/* ================================================================ *
 *  class ServiceIF
 * ================================================================ */


ServiceIF::ServiceIF(MapService_t& ms, RouteService_t& rs)
	: ms_(ms),
	  rs_(rs),
	  have_dest_(false),
	  have_currrent_pos_(false),
	  have_pgrid_(false),
	  have_pmap_(false),
	  pgrid_depth_(256)
{
	FUNC_TRACE;
	initialize();
}
void
ServiceIF::initialize()
{
	have_dest_ = false;
	have_currrent_pos_ = false;
	have_pgrid_ = false;
	have_pmap_ = false;
}
bool
ServiceIF::setDestination(const PntC& dest, PotentialGrid& grid)
{
	FUNC_TRACE;

	if (have_dest_ && dest_ == dest) {
		cout << "same as cached destination\n";
		return true;
	}
	dest_ = dest;
	
	have_pgrid_ = false;		// change of dest invalidates it.
	if (emit_pgrid(grid)) {
		have_dest_ = true;
		return true;
	} else {
		return false;
	}
}
bool
ServiceIF::setCurrentPosition(const CurPos& pos, Path& path)
{
	FUNC_TRACE;
	current_pos_ = pos;
#if 0
	have_currrent_pos_ = true;
	return getCurrentPath(path);
#else
	if (!have_currrent_pos_ || pos.reset) {
		have_currrent_pos_ = true;
		return getCurrentPath(path);
	} else {
		path = Path();			// empty
		return true;
	}
#endif
}
bool
ServiceIF::getCurrentPath(Path& path)
{
	FUNC_TRACE;
	if (!have_pgrid_) {
		path = Path();		// empty;
		return false;
	} else {
		return emit_path(path);
	}
}
bool
ServiceIF::storeProbabilityMap(const GridMap& gm)
{
	FUNC_TRACE;

	have_pmap_ = true;
	pmap_ = gm;
	merge_grid();
	return true;
}
bool
ServiceIF::recalcPotentialGrid(PotentialGrid& grid)
{
	FUNC_TRACE;
	if (!have_dest_) {
		cout << "no destination\n";
		return true;
	}
	if (!have_pgrid_) {
		cout << "no pgrid!\n";
		return true;
	}
	return emit_pgrid(grid);
}
/// したうけ
bool
ServiceIF::emit_pgrid(PotentialGrid& grid)
{
	FUNC_TRACE;
	if (!update_pgrid())
		return false;

	const GridInfoC& info = merged_grid_.info; // just for readability
	const size_t n_cells = info.numCellsW * info.numCellsH;
	cout << "n_cells = " << n_cells << "\n";
	if (n_cells == 0) {
		cout << "empty grid!\n";
		return false;
	}
	grid.info = NavData::GridInfo(info);
	cout << "merged_grid_.data.length() = " 
		 << merged_grid_.data.length() << "\n";

	const PntC& ul = info.position;

	// この計算方法は回転を考慮していないが，RGIS側でrotation をzero
	// にする制約があるのため，これで良しとする。
	const PntC  dr = {ul.x + info.numCellsW*info.cellWidth,
					  ul.y + info.numCellsH*info.cellHeight, 
					  0.0 };

	cout << "upleft:    (" << ul.x << ", " << ul.y << ")\n";
	cout << "downRight: (" << dr.x << ", " << dr.y << ")\n";

	const double* begin = (const double*)(merged_grid_.data.NP_data());
	const double* end = begin + n_cells;

	if (0) {
		ofstream ofs("out.dat");
		if (!ofs) {
			cout << "cannot open file\n";
		} else {
			copy(begin, end, ostream_iterator<double>(ofs, "\n"));
		}
	}

	double minval = numeric_limits<double>::max();
	double maxval = numeric_limits<double>::min();
	if (!info.useNoData) {
		cout << "not use noDataVal\n";
		minval = *min_element(begin, end);
		maxval = *max_element(begin, end);
	} else {
		cout << "noDataVal = " << info.noDataVal << "\n";
		for (const double* dp = begin; dp != end; ++dp) {
			double d = *dp;
			if (d != info.noDataVal) {
				if (d < minval)
					minval = d;
				if (d > maxval)
					maxval = d;
			}
        }
	}
	cout << "(min, max) = (" << minval << ", " << maxval << ")\n";

	grid.data.resize(n_cells);
	grid.data.setRange(minval, maxval);
	grid.data.setNumberOfValues(pgrid_depth_);
	cout << "n_o_v0:    " << grid.data.numberOfValues() << "\n";
	grid.data.assign(begin, end);

	cout << "n_o_v1:    " << grid.data.numberOfValues() << "\n";
	cout << "range_min: " << grid.data.rangeMin() << "\n";
	cout << "range_max: " << grid.data.rangeMax() << "\n";

	cout << "emit merged grid as xml !\n";
	return true;
}
bool
ServiceIF::update_pgrid()
{
	FUNC_TRACE;
	if (have_pgrid_)
		return true;

	cout <<  "createPotentialGrid\n";

	BoxC extent = {0.0};
	GridC_var grid;
	long ret = rs_->createPotentialGrid(grid, dest_, extent);
	if (ret != 0) {
		cout << "createPotentialGrid error: " << ret << "\n";
		return false;
	}
	const GridInfoC& info = grid->info; // just for readability

	const size_t n_cells = info.numCellsW * info.numCellsH;
	const size_t n_bytes = grid->data.length();
	if (n_cells * sizeof(double) != n_bytes) {
		cout << "corrupted data !\n";
		return false;
	}
	// ok copy and store them.
	pgrid_ = grid;
	have_pgrid_ = true;

	merge_grid();


	// now, merged_grid_ is the grid ready for output.
	return true;
}
void
ServiceIF::merge_grid()
{
	FUNC_TRACE;
	if (!have_pgrid_) {
		cout << "no grid to merge!!!\n";
		return;
	} else if (!have_pmap_) {
		merged_grid_ = pgrid_;		// just copy for now.
	} else {
		cout << "merge pmap_ and pgrid_\n";

		do_merge(pgrid_, pmap_, current_pos_, obstacle_thresh_);
	}
}
namespace /* unnamed */ {
inline double
now()
{
	struct timeval tv;
	struct timezone dummy;
	gettimeofday(&tv, &dummy);
	return double(tv.tv_sec) + tv.tv_usec * 1.0e-6;
}

} /* unnamed namespace */

bool
ServiceIF::emit_path(Path& result)
{
	FUNC_TRACE;
	vector<PntC> path(get_path());
	vector<Node> nodes(path.size());
	for (size_t i = 0, imax = path.size(); i < imax; ++i) {
		nodes[i] = Node(path[i].x, path[i].y, 0.0);
	}

	if (path.empty()) {
		cout << "error get_path\n";
		result = Path();	// empty nodes
		return false;
	} else {
		result.time = now();
		result.path.swap(nodes);
		return true;
	}
}
vector<PntC>
ServiceIF::get_path()
{
	FUNC_TRACE;
	update_pgrid();

	vector<PntC> result;
	
	LineStringC_var path;
	long ret = rs_->getShortestPathFromGrid(path, merged_grid_, current_pos_.pos);
	if (ret != 0) {
		cout << "getShortestPathFromGrid error: " << ret << "\n";
		return result;			// empty;
	}

	size_t path_len = path->length();
	result.resize(path_len);
	for (size_t i = 0; i < path_len; ++i) {
		result[i] = path[i];
	}
	return result;
}
bool
ServiceIF::getDestination(PntC& dest) const
{
	if (!have_dest_)
		return false;
	dest = dest_;
	return true;
}	
bool
ServiceIF::getCurrentPos(CurPos& pos) const
{
	if (!have_currrent_pos_)
		return false;
	pos = current_pos_;
	return true;
}
void
ServiceIF::setObstacleThreshold(double th)
{
	if (th < 0.0)
		th = 0.0;
	else if (1.0 < th)
		th = 1.0;
	obstacle_thresh_ = th;
}
void
ServiceIF::setPotentialGridDepth(int depth)
{
	if (depth < 2)
		pgrid_depth_ = 2;
	else if (depth > 65536)
		pgrid_depth_ = 65536;
	else
		pgrid_depth_ = depth;
}
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */

