/*!
 * @file config_data.cpp
 * @brief Configuration Data for the MapViewer Component
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */ 
#include <ostream>
#include <iostream>
#include <vector>
#include <string>

#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>

#include <xercesc/sax2/SAX2XMLReader.hpp>
#include <xercesc/sax2/XMLReaderFactory.hpp>
#include <xercesc/util/PlatformUtils.hpp>

#include "XmlHandler.h"

#include "config_data.h"

using namespace std;
using namespace boost;

namespace /*unnamed  */{

/*! 
 * @class XmlParserRAII
 * @brief RAIIを使ってXmlParserの後始末をさせるためのヘルパクラス
 *
 * 現行の@a XmlHander.hではParserの初期化，後始末に自由関数 @a
 * InitXmlParser() @a TerminateXmlParser()を呼ぶ必要があるが，設定ファ
 * イルにXMLエントリを追加／削除する際に，これを忘れることのないよう，
 * リソースを管理するクラスとして作成した。
 */
class XmlParserRAII {
public:
	
	/*! コンストラクタ.
	 *  @a InitXmlParser()をコールしてparserを初期化
	 *  @param [in] ファイル名
	 */
	XmlParserRAII(const string& filename) : 
		filename_(filename)
	{
		InitXmlParser();
	}
	/*! デストラクタ
	 *  @a TerminateXmlParser()をコールしてparserの後始末を行う
	 */
	~XmlParserRAII() {
		TerminateXmlParser();
	}

	/*! XML中から型@a Tのエントリを読み込むテンプレートメンバ
	 * @tparam T 探したいエントリの型。一度文字列にした後に@a Tへの変換
	 * を試みる
	 * @param [in] key 探索のキーとするタグ名
	 * @param [in] attr 属性フィ−ルド
	 * @param [out] result 結果の値
	 * @return エントリが見つかって読み出しが出来たかどうか
	 */
	template <typename T> 
	bool find_entry(const string& key, const string& attr, T& result)
	{
		if (!XmlParser(filename_, key, attr, &strs_)) {
			cerr << "cannot find " << key << " in " << filename_ << endl;
			return false;
		} else if (strs_.empty()) {
			cerr << "empty strs" << endl;
			return false;
		}

		try {
			result = lexical_cast<T>(strs_[0]);
		} catch (const bad_lexical_cast& e) {
			cerr << "Cannot parse " << strs_[0]
				 <<  " for " << key
				 << " in file: " << filename_ << endl;
			return false;
		}
		return true;
	}
private:
	vector<string> strs_;		//!< find_entryで使うためのバッファ
	string filename_;			//!< XMLファイル名
};

} /* unnamed namespace */

/* ---------------------------------------------------------------- *
 *  Class ConfigData
 * ---------------------------------------------------------------- */

const char ConfigData::INPUT_NUMBER[] = "input_number";
const char ConfigData::STARTUP_DIRNAME[] = "startup_dir";
const char ConfigData::POTENTIAL_MIN[] = "potential_min";
const char ConfigData::POTENTIAL_MAX[] = "potential_max";

ConfigData::ConfigData()
	: n_inports_(1),
	  startup_dirname_("./"),
	  pot_min_(0.0),
	  pot_max_(1000.0)
{
}
bool
ConfigData::readFromFile(const string& filename)
{
	bool all_ok = true;
	XmlParserRAII parser(filename);

	if (!parser.find_entry(INPUT_NUMBER, "", n_inports_))
		all_ok = false;

	if (!parser.find_entry(STARTUP_DIRNAME, "", startup_dirname_))
		all_ok = false;

	if (!parser.find_entry(POTENTIAL_MIN, "", pot_min_))
		all_ok = false;

	if (!parser.find_entry(POTENTIAL_MAX, "", pot_max_))
		all_ok = false;

	return all_ok;
}
ostream&
operator<<(std::ostream& ost, const ConfigData& cd)
{
	ost << ConfigData::INPUT_NUMBER    << ": " << cd.numberOfInPorts() << "\n"
		<< ConfigData::STARTUP_DIRNAME << ": " << cd.startupDirectory() << "\n"
		<< ConfigData::POTENTIAL_MIN   << ": " << cd.potentialMin() << "\n"
		<< ConfigData::POTENTIAL_MAX   << ": " << cd.potentialMax() << "\n"
		<< flush;

	return ost;
}


/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
