/*!
 * @file gui_main.cpp
 * @brief The entry point of MapViewer GUI 
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#include <QtGui>
#include <QApplication>

#include <fstream>
#include <iostream>
#include <iomanip>
#include <string>

#include "map_view.h"
#include "layer_view.h"
#include "map_layer.h"
#include "map_manager.h"

#include "baseRTC.h"
#include "xml_serdes.h"
#include "navigation_data.h"
#include "shared_queue.h"
#include "debug_utils.h"
#include "gui_main.h"
#include "config_data.h"
#include <rtm/Manager.h>
using namespace std;
/* ---------------------------------------------------------------- *
 *  Implementation of MapViewerGUI
 * ---------------------------------------------------------------- */

/*! 
 * @class MapViewerGUI::Impl
 * @brief MapViewerGUIの実装詳細クラス
 *
 * MapViewerGUIの詳細情報，プライベートメンバ等を隠蔽するためのクラス(pimpl)
 */
class MapViewerGUI::Impl {
public:
	typedef SharedQueue<string> Queue_t;

	//! コンストラクタ
	Impl();

	/*! blockせずにqueueからのデータ取り出しを試みる。
	 *
	 * 内部で管理するデータキューは条件変数とmutexによりlockされている。
	 * ここでブロックが発生すると，GUIのイベントループ／RTCの実行コンテ
	 * クストを停止してしまうため，ブロックされている場合にはfalseで戻る
	 * メソッドを作成した。
	 *
	 * @param[out] result queueから取り出された文字列
	 * @return dequeueに成功した。(データが返された)
	 */
	bool tryDequeueResult(std::string& result);

	//! 入力queueにXMLを入れる
	//! @param[in] xml queueに入れられる文字列
	void enqueueXml(const string& xml);

	/*! ポテンシャルをカラー表示する際の閾値を設定
	 *  @param[in] minval これ以下は最小値として全て同じ色になる
	 *  @param[in] maxval これ以上は最大値として全て同じ色になる
	 */
	void setPotentialRange(double minval, double maxval);

	/*! 起動時にデータを読み込むディレクトリ名を設定
	 *  @param[in] newdir ディレクトリ名
	 */
	void setStartupDirectory(const QString& newdir) { startup_dir_ = newdir; }

	/*! 別スレッドから起動されるサービスルーチン.
	 *  ここではGUIのイベントループを走らせる。
	 *  @return QTのイベントループの戻り値を返す
	 */
	int svc();

	bool is_running() const { return is_running_; } 
private:
	Queue_t xmlq_;
	Queue_t resultq_;
	double pot_min_;
	double pot_max_;
	QString startup_dir_;
	bool is_running_;
};
MapViewerGUI::Impl::Impl() 
	: pot_min_(0.0), pot_max_(1000.0), is_running_(false)
{
}
int
MapViewerGUI::Impl::svc()
{

	int argc = 1;
	char name[] = "MapViewer";
	char *argv[] = { name, 0 };

	Q_INIT_RESOURCE(images);
	QApplication app(argc, argv);
	is_running_ = true;

	MapView mv;
	LayerView lv;
	MapManager mgr(&mv, &lv);

	mgr.setStartupDirectory(startup_dir_);
	mgr.setPotentialRange(pot_min_, pot_max_);
	mgr.setIOQueue(&xmlq_, &resultq_);

	mgr.loadGuiSettings();
	mgr.loadAllLayers();

	lv.show();
	mv.show();

	int ret = app.exec();

	mgr.saveAllLayers();
	mgr.saveGuiSettings();
	is_running_ = false;

	RTC::Manager::instance().terminate();

	return ret;
}
bool
MapViewerGUI::Impl::tryDequeueResult(std::string& result)
{
//        FUNC_TRACE;
	return resultq_.try_dequeue(result);
}
void
MapViewerGUI::Impl::enqueueXml(const std::string& xml)
{
//        FUNC_TRACE;
	xmlq_.enqueue(xml);
}
void
MapViewerGUI::Impl::setPotentialRange(double minval, double maxval)
{
	pot_min_ = min(minval, maxval);
	pot_max_ = max(minval, maxval);
}

/* ---------------------------------------------------------------- *
 *  Class MapViewerGUI
 * ---------------------------------------------------------------- */
MapViewerGUI::MapViewerGUI()
	: impl_(new Impl)
{
}
int
MapViewerGUI::svc()
{
	return impl_->svc();
}
bool
MapViewerGUI::tryDequeueResult(std::string& result)
{
	return impl_->tryDequeueResult(result);
}
void
MapViewerGUI::enqueueXml(const std::string& xml)
{
	return impl_->enqueueXml(xml);
}
void
MapViewerGUI::readConfigFrom(const ConfigData& cd)
{
	impl_->setPotentialRange(cd.potentialMin(), cd.potentialMax());
	impl_->setStartupDirectory(cd.startupDirectory().c_str());
}
bool
MapViewerGUI::is_running() const
{
	return impl_->is_running();
}

/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
