/*!
 * @file gui_main.h
 * @brief The entry point of MapViewer GUI 
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#if !defined(GUI_MAIN_HEADER_)
#define      GUI_MAIN_HEADER_

#include <string>
#include <ace/OS.h>
#include <ace/Task.h>
#include <ace/Synch.h>

#include <boost/utility.hpp>
#include <boost/shared_ptr.hpp>

struct ConfigData;

/*! 
 * @class MapViewerGUI
 * @brief RTCから見たGUIのinterface
 *
 * RTコンポーネントから見たGUIのエントリクラス。ACE_Taskを継承し，別ス
 * レッドとして@a MapViewer のGUI部分を動かす。RTCとの通信は
 * @a SharedQueueを介して行う。
 *
 * スレッドのルーチンとして何を選ぶか迷ったが，以下の理由でACEを選択した。
 * 
 * <ul>
 *   <li> RTC, GUIともにメインループを持つためスレッドが必要</li>
 *   <li> RTC内部ではACEが使用されている。</li>
 *
 *   <li> RTCのmanagerをblockingモードで起動しても，裏にはORB等のスレッ
 *        ドが生成されている。従って，ここで使用するスレッドはRTC内部と
 *        互換でなくてはならない</li>
 *
 *   <li> RTCのマネージャをバックグラウンドで動かす方法では，終了処理が
 *   うまくゆかなかった。（調査不十分かもしれない）</li>
 *   
 *   <li> QThreadとpthread, ACEとpthreadの共存条件が不明
 *       (現在の環境ではQThread，ACEともにバックエンドはpthreadらしいが)</li>
 * </ul>
 *
 * また，複数のスレッドライブラリの共存を許さないとGUIからSharedQueueの
 * 監視を行うスレッドをQthreadから起動することが選択肢から外れてしまう。
 * QThreadがつかえると，signal/slotでスレッド間通信ができて楽なのだが，
 * これは出来ないため，GUIはTimerイベントでqueueをpollすることとした。
 * 
 * Queueをブロックせずにpollするために，@SharedQueue にtry_dequeueメソッ
 * ドを用意し，ブロックの必要がある場合は即座にfalseを返して抜けるよう
 * にした。
 */
class MapViewerGUI : public ACE_Task<ACE_MT_SYNCH>, boost::noncopyable {
public:
	MapViewerGUI();				//!< コンストラクタ
	virtual ~MapViewerGUI() { }	//!< 仮想デストラクタ。何もしない

	//!実装クラスの同メソッドをコール
	//!  @see MapViewerGUI::Impl::tryDequeueResult()
	bool tryDequeueResult(std::string& result);

	//!実装クラスの同メソッドをコール
	//!  @see MapViewerGUI::Impl::enqueueXml()
	void enqueueXml(const std::string& xml);
	
	//! 設定データを読み込む
	//! @param[in] cd 設定データ
	void readConfigFrom(const ConfigData& cd);

	bool is_running() const;

protected:
	//! 実装クラスの同メソッドをコール
	//! @see MapViewerGUI::Impl::svc()
	int svc();
private:
	class Impl;
	//! 内部実装クラスへのスマートポインタ
	//! @see MapViewerGUI
	boost::shared_ptr<Impl> impl_;
};


#endif    /*!GUI_MAIN_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
