/*!
 *  @file layer_archive.cpp
 *  @brief 
 *  @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#include <vector>
#include <fstream>
#include <string>
#include "xml_serdes.h"
#include "layer_archive.h"
#include "map_layer.h"

using namespace std;

namespace boost {
namespace serialization {

#define NVP(obj, member)  make_nvp(#member, obj.member)
template <class Archive> 
void
serialize(Archive& ar, LayerArchive::Data& d, const unsigned int /*version*/)
{
	ar & NVP(d, name);
	ar & NVP(d, isVisible);
	ar & NVP(d, opacity);
	ar & NVP(d, xml);
}
#undef NVP

} /* namespace serialization */
} /* namespace boost */



struct LayerArchive::Impl {
	typedef LayerArchive::data_type data_type;
	typedef LayerArchive::container_type container_type;
	

	Impl() { }
	void add(const MapLayer* layer);
	int saveToFile(const char* filename) const;
	int loadFromFile(const char* filename);

	const container_type& getDatas() const {
		return datas;
	}

	container_type datas;
	static const char TYPE_ID[];
	
	/*! 自身をXMLとして書き出す.
	 *  @param[out] os 書き出し先ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	writeToXml(std::ostream& os) const
	{
		return XmlSerDes::writeToXmlSub(os, *this, TYPE_ID);
	}

	/*! 自身をXMLから読み出す(値を設定).
	 *  @param[out] is 読み出し元ストリーム
	 *  @retval 0 成功
	 *  @retval negative 失敗
	 */
	int
	readFromXml(std::istream& is)
	{
		return XmlSerDes::readFromXmlSub(is, *this, TYPE_ID);
	}

	template <class Archive>
	void serialize(Archive& ar, const unsigned int /* version */)
	{
		ar & BOOST_SERIALIZATION_NVP(datas);
	}
};
const char LayerArchive::Impl::TYPE_ID[] = "LayerArchive";

void
LayerArchive::Impl::add(const MapLayer* layer)
{
	data_type data;
	data.name = string(layer->name().toAscii().data());
	data.isVisible = layer->isVisible();
	data.opacity = layer->opacity();
	data.xml = layer->getXml().toAscii().data();

	datas.push_back(data);
}
int
LayerArchive::Impl::saveToFile(const char* filename) const
{
	ofstream ofs(filename);
	if (!ofs) {
		return -1;
	}
	return writeToXml(ofs);
}
int
LayerArchive::Impl::loadFromFile(const char* filename)
{
	ifstream ifs(filename);
	if (!ifs) {
		return -1;
	}
	return readFromXml(ifs);
}


LayerArchive::LayerArchive()
	: impl_(new Impl())
{
}
int
LayerArchive::saveToFile(const char* filename) const
{
	return impl_->saveToFile(filename);
}
int
LayerArchive::loadFromFile(const char* filename)
{
	return impl_->loadFromFile(filename);
}
void
LayerArchive::add(const MapLayer* layer)
{
	return impl_->add(layer);
}
const LayerArchive::container_type&
LayerArchive::datas() const
{
	return impl_->getDatas();
}
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
