/*!
 * @file layer_view.cpp
 * @brief Widget for displaying informations of the layers
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#include <QtGui>

#include "map_layer.h"
#include "map_layer_list_model.h"
#include "map_layer_delegate.h"
#include "layer_view.h"
#include "map_layer_table_view.h"

using namespace std;
/* ================================================================ *
 *  Class for customized view for MapLayer;
 * ================================================================ */
MapLayerTableView::MapLayerTableView(QWidget* parent)
	: QTableView(parent)
{
	setMinimumSize(MIN_WIDTH, MIN_HEIGHT);
}
void
MapLayerTableView::moveUpSelected()
{
	qDebug() << "moveUpSelected()";
	moveSelectedRows(UP);
}
void
MapLayerTableView::moveDownSelected()
{
	qDebug() << "moveDownSelected()";
	moveSelectedRows(DOWN);
}

QVector<int>
MapLayerTableView::selectedLayerIndexes()
{
	QVector<int> indexes;

	QHeaderView* vh = verticalHeader();
	foreach(int vidx, selectedRowVisualIndexes()) {
		int lidx = vh->logicalIndex(vidx);
		qDebug() << "vidx: " << vidx << "lidx: " << lidx;

		indexes.push_back(lidx);
	}

	return indexes;
}

QVector<int>
MapLayerTableView::selectedRowVisualIndexes()
{
	QVector<int> rows;
	QHeaderView* vh = verticalHeader();
	foreach (QModelIndex l, selectedIndexes()) {
		if (l.column() == 0) {
			qDebug() << "row " << l.row() << "selected"
					 << "vidx = " << vh->visualIndex(l.row());
			rows.append(vh->visualIndex(l.row()));
		}
	}
	return rows;
}
void 
MapLayerTableView::moveSelectedRows(MoveDirection_t dir)
{
	QVector<int> rows(selectedRowVisualIndexes());

	QAbstractItemModel* dataModel = model();
	if (!dataModel) {
		Q_ASSERT(false && "BUG: no dataModel");
		return;
	}
	const int n_rows = dataModel->rowCount(rootIndex());
	if (rows.size() == 0) {
		qDebug() << "None selected";
		return;
	} else if (rows.size() == n_rows) {
		qDebug() << "All selected";
		return;
	}

	if (dir == UP)
		qSort(rows.begin(), rows.end(), qLess<int>());
	else
		qSort(rows.begin(), rows.end(), qGreater<int>());

	if ((dir==UP && rows[0]==0) || (dir==DOWN && rows[0]==n_rows-1)) {
		qDebug() << "already at top or bottom";
		return;
	}
	QHeaderView* vh = verticalHeader();
	foreach (int i, rows) {
		qDebug() << "row: " << i;

		vh->moveSection(i, dir==UP ? i-1 : i+1);
	}
	for (int i = 0; i < n_rows; ++i) {
		qDebug() << "i = " << i << ": "
				 << vh->sectionPosition(i) << ", "
				 << vh->logicalIndex(i) << ", "
				 << vh->visualIndex(i);
	}
}
/* ================================================================ *
 *  Method Definitions for LayerView
 * ================================================================ */
LayerView::LayerView(QWidget* parent)
	: QFrame(parent)
{
	model_  = new MapLayerListModel(this);
//        view_   = new QTableView(this);
	view_   = new MapLayerTableView(this);
	delegate_ = new MapLayerDelegate(MapLayerListModel::COL_VISIBILITY,
									 MapLayerListModel::COL_OPACITY,
									 MapLayerListModel::COL_NAME, this);
	view_->setModel(model_);
	view_->setItemDelegate(delegate_);
	view_->horizontalHeader()->setStretchLastSection(true);

	view_->setSelectionBehavior(QAbstractItemView::SelectRows);
	view_->setSelectionMode(QAbstractItemView::SingleSelection);
	view_->resizeColumnToContents(MapLayerListModel::COL_OPACITY);

	view_->horizontalHeader()->setMovable(false);
	view_->verticalHeader()->setMovable(false);

	view_->verticalHeader()->setResizeMode(QHeaderView::Fixed);

//        int size = style()->pixelMetric(QStyle::PM_ToolBarIconSize);
	int size = style()->pixelMetric(QStyle::PM_ListViewIconSize);
	QSize iconSize(size, size);

	upAction_ = new QAction(QIcon(":images/up_arrow.png"), tr("Move Up selected"), this);
	QToolButton* upButton = new QToolButton;
	upButton->setDefaultAction(upAction_);
	upButton->setIconSize(iconSize);

	downAction_ = new QAction(QIcon(":images/down_arrow.png"), tr("Move Down selected"), this);
	QToolButton* downButton = new QToolButton;
	downButton->setDefaultAction(downAction_);
	downButton->setIconSize(iconSize);

	saveAction_ = new QAction(QIcon(":images/floppy.png"), tr("Save Selected Layer"), this);
	QToolButton* saveButton = new QToolButton;
	saveButton->setDefaultAction(saveAction_);
	saveButton->setIconSize(iconSize);

	// 削除/Openはボタン無し。コンテクストメニューのみ
	deleteAction_ = new QAction(tr("Delete Selected Layer"), this);
	openAction_   = new QAction(tr("New Layer"), this);

	QHBoxLayout* buttonsLayout = new QHBoxLayout;
	buttonsLayout->addSpacing(size);
	buttonsLayout->addWidget(upButton);
	buttonsLayout->addWidget(downButton);
	buttonsLayout->addStretch();
	buttonsLayout->addWidget(saveButton);
	buttonsLayout->addSpacing(size);

	QVBoxLayout* topLayout = new QVBoxLayout;
	topLayout->addWidget(view_);
	topLayout->addLayout(buttonsLayout);
	setLayout(topLayout);

	connect(view_->verticalHeader(), SIGNAL(sectionMoved(int,int, int)),
			this, SLOT(on_sectionMoved(int, int, int)));

	connect(model_, SIGNAL(dataChanged(const QModelIndex&, const QModelIndex&)),
			this, SLOT(on_dataChanged(const QModelIndex&, const QModelIndex&)));
	connect(view_, SIGNAL(pressed(const QModelIndex &)),
			this,  SLOT(on_viewPressed(const QModelIndex &)));

	connect(upAction_,   SIGNAL(triggered()), 
			view_, SLOT(moveUpSelected()));
	connect(downAction_, SIGNAL(triggered()), 
			view_, SLOT(moveDownSelected()));
	connect(saveAction_, SIGNAL(triggered()), 
			this, SLOT(saveSelected()));

	connect(deleteAction_, SIGNAL(triggered()),
			this, SLOT(deleteSelected()));

	// signal -> signal
	connect(openAction_, SIGNAL(triggered()),
			this, SIGNAL(newLayerRequested()));
        

	setWindowTitle("Layers");
}
int
LayerView::addMapLayer(MapLayer* layer)
{
	int n_layers = model_->addMapLayer(layer);

	qDebug() << "addMapLayer: " << n_layers;

	// work around for linux :
#if 1
	view_->setModel(NULL);
	view_->setModel(model_);
#endif
	return n_layers;
}
MapLayer*
LayerView::layerOfName(const QString& name)
{
	return model_->layerOfName(name);
}
int
LayerView::visualIndexOfLayer(const MapLayer* l)
{
	int lidx = logicalIndexOfLayer(l);
	if (lidx < 0)
		return -1;
	return view_->verticalHeader()->visualIndex(lidx);
}
int
LayerView::logicalIndexOfLayer(const MapLayer* l)
{
	QHeaderView* vh = view_->verticalHeader();
	int n_layers = model_->rowCount(view_->rootIndex());
	for (int i = 0; i < n_layers; ++i) {
		int lidx = vh->logicalIndex(i);
		if (model_->layerAt(lidx) == l)
			return lidx;
	}
	return -1;
}
void
LayerView::removeLayer(MapLayer* layer)
{
	model_->removeLayer(layer);
}
QVector<MapLayer*>
LayerView::getLayers() const
{
#if 0
	int n_layers = model_->rowCount(view_->rootIndex());

	QVector<MapLayer*> vl(n_layers);
	QHeaderView* vh = view_->verticalHeader();
	qDebug() << "getLayers: n_layers = " << n_layers;
	for (int i = 0; i < n_layers; ++i) {
		int lidx = vh->logicalIndex(i);
		MapLayer* l = model_->layerAt(lidx);
		qDebug() << "logicalIndex[" << i << "] = " << lidx;

		Q_ASSERT(l != NULL);
		vl[i] = l;
	}
#else
	int n_layers = model_->rowCount(view_->rootIndex());

	QVector<MapLayer*> vl;
	QHeaderView* vh = view_->verticalHeader();
	for (int i = 0; i < n_layers; ++i) {
		int lidx = vh->logicalIndex(i);
		MapLayer* l = model_->layerAt(lidx);
		qDebug() << "logicalIndex[" << i << "] = " << lidx;
		if (l != NULL)
			vl.push_back(l);
	}
#endif
	return vl;
}
void
LayerView::moveSectionOfName(const QString& name, int toVisualIndex)
{
	MapLayer* l = layerOfName(name);
	if (l == NULL)
		return;
	int fromVisualIndex = visualIndexOfLayer(l);
	
	view_->verticalHeader()->moveSection(fromVisualIndex, toVisualIndex);
}
void
LayerView::emit_layersChanged()
{
	QVector<MapLayer*> v = getLayers();
	if (!v.empty()) 
		emit layersChanged(v);
}
void 
LayerView::contextMenuEvent(QContextMenuEvent* event)
{
	QMenu contextMenu(this);

	if (view_->selectedLayerIndexes().empty()) {
		contextMenu.addAction(openAction_);
	} else {
		contextMenu.addAction(upAction_);
		contextMenu.addAction(downAction_);
		contextMenu.addSeparator();
		contextMenu.addAction(saveAction_);
		contextMenu.addAction(openAction_);
		contextMenu.addSeparator();
		contextMenu.addAction(deleteAction_);
	}
        
	contextMenu.exec(event->globalPos());
}
void
LayerView::updateLayers()
{
	emit_layersChanged();
}
void
LayerView::on_dataChanged(const QModelIndex& i0, const QModelIndex& i1)
{
	Q_UNUSED(i0);
	Q_UNUSED(i1);

	emit_layersChanged();
}
void
LayerView::on_sectionMoved(int logicalIndex, int oldVisualIndex,
                           int newVisualIndex)
{
	qDebug() << logicalIndex << ": " 
             << oldVisualIndex << " -> " << newVisualIndex << "\n";
	emit_layersChanged();
}
void
LayerView::on_viewPressed(const QModelIndex &index)
{
	Q_UNUSED(index);
	qDebug() << "on_viewPressed"
             << "index->row() = " << index.row() << "\n"
             << "index->column() = " << index.column() << "\n";
}
void
LayerView::saveSelected()
{
	// N.B.  これはこのクラスでなくMapManagerのするべき仕事かもしれない
	qDebug() << "saveSelected()";
        
	QVector<int> indexes(view_->selectedLayerIndexes());
        
	foreach (int i, indexes) {
		MapLayer* layer = model_->layerAt(i);
		QString fileName = QFileDialog::
			getSaveFileName(this, tr("Save Layer"), 
							QString("layer_%1.xml").arg(i),
							tr("Layer Xml (*.xml)"));

		if (!fileName.isEmpty()) {
			QFile file(fileName);
			if (!file.open(QIODevice::WriteOnly)) {
				QMessageBox::warning(this, tr("MapViewer"),
									 tr("Cannot write file %1:\n%2.")
									 .arg(file.fileName())
									 .arg(file.errorString()));
			} else {
				QTextStream ts(&file);
				ts << layer->getXml();
			}
		} 
	}
}
void
LayerView::deleteSelected()
{
	qDebug() << "deleteSelected()";
	QVector<int> indexes(view_->selectedLayerIndexes());
        
	foreach (int i, indexes) {
		MapLayer* layer = model_->layerAt(i);
		emit removeLayerRequested(layer);
	}
}

/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
