/*!
 * @file map_grid.h
 * @brief Special QGraphicItem for displaying coordinate grid.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#ifndef   MAP_GRID_HEADER_
#define   MAP_GRID_HEADER_

#include <QObject>
#include <QGraphicsItem>

/* @class MapGrid
 * @brief Mapに座標格子を表示させるクラス。
 * 
 * MapView内部で使用され，特殊なレイヤとして扱われる。
 * メッシュの大きさを倍率から自動計算する。
 */
class MapGrid : public QObject, public QGraphicsItem
{
	Q_OBJECT;
public:
	static const qreal NO_GRID = -1.0; //!< grid非表示の意味の定数
	static const qreal DEFAULT_MARGIN = 0.1; //!< margin_非指定時初期値

	//! コンストラクタ。
	//! @param[in] extent 画像座標系で指定した座標格子範囲
	MapGrid(const QRectF& extent=QRectF());
	virtual ~MapGrid() { }   //!< 仮想デストラクタ

	//! 画像座標系での表示範囲を返す。@a gridExtent_ に@a  margin_ が加わる
	QRectF boundingRect() const;

	//!< 自身を描画(格子を描く)
	//!< @see QGraphicsItem::paint()
	void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
			   QWidget *widget);

	void setGridExtent(const QRectF& rect); //!< 格子表示範囲を@a rectに指定
	QRectF gridExtent() const;              //!< 座標格子表示範囲を返す

	void setMargin(qreal margin);            //!< マージンを@a margin に設定
	qreal margin() const { return margin_; } //!< マージンを返す

	void setVisible(bool visible); //!< 可視／不可視を設定

signals:
	/*! SIGNAL: gridの大きさが変化した場合に発火
	 *  @param[in] xGridSpacing     x方向のグリッド間隔
	 *  @param[in] yGridSpacing     y方向のグリッド間隔
	 *  @param[in] xFineGridSpacing x方向の詳細グリッド間隔
	 *  @param[in] yFineGridSpacing y方向の詳細グリッド間隔
	 */
	void gridSpacingChanged(qreal xGridSpacing, qreal xFineGridSpacing,
							qreal yGridSpacing, qreal yFineGridSpacing);
protected:
	/*! gridを実際に描画する下請け関数
	 *  @param[in] painter Qtのペインタ
	 *  @param[in] rect 描画範囲
	 *  @param[in] level 詳細レベル @see QStyleOptionGraphicsItem::levelOfDetail
	 *  @param[in] opacity 描画する線の不透明度
	 *  @param[out] xGridSpacing x方向グリッド間隔を返す
	 *  @param[out] yGridSpacing x方向グリッド間隔を返す
	 */
	void drawGrid(QPainter* painter, const QRectF& rect, 
				  qreal level, const QPen& pen, qreal opacity,
				  qreal* xGridSpacing, qreal* yGridSpacing);
private:
	QRectF gridExtent_;      //!< 座標格子の表示範囲
	qreal xGridSpacing_;     //!< x座標格子のメモリ間隔
	qreal xFineGridSpacing_; //!< x座標詳細格子のメモリ間隔
	qreal yGridSpacing_;     //!< y座標格子のメモリ間隔
	qreal yFineGridSpacing_; //!< y座標詳細格子のメモリ間隔
	qreal margin_;           //!< 格子外側のマージン割合(相対値:1.0->100%)
};
#endif /*!___MAP_GRID_HEADER___*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
