/*!
 * @file map_layer.cpp
 * @brief Classes for Map Layer Objects.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */

#include <QtGui>
#include <limits>

#include "map_layer.h"

/* ================================================================ *
 *   Utility functions in unnamed namespace 
 * ================================================================ */

namespace /* unnamed */ {

/*!
 * @brief  normalize the opacity between [0.0, 1.0].
 * @param  opacity  opacity to be normalized.
 * @return normalized value.
 */
qreal
normalizeOpacity(qreal opacity)
{
	if (opacity < 0.0)
		return 0.0;
	else if (opacity >= 1.0)
		return 1.0;
	else
		return opacity;
}
/*!
 * @brief returns default QPen object used by VectorMapLayer
 */
QPen
defaultPen(void)
{
	return QPen(Qt::blue, 2.0, Qt::SolidLine, Qt::RoundCap, Qt::MiterJoin);
}
/*!
 * @brief coordinate transformation form geo to image for (QPointF)
 */
QPointF
geoToImage(const QPointF& pgeo)
{
	return QPointF(pgeo.x(), -pgeo.y());
}
/*!
 * @brief coordinate transformation form geo to image for QRectF
 */
QRectF
geoToImage(const QRectF& rgeo)
{
	QPointF topLeft_image(geoToImage(rgeo.topLeft()));
	QPointF bottomRight_image(geoToImage(rgeo.bottomRight()));
	return QRectF(topLeft_image, bottomRight_image).normalized();
}
/*!
 * @brief coordinate transformation form geo to image for QPolygonF
 */
QPolygonF
geoToImage(const QPolygonF& polygeo)
{
	QPolygonF polyimage(polygeo); // inefficient!?
	const int n_points = polyimage.size();

	for (int i = 0; i < n_points; ++i) {
		polyimage[i].ry() *= -1.0;
	}

	return polyimage;
}


inline qreal
deg2rad(qreal deg)
{
        return 3.1415926535 * deg / 180.0;
}
inline qreal
rad2deg(qreal rad)
{
        return rad * 180 / 3.1415926535;
}

} /* unnamed */


/* ================================================================ *
 *   Class MapLayer
 * ================================================================ */

MapLayer::MapLayer(const QString& name, bool visible, qreal opacity,
                   const QRectF& extent)
	: name_(name), 
	  geoExtent_(extent)
{
	setVisible(visible);
	setOpacity(opacity);
}
QRectF
MapLayer::boundingRect() const
{
	return geoToImage(geoExtent_);
}
/* ================================================================ *
 *   Class RasterMapLayer
 * ================================================================ */

RasterMapLayer::RasterMapLayer(const QString& name, bool visible, qreal opacity,
                               const QRectF& extent)
	: MapLayer(name, visible, opacity, extent)
{
        
}
void 
RasterMapLayer::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                      QWidget *widget)
{
	Q_UNUSED(option);
	Q_UNUSED(widget);
	painter->drawPixmap(geoToImage(geoExtent_), pixmap_, pixmap_.rect());
}
void
RasterMapLayer::setPixmap(const QPixmap& pixmap, const QRectF& extent)
{
	pixmap_ = pixmap;
	geoExtent_ = extent;
}

/* ================================================================ *
 *   Class VectorMapLayer
 * ================================================================ */

VectorMapLayer::VectorMapLayer(const QString& name, bool visible, qreal opacity,
                               const QRectF& extent)
	: MapLayer(name, visible, opacity, extent)
{
        
}
void
VectorMapLayer::setPen(int id, const QPen& pen)
{
	pens_[id] = pen;
}
void
VectorMapLayer::setPath(int id, const QVector<QPointF>& path)
{
	// We need transformation to geo -> image coordinate.
	// Y axis of image coord extends below but geo's above.
	const int n_points = path.size();
	if (n_points <= 0)
		return;

	paths_[id] = QPolygonF(path);

	updateExtent();
}
void
VectorMapLayer::clearPath(int id)
{
	paths_.remove(id);
}
void
VectorMapLayer::clearPen(int id)
{
	pens_.remove(id);
}
void
VectorMapLayer::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                      QWidget *widget)
{
	Q_UNUSED(option);
	Q_UNUSED(widget);

	QMapIterator<int, QPolygonF> i(paths_);
	while (i.hasNext()) {
		i.next();
		const int id = i.key();
		const QPolygonF& polygon = i.value();

		if (pens_.contains(id))
			painter->setPen(pens_[id]);
		else
			painter->setPen(defaultPen());
                                
		painter->drawPolyline(geoToImage(polygon));
	}
}
void
VectorMapLayer::updateExtent()
{
	QMapIterator<int, QPolygonF> i(paths_);
	bool first = true;
	QRectF brect;
	while (i.hasNext()) {
		i.next();
		const QRectF tmp = i.value().boundingRect();
		if (first) {
			brect = tmp;
			first  = false;
		} else {
			brect = brect.united(tmp);
		}
		qDebug() << "brect = " << brect;
	}
	geoExtent_ = brect;
}
/* ================================================================ *
 *   Class TextMapLayer
 * ================================================================ */

TextMapLayer::TextMapLayer(const QString& name, bool visible, qreal opacity,
                           const QRectF& extent)
	: MapLayer(name, visible, opacity, extent)
{
}
void
TextMapLayer::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                    QWidget *widget)
{
	Q_UNUSED(option);
	Q_UNUSED(widget);

	QMapIterator<int, TextInfo> i(texts_);
	while (i.hasNext()) {
		i.next();
		const TextInfo& t = i.value();
		painter->save();

		painter->setFont(t.font);
		painter->setPen(t.pen);
		painter->translate(geoToImage(t.pos));
		painter->scale(0.1, 0.1);
		painter->drawText(QPoint(0, 0), t.text);
                
		painter->restore();
	}
}
void 
TextMapLayer::setText(int id, const QString& text, const QPointF& pos,
                      const QFont& font, const QPen& pen)
{
	texts_[id] = TextInfo(text, pos, font, pen);
	updateExtent();
}
void 
TextMapLayer::clearText(int id)
{
	texts_.remove(id);
}
QString 
TextMapLayer::text(int id) const
{
	return texts_[id].text;
}
QPointF 
TextMapLayer::position(int id) const
{
	return texts_[id].pos;
}
QFont 
TextMapLayer::font(int id) const
{
	return texts_[id].font;
}

QPen 
TextMapLayer::pen(int id) const
{
	return texts_[id].pen;
}
QRectF 
TextMapLayer::textBoundingRect(int id) const
{
	QMap<int, TextInfo>::const_iterator i = texts_.constFind(id);
	if (i == texts_.constEnd())
		return QRectF();

	QFontMetricsF fm(i->font);

	return fm.boundingRect(i->text).translated(i->pos);

}
void
TextMapLayer::updateExtent()
{
	QMapIterator<int, TextInfo> i(texts_);
	bool first = true;
	QRectF brect;
	while (i.hasNext()) {
		i.next();
		const QRectF tmp = textBoundingRect(i.key());
		if (first) {
			brect = tmp;
			first  = false;
		} else {
			brect = brect.united(tmp);
		}
		qDebug() << "brect = " << brect;
	}
	geoExtent_ = brect;
}

/* ================================================================ *
 *   Class MarkerMapLayer
 * ================================================================ */
MarkerMapLayer::MarkerMapLayer(const QString& name, bool visible, qreal opacity,
							   const QRectF& extent)
	: MapLayer(name, visible, opacity, extent)
{
}

void
MarkerMapLayer::setMarker(int id, const QPointF& pos, const Vector2D_t& vel,
						  qreal rx, qreal ry, qreal theta, const QPen& pen)
{
	marks_[id] = MarkerInfo(pos, vel, rx, ry, theta, pen);
	updateExtent();
}

void
MarkerMapLayer::clearMarker(int id)
{
	marks_.remove(id);
}
void
MarkerMapLayer::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                      QWidget *widget)
{
	qDebug() << " MarkerMapLayer::paint";
	Q_UNUSED(option);
	Q_UNUSED(widget);

#if QT_VERSION >= 0x040600
	const qreal lod = option->levelOfDetailFromTransform(painter->worldTransform());
#else
	const qreal lod = option->levelOfDetail;
#endif
	QPainterPath arrow;
	arrow.moveTo(0.5, 0.0);
	arrow.lineTo(-0.5, +0.1);
	arrow.lineTo(-0.5, -0.1);
	arrow.lineTo(0.5, 0.0);
	arrow.closeSubpath();
	arrow.setFillRule(Qt::WindingFill);

	QPainterPath star;
	star.moveTo(1.0, 0.0);
	star.lineTo(-0.80902, 0.58779);
	star.lineTo(0.30902, -0.95106);
	star.lineTo(0.30902, 0.95106);
	star.lineTo(-0.80902, -0.58779);
	star.lineTo(1.0, 0.0);
	star.closeSubpath();
	star.setFillRule(Qt::WindingFill);

	const qreal unitlen = 5.0/lod;
	qDebug() << "MarkerMapLayer:: lod = " << lod ;
        
	QMapIterator<int, MarkerInfo> i(marks_);
	while (i.hasNext()) {
		i.next();
		const MarkerInfo& mi = i.value();
		painter->save();
		painter->translate(geoToImage(mi.pos));

		// error ellipse
		painter->save();
		painter->rotate(-rad2deg(mi.theta));
		painter->setPen(mi.pen);
		painter->drawEllipse(QPointF(0, 0), mi.rx, mi.ry);
		painter->drawEllipse(QPointF(0, 0), 3.0*mi.rx , 3.0*mi.ry);
		painter->restore();

		if (mi.velocity.isNull()) {
			// draw star : position  
			painter->save();
			painter->scale(3.0*unitlen, 3.0*unitlen);
			painter->setBrush(Qt::blue);
			painter->setPen(Qt::darkBlue);
			painter->setOpacity(0.5);
			painter->drawPath(star);
			painter->restore();
		} else {
			// draw an arrow for direction and circle for position
			painter->save();
			painter->setPen(QPen(Qt::magenta, 0.3*unitlen));
			painter->drawEllipse(QPointF(0,0), 1.7*unitlen, 1.7*unitlen);

			qreal theta = atan2(mi.velocity.y(), mi.velocity.x());
			painter->rotate(-rad2deg(theta));
			painter->scale(5*unitlen, 5*unitlen);
			painter->setBrush(Qt::magenta);
			painter->setPen(Qt::magenta);
			painter->drawPath(arrow);
			painter->restore();
		}
		painter->restore();
	}
}

namespace /* unnamed */ {

/*! make QRectF from center position and half width/height
 *  used by MarkerMapLayer::updateExtent()
 */
QRectF
rectFromCenter(const QPointF& center, qreal half_w, qreal half_h)
{
	QPointF topLeft(    center.x() - half_w, center.y() - half_h);
	QPointF bottomRight(center.x() + half_w, center.y() + half_h);

	return QRectF(topLeft, bottomRight);
}
} /* unnamed namespace */

void
MarkerMapLayer::updateExtent()
{
	QMapIterator<int, MarkerInfo> i(marks_);
	bool first = true;
	QRectF brect;
	while (i.hasNext()) {
		i.next();
		const MarkerInfo& mi = i.value();
		const QRectF tmp(rectFromCenter(mi.pos, mi.rx, mi.ry));
		if (first) {
			brect = tmp;
			first  = false;
		} else {
			brect = brect.united(tmp);
		}
		qDebug() << "brect = " << brect;
	}
	geoExtent_ = brect;
}

/* ---------------------------------------------------------------- *
 * GridMaplayer
 * ---------------------------------------------------------------- */
GridMapLayer::GridMapLayer(const QString& name, bool visible,
                           qreal opacity, const QRectF& extent)
        : MapLayer(name, visible, opacity, extent),
          origin_(),
          theta_(0.0),
          rmax_(0.0)
{
}
void
GridMapLayer::setPixmap(const QPixmap& pixmap)
{
        pixmap_ = pixmap;
}
namespace /* unnamed */ {
qreal
norm(qreal x, qreal y)
{
        return sqrt(x*x + y*y);
}
qreal
norm(const QPointF& p)
{
        return norm(p.x(), p.y());
}
} /* unnamed namespace */

void
GridMapLayer::setGridInfo(const QPointF& origin, const QRectF& grid_area,
                          qreal theta)
{
        origin_ = origin;
        grid_area_ = grid_area;
        theta_ = theta;

        double r0 = norm(grid_area_.bottomLeft());
        double r1 = norm(grid_area_.bottomRight());
        double r2 = norm(grid_area_.topLeft());
        double r3 = norm(grid_area_.topLeft());

        rmax_ = qMax(qMax(r0, r1), qMax(r2, r3));
}


QRectF
GridMapLayer::boundingRect() const
{
        QRectF br(QPointF(origin_.x() + rmax_, -origin_.y() + rmax_),
                  QPointF(origin_.x() - rmax_, -origin_.y() - rmax_));

        qDebug() << "BoundingRect: " << br << "\n";

        return br;
}
void 
GridMapLayer::paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
                    QWidget *widget)
{
        Q_UNUSED(option);
        Q_UNUSED(widget);

        painter->save();

        painter->translate(geoToImage(origin_));
        painter->rotate(-rad2deg(theta_));
        painter->drawPixmap(geoToImage(grid_area_), pixmap_, pixmap_.rect());

        painter->restore();
}

/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
