/*!
 * @file map_layer.h
 * @brief Classes for Map Layer Objects.
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#ifndef  MAP_LAYER_HEADER_
#define  MAP_LAYER_HEADER_

#include <QString>
#include <QPixmap>
#include <QMap>
#include <QGraphicsItem>
#include <QPolygonF>
#include <QPen>
#include <QFont>
#include "common_defs.h"
QT_FORWARD_DECLARE_CLASS(QPainter);


/*! @class MapLayer
 *  @brief Viewerに表示するレイヤの抽象規定クラス。
 * 
 *   全て種類のレイヤに共通の機能(name, geoExtent, boundingRect)の実装
 *   と，純粋仮想関数paintのsignatureを提供する。QtのQGraphicsView
 *   フレームワークで画像を表示させるため，QGraphicsItemを継承。
 *   このため，boundingRect, paint, advanceの仮想関数を継承。
 *   これらの詳細はQtのQGraphicsItem/QGraphicsView/QGraphicsSceneの
 *   を参照
 *
 */
class MapLayer : public QObject, public QGraphicsItem
{
	Q_OBJECT;
public:
	/*! コンストラクタ
	 *  @param [in] name  レイヤ名
	 *  @param [in] visible 可視／不可視のフラグ
	 *  @param [in] opacity 不透明度
	 *  @param [in] extent 地図存在範囲
	 */
	MapLayer(const QString& name, bool visible=true, qreal opacity=1.0,
			 const QRectF& extent = QRectF(0.0, 0.0, 0.0, 0.0));

	//! デストラクタ。抽象基底なのでpublic 仮想とする
	virtual ~MapLayer() { }

	//! レイヤ名を返す
	virtual QString name() const   { return name_; }
	//! レイヤ名を設定
	//! @param[in] name あたらしいレイヤ名
	virtual void setName(const QString& name) { name_ = name; }

	//! 地図の存在範囲を返す。地図座標系での値が帰る
	virtual QRectF geoExtent() const { return geoExtent_; }
	//! 地図の存在範囲(extent)を設定
	//! @param[in] extent 新しい extent
	virtual void setGeoExtent(const QRectF& extent) { geoExtent_ = extent; }

	/*! QGraphicsItemの仮想関数。このオブジェクトの存在範囲を返す。
	 *  QGraphicsViewから呼ばれるため画像座標での値を返す。
	 *  この基底クラスでは，デフォルト実装として，extentの値を
	 *  地図座標系から画像座標系に変換して返す。
	 */
	virtual QRectF boundingRect() const;

	/*! 自身を描画する。このクラスでは純粋仮想
	 *  @see QGraphicsItem::paint()
	 */
	virtual void paint(QPainter *painter,
					   const QStyleOptionGraphicsItem *option,
					   QWidget *widget) = 0;

	//! XMLデータをセット
	//! @param[in] xml 設定するXML文字列
	virtual void setXml(const QString& xml) { xml_ = xml; }
	//! XMLデータを返す
	virtual QString getXml() const { return xml_; }

#if QT_VERSION < 0x040500
#warning "opacity is not supported Qt older than 4.5"

   // dummy for older Qt without void QGraphicsItem::setOpacity(qreal);
	virtual void setOpacity(qreal opacity) 
	{
		opacity_ = opacity;
	}
#endif

protected:
	//! QGraphicsItem の仮想関数。描画前と描画後の通知に使われる。
	virtual void advance(int /*step*/) { }
        
	QString name_;        //!< レイヤー名称
	QRectF geoExtent_;    //!< このレイヤの存在範囲(地図座標系)
	QString xml_;         //!< このレイヤに対応するXMLデータ

#if QT_VERSION < 0x040500
	qreal opacity_;		  //!< opacityがサポートされていない場合のダミー
#endif

};
/*! @class RasterMapLayer
 *  ラスターデータを持つ地図レイヤ。pixmapをprivateデータに持つ。
 */
class RasterMapLayer : public MapLayer
{
	Q_OBJECT;
public:
	/*! コンストラクタ
	 *  @param [in] name  レイヤ名
	 *  @param [in] visible 可視／不可視のフラグ
	 *  @param [in] opacity 不透明度
	 *  @param [in] extent 地図存在範囲
	 */
	RasterMapLayer(const QString& name, 
				   bool visible=true, 
				   qreal opacity=1.0,
				   const QRectF& extent = QRectF(0.0, 0.0, 0.0, 0.0));
	//! デストラクタ。
	virtual ~RasterMapLayer() { }

	//! 自身を描画する。メンバに持つpixmapの描画を行う。
	//! @see QGraphicsItem::paint()
	void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
			   QWidget *widget);

	/*! 描画用pixmapデータを設定する。
	 * @param[in] pixmap 設定するpixmap
	 * @param[in] extent 地図上の存在範囲
	 */
	void setPixmap(const QPixmap& pixmap, const QRectF& extent);
private:
	QPixmap pixmap_;        //!< 描画用QPixmapデータ
};

/* ---------------------------------------------------------------- *
 *
 *  TODO: {Vector,Text,Marker}MapLayer に共通部分多し。refactorできるか？
 *
 * ---------------------------------------------------------------- */

/*! @class VectorMapLayer
 *  @brief ベクタデータを持つ地図レイヤ。描画用QPenデータと点列データを持つ。
 *
 *  点列データとペンデータはid番号で管理され，複数持つことも出来る
 */
class VectorMapLayer : public MapLayer
{
	Q_OBJECT;
public:
	/*! コンストラクタ
	 *  @param [in] name  レイヤ名
	 *  @param [in] visible 可視／不可視のフラグ
	 *  @param [in] opacity 不透明度
	 *  @param [in] extent 地図存在範囲
	 */
	VectorMapLayer(const QString& name, 
				   bool visible=true, 
				   qreal opacity=1.0,
				   const QRectF& extent = QRectF(0.0, 0.0, 0.0, 0.0));

	//! デストラクタ。
	virtual ~VectorMapLayer() { }

	//! @a id番のペンデータを設定
	void setPen(int id, const QPen& pen);

	//! @a id番の点列データを設定
	void setPath(int id, const QVector<QPointF>& pathdata);

	//! @a id番のパスデータをクリア
	void clearPath(int id);

	//! @a id番のペンデータをクリア。ペンをクリアするとデフォルトが使用される
	void clearPen(int id);

	//! 自身を描画する。設定されたパスを全て描画する。
	//! @see QGraphicsItem::paint()
	void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
			   QWidget *widget);
protected:
	//! 自身の地図上での存在範囲を更新する。pathの設定更新時に呼ばれる
	void updateExtent();
private:
	QMap<int, QPen> pens_;       //!< 描画用penデータ
	QMap<int, QPolygonF> paths_; //!< 描画用pathデータ
};

/*! @class TextMapLayer
 *  @brief 地図上にテキストを表示するレイヤ
 *
 */
class TextMapLayer : public MapLayer
{
	Q_OBJECT;
public:
	/*! コンストラクタ
	 *  @param [in] name  レイヤ名
	 *  @param [in] visible 可視／不可視のフラグ
	 *  @param [in] opacity 不透明度
	 *  @param [in] extent 地図存在範囲
	 */
	TextMapLayer(const QString& name, 
				 bool visible=true, 
				 qreal opacity=1.0,
				 const QRectF& extent = QRectF(0.0, 0.0, 0.0, 0.0));

	//! デストラクタ。
	virtual ~TextMapLayer() { }

	//! @a id 番のテキストを@a textに,位置を@a posに，fontを@a fontに，ペンを@a penに設定
	void setText(int id, const QString& text, const QPointF& pos,
				 const QFont& font = QFont(), const QPen& pen = QPen());

	//! @a id 番のテキストを削除
	void clearText(int id);

	//! @a id 番のテキストを返す
	QString text(int id) const;
	//! @a id 番のテキストの場所を返す
	QPointF position(int id) const;
	//! @a id 番のテキストのfontを返す
	QFont font(int id) const;
	//! @a id 番のテキストのfontを返す
	QPen pen(int id) const;
	//! @a id 番のテキストの外接長方形を返す
	QRectF textBoundingRect(int id) const;
        
	//! 自身を描画する。
	//! @see QGraphicsItem::paint()
	//! 自身の持つテキストを描画する
	void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
			   QWidget *widget);
protected:
	//! 自身の地図上での存在範囲を更新する。text/style更新時に呼ばれる。
	void updateExtent();
private:
	//! @struct TextInfo 
	//! @brief 描画テキスト属性をまとめた構造体
	typedef struct TextInfo {
		QString text;   //!< 文字列
		QPointF pos;    //!< 表示位置(地図座標)
		QFont font;     //!< フォント
		QPen pen;       //!< ペン
		TextInfo() { }  //!< デフォルトコンストラクタ

		//! @a text @a pos @a font および@a penを指定したコンストラクタ
		TextInfo(QString text, QPointF pos, QFont font, QPen pen)
			: text(text), pos(pos), font(font), pen(pen)
		{ }
	};

	QMap<int, TextInfo> texts_; //!< 描画用テキスト及び属性のMap
};


/*! @class MarkerMapLayer 
 *  位置マーカ表示のための地図レイヤ。（現在位置など。)
 *  マーカは楕円で描かれる
 */
class MarkerMapLayer : public MapLayer
{
	Q_OBJECT;
public:
	typedef QPointF Vector2D_t; // Memo: Qt 4.6で QVector2Dが導入されているが，

	/*! コンストラクタ
	 *  @param [in] name  レイヤ名
	 *  @param [in] visible 可視／不可視のフラグ
	 *  @param [in] opacity 不透明度
	 *  @param [in] extent 地図存在範囲
	 */
	MarkerMapLayer(const QString& name, 
				   bool visible=true, 
				   qreal opacity=1.0,
				   const QRectF& extent = QRectF(0.0, 0.0, 0.0, 0.0));

	//! デストラクタ。
	virtual ~MarkerMapLayer() { /* empty */ }

	//! id番目のマーカー情報を設定
	void setMarker(int id, const QPointF& pos, const Vector2D_t& vel, 
				   qreal rx, qreal ry, qreal theta = 0.0,
				   const QPen& pen = QPen());

	//! id番目のマーカー情報をクリア
	void clearMarker(int id);
        
	//! 自身を描画する。楕円で位置を表示
	//! @see QGraphicsItem::paint()
	void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
			   QWidget *widget);
protected:
	//! 自身の地図上での存在範囲を更新する。マーカ情報更新時に呼ばれる。
	void updateExtent();
private:
	//! @struct マーカ属性
	typedef struct MarkerInfo {
		QPointF pos;    //!< 表示位置(地図座標)
		Vector2D_t velocity;
		qreal rx;       //!< x方向半径
		qreal ry;       //!< y方向半径
		qreal theta;    //!< 回転方向[rad]
		QPen pen;       //!< ペン

		MarkerInfo() { } //!< デフォルトコンストラクタ
		//! 位置，半径(x, y)，ペンを指定したコンストラクタ
		//! @param [in] pos 場所
		//! @param [in] velocity 速度
		//! @param [in] rx x半径
		//! @param [in] ry y半径
		//! @param [in] theta 回転[rad]
		//! @param [in] pen ペンデータ
		MarkerInfo(QPointF pos, Vector2D_t velocity, 
				   qreal rx, qreal ry, qreal theta, QPen pen)
			: pos(pos), velocity(velocity), 
			  rx(rx), ry(ry), theta(theta), pen(pen)
		{ }
	};

	QMap<int, MarkerInfo> marks_; //!< idとマークを対応づけるmap
};


/*! @class GridMapLayer 
 *  GridMapのためのレイヤ。ロボットローカル座標系でのgridmapを描画する。
 *  
 */
class GridMapLayer : public MapLayer
{
	Q_OBJECT;
public:
	/*! コンストラクタ
	 *  @param [in] name  レイヤ名
	 *  @param [in] visible 可視／不可視のフラグ
	 *  @param [in] opacity 不透明度
	 *  @param [in] extent 地図存在範囲
	 */
	GridMapLayer(const QString& name, 
				 bool visible=true, 
				 qreal opacity=1.0,
				 const QRectF& extent = QRectF(0.0, 0.0, 0.0, 0.0));

	QRectF boundingRect() const;
	virtual ~GridMapLayer() { } //!< デストラクタ。

	//! 自身を描画する。
	//! @see QGraphicsItem::paint()
	void paint(QPainter *painter, const QStyleOptionGraphicsItem *option,
			   QWidget *widget);

	//! GridMapのデータをpixmapとして登録する
	//! @param[in] pixmap 描画データとなるpixmap
	void setPixmap(const QPixmap& pixmap);

	/*! gridの座標情報を設定する
	 *  @param[in] origin ローカル(self)座標の原点がワールドのどこに有るか
	 *  @param[in] grid_area ローカル座標系でのグリッドの存在範囲
	 *  @param[in] theta ローカル座標系のワールドに対する回転角度(rad)
	 */
	void setGridInfo(const QPointF& origin, const QRectF& grid_area, qreal theta);
private:
	QPixmap pixmap_;
	QPointF origin_;
	QRectF grid_area_;
	qreal theta_;
	qreal rmax_;
};


#endif /*MAP_LAYER_HEADER_*/
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
