/*!
 * @file map_layer_list_model.cpp
 * @brief A Model to represent for the MapLayer in ItemView framework
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */


#include "map_layer.h"
#include "map_layer_list_model.h"

#include <QtGui>
#include <iostream>

#include "debug_utils.h"

using namespace std;

MapLayerListModel::MapLayerListModel(QObject *parent)
	: QAbstractListModel(parent), mapLayers_()
{
}

int 
MapLayerListModel::rowCount(const QModelIndex &parent) const
{
	Q_UNUSED(parent);
	return mapLayers_.count();
}
int 
MapLayerListModel::columnCount(const QModelIndex &parent) const
{
	Q_UNUSED(parent);
	return N_COLUMNS;
}
QVariant
MapLayerListModel::data(const QModelIndex &index, int role) const
{
	if (!index.isValid())
		return QVariant();

	if (index.row() >= mapLayers_.size())
		return QVariant();

	const MapLayer* layer = mapLayers_.at(index.row());
	switch(index.column()) {
	case COL_VISIBILITY:
		if (role == Qt::DisplayRole || role == Qt::EditRole) {
			return layer->isVisible();
		} else
			return QVariant();
	case COL_OPACITY:
		if (role == Qt::DisplayRole || role == Qt::EditRole) {
			return 100.0 * layer->opacity(); // as percent
		} else
			return QVariant();
		break;
	case COL_NAME:
		if (role == Qt::DisplayRole || role == Qt::EditRole)
			return layer->name();
		else
			return QVariant();
	default:
		return QVariant();
	}
}
QVariant
MapLayerListModel::headerData(int section, Qt::Orientation orientation,
                              int role) const
{

	if (role != Qt::DisplayRole)
		return QVariant();

	if (orientation == Qt::Horizontal) {
		switch (section) {
		case COL_VISIBILITY:
			return QString(tr("Visible"));
		case COL_OPACITY:
			return QString(tr("Opacity(%)"));
		case COL_NAME:
			return QString(tr("Layer Name"));
		default:
			return QVariant();
		}
	} else {
		return QString("%1").arg(section);
		//return QString();
	}
}
Qt::ItemFlags
MapLayerListModel::flags(const QModelIndex &index) const
{
	if (!index.isValid()) 
		return Qt::NoItemFlags;
	else
		return QAbstractItemModel::flags(index) | Qt::ItemIsEditable;
}
bool
MapLayerListModel::setData(const QModelIndex &index, const QVariant &value,
                           int role)
{
	if (index.isValid() && role == Qt::EditRole) {
		Tracer t("Editing MapLayer", std::cout);
		MapLayer* layer = mapLayers_.at(index.row());
		bool changed = false;
		switch (index.column()) {
		case COL_VISIBILITY:
			if (value.canConvert(QVariant::Bool)) {
				layer->setVisible(value.toBool());
				changed = true;
			} else {
				cout << "setData::COL_VISIBILITY!!!!\n";
			}
			break;
		case COL_OPACITY:
			if (value.canConvert(QVariant::Double)) {
				qreal percent = value.toDouble();
				cout << "setData/COL_OPACITY " 
					 << percent << "\n";
				layer->setOpacity(percent / 100.0);
				changed = true;
			} else {
				cout << "setData::COL_OPACITY!!!!\n";
			}
			break;
		case COL_NAME:
			if (value.canConvert(QVariant::String)) {
				layer->setName(value.toString());
				changed = true;
			} else {
				cout << "setData::COL_NAME!!!!\n";
			}
		default:
			changed = false;
			break;
		}
		if (changed) {
			qDebug() << "\ndata is changed\n";
			emit dataChanged(index, index);
		} else {
			qDebug() << "\ndata is not changed\n";
		}
		return changed;
	}
	return false;
}

int 
MapLayerListModel::addMapLayer(MapLayer* layer)
{
	mapLayers_.append(layer);
	return mapLayers_.size();
}
int
MapLayerListModel::removeLayer(MapLayer* layer)
{
	int idx = mapLayers_.indexOf(layer);
	qDebug() << "removeLayer(not deleted here): idx = " << idx << "\n";
	if (idx >= 0) {
		mapLayers_.removeOne(layer);
	}
	return mapLayers_.size();
}
MapLayer* 
MapLayerListModel::layerAt(int index) const
{
	if (index < 0 || mapLayers_.size() <= index) {
		return NULL;
	} else {
		return mapLayers_[index];
	}
}

MapLayer*
MapLayerListModel::layerOfName(const QString& name) const
{
	QListIterator<MapLayer*> i(mapLayers_);
	while (i.hasNext()) {
		MapLayer* l = i.next();
		if (name == l->name())
			return l;
	}
	return NULL;
}

#if 0
// just for debugging
void
MapLayerListModel::dumpLayers() const
{
//        QListIterator<MapLayer*> i(mapLayers_);
	QListIterator<MapLayer*> i(mapLayers_);
	while (i.hasNext()) {
		MapLayer* l = i.next();
		cout << "opacity " << l->opacity() << "\n";
	}
}
#endif
/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
