/*!
 * @file map_layer_list_model.h
 * @brief A Model to represent for the MapLayer in ItemView framework
 * @author SAGAMI, Tsuyoshi <sagami@brains.co.jp>
 */
#ifndef   MAP_LAYER_LIST_MODEL_HEADER_
#define   MAP_LAYER_LIST_MODEL_HEADER_

#include <QObject>
#include <QAbstractListModel>
#include <QList>

QT_FORWARD_DECLARE_CLASS(QModelIndex);
QT_FORWARD_DECLARE_CLASS(QString);
QT_FORWARD_DECLARE_CLASS(QVariant);

class MapLayer;

/*! 
 * @class MapLayerListModel
 * @brief 地図レイヤ情報のためのmodel。
 *
 * 詳細はQtのmodel/view/delegateフレームワーク参照。（今回のview実装で
 * は)テーブル風に表示しているが，データ管理はレイヤー毎であるため
 * QAbstractListModelを継承。ただし，各属性は列挙定数でcolumnを指定する
 * ことでアクセスする。(より良い実装が有るかもしれない)
 *
 */
class MapLayerListModel : public QAbstractListModel
{
	Q_OBJECT;
public:
	/*! 
	 * レイヤの各属性にアクセスするための「論理」列番号。
	 *  viewでの表示列番号とは無関係であることに注意。
	 */
	enum {
		COL_VISIBILITY = 0,		//!< visibility表示行
		COL_OPACITY,			//!< opacity表示行
		COL_NAME,				//!< name表示行
		N_COLUMNS				//!< 行の数
	};

	//! コンストラクタ
	//! @param[in] parent 親オブジェクト
	MapLayerListModel(QObject *parent = 0); 

	//! 行数を返す。
	//! @see QAbstractListModel::rowCount()
	int rowCount(const QModelIndex &parent = QModelIndex()) const;
	//! 列数を返す。
	//! @see QAbstractListModel::columnCount()
	int columnCount(const QModelIndex &parent = QModelIndex()) const;

	//! indexで指定された場所のデータを返す。
	//! @see QAbstractItemModel::data()
	QVariant data(const QModelIndex &index, int role) const;

	//! indexで指定された場所のヘッダを返す。
	//! @see QAbstractItemModel::headerData()
	QVariant headerData(int section, Qt::Orientation orientation,
						int role = Qt::DisplayRole) const;
        
	//! indexで指定された場所のitemflagを返す
	//! @see QAbstractItemModel::flags()
	//! @see Qt::ItemFlags
	Qt::ItemFlags flags(const QModelIndex &index) const;

	/*! indexで指定された場所のroleに対応するdataを設定。成功したか
	 *  どうかを返す。
	 * @see QAbstractItemModel::setData()
	 */
	bool setData(const QModelIndex &index, const QVariant &value,
				 int role = Qt::EditRole);

	/*! このモデルに新たな地図レイヤを追加する。追加後のレイヤ数を返す
	 *  @param[in] layer 追加したいlayer
	 *  @return 追加後のレイヤの数
	 */
	int addMapLayer(MapLayer* layer);

	/*! このモデルからレイヤを削除する。削除後のレイヤ数を返す
	 *  @param[in] layer 削除したいlayer 
	 *  @return 削除後のレイヤの数。
	 */
	int removeLayer(MapLayer* layer);

	//void dumpLayers() const;

	//! index番目のレイヤへのポインタを返す。
	// @param[in] index 欲しいレイヤのindex
	// @retval nun-null レイヤへのポインタ
	// @retval null 見つからなかった場合
	MapLayer* layerAt(int index) const;

	//! 名前がnameであるレイヤへのポインタを返す。
	// @param[in] name 欲しいレイヤの名前
	// @retval nun-null レイヤへのポインタ
	// @retval null 見つからなかった場合
	MapLayer* layerOfName(const QString& name) const;

private:

	QList<MapLayer*> mapLayers_; //!< 管理するレイヤを格納するコンテナ
};

#endif /*!MAP_LAYER_LIST_MODEL_HEADER_*/


/*
 * Local Variables:
 * mode: c++
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: t
 * End:
 *
 */
