#ifndef  _GRID_MAP_H__
#define  _GRID_MAP_H__

#include <string>
#include <vector>
#include <iostream>

#include "xml_serdes.h"
#include "quantized_vector.h"

/** 存在確率グリッドマップ
 */
struct GridMap{
public:
	/** コンストラクタ
	 */
	GridMap():
		time(0.0),
		id(""),
		coordinate("self"),
		area_x0(0),
		area_y0(0),
		area_x1(0),
		area_y1(0),
		size_x(0),
		size_y(0),
		range_min (0.0),
		range_max (0.0),
		range_step(0.0),
		map(0)
	{};

	/** デストラクタ
	 */
	~GridMap(){
		map.clear();
	};

	/** GridMap の Xサイズを求める
	 *
	 * Xサイズを, area_x0, x1, size_x, から計算する
	 */
	int sizeX(void) const;

	/** GridMap の Yサイズを求める
	 *
	 * Yサイズを, area_y0, y1, size_y, から計算する
	 */
	int sizeY(void) const;

	/** GridMap のデータサイズを変更する
	 *
	 * データサイズを, area_x0, x1, y0, y1, size_x, y から計算し, mapのサイズを設定する
	 */
	int resize(void);

	/** GridMap に他のマップをスケーリングしてマッピングする
	 *
	 */
	int Mapping(GridMap &src);

	/** GridMapの領域を所定の値で埋める．
	 *
	 */
	int FillRectangle(float x0, float y0, float x1, float y1, float val);

	/** GridMapにLineを引く
	 *
	 */
	int DrawPoint(float x, float y, float val);

	/** GridMapにLineを引く
	 *
	 */
	int DrawLine(float x0, float y0, float x1, float y1, float val);

	double time;            ///< 取得時間(s)
	std::string id;         ///< 機器のID (URG0など)
	std::string coordinate; ///< 座標系(self or 9)

	float area_x0;          ///< データの領域座標(m)
	float area_y0;          ///< データの領域座標(m)
	float area_x1;          ///< データの領域座標(m)
	float area_y1;          ///< データの領域座標(m)

	float size_x;           ///< グリッドの大きさ(m)
	float size_y;           ///< グリッドの大きさ(m)

	float range_min;        ///< データの最小値
	float range_max;        ///< データの最大値
	float range_step;       ///< データのステップ数

	std::vector<float> map; ///< グリッドマップ（存在確率 0.0-1.0）

	// type名
	static const char TYPE_ID[]; ///< タイプID文字列

	// 送受信メソッド
	/** データをXMLとして書き出す
	 *
	 * @param [out] os 書き出し先ストリーム
	 * @retval 成功時0, 失敗時は負の値を返す
	 */
	int writeToXml(std::ostream& os) const
	{
		return XmlSerDes::writeToXmlSub(os, *this, TYPE_ID);
	}

	/** XMLからデータを読み出し自分自身を更新する
	 *
	 * @param [in] is 読み出し元ストリーム
	 * @retval 成功時0, 失敗時は負の値を返す
	 */
	int readFromXml(std::istream& is)
	{
		return XmlSerDes::readFromXmlSub(is, *this, TYPE_ID);
	}

	/**
	 * \overload int writeToXml(std::string& s) const
	 * @param [out] s 書き出し先文字列
	 */
	int writeToXml(std::string& s) const
	{
		std::ostringstream os(s);
		return writeToXml(os);
	}

	/** XMLからデータを読み出し自分自身を更新する
	 *
	 * \overload int readFromXml(const std::string& s)
	 * @param [in] s 読み出し元文字列
	 */
	int readFromXml(const std::string& s)
	{
		std::istringstream is(s);
		return readFromXml(is);
	}

private:
	// シリアライズメソッド
	friend class boost::serialization::access;
	BOOST_SERIALIZATION_SPLIT_MEMBER();
	template <class Archive>
	void save(Archive& ar, const unsigned int /* version */) const{
		using namespace boost::serialization;
		using namespace XmlSerDes;

		QuantizedVector qv;

		int data_num;

		ar & BOOST_SERIALIZATION_NVP(time);
		ar & BOOST_SERIALIZATION_NVP(id);
		ar & BOOST_SERIALIZATION_NVP(coordinate);

		ar & BOOST_SERIALIZATION_NVP(area_x0);
		ar & BOOST_SERIALIZATION_NVP(area_y0);
		ar & BOOST_SERIALIZATION_NVP(area_x1);
		ar & BOOST_SERIALIZATION_NVP(area_y1);

		ar & BOOST_SERIALIZATION_NVP(size_x);
		ar & BOOST_SERIALIZATION_NVP(size_y);

		ar & BOOST_SERIALIZATION_NVP(range_min);
		ar & BOOST_SERIALIZATION_NVP(range_max);
		ar & BOOST_SERIALIZATION_NVP(range_step);

		// mapをバイナリデータとして, ビット数を減らして送信
		data_num = ((range_max - range_min) / (range_step)) + 1; // データ数を計算 (1-2個 = 1bit, 3-4個 = 2bit, 5-8個 = 3bit, 9-16個 = 4bit ...)
		qv.setNumberOfValues(data_num);
		qv.setRange(range_min, range_max);

		// データを割り当てる (ビット数の変換)
		qv.assign(map.begin(), map.end());
		ar & make_nvp("map", qv);
	};

	template <class Archive>
	void load(Archive& ar, const unsigned int /* version */){
		using namespace boost::serialization;
		using namespace XmlSerDes;

		QuantizedVector qv;

		ar & BOOST_SERIALIZATION_NVP(time);
		ar & BOOST_SERIALIZATION_NVP(id);
		ar & BOOST_SERIALIZATION_NVP(coordinate);

		ar & BOOST_SERIALIZATION_NVP(area_x0);
		ar & BOOST_SERIALIZATION_NVP(area_y0);
		ar & BOOST_SERIALIZATION_NVP(area_x1);
		ar & BOOST_SERIALIZATION_NVP(area_y1);

		ar & BOOST_SERIALIZATION_NVP(size_x);
		ar & BOOST_SERIALIZATION_NVP(size_y);

		ar & BOOST_SERIALIZATION_NVP(range_min);
		ar & BOOST_SERIALIZATION_NVP(range_max);
		ar & BOOST_SERIALIZATION_NVP(range_step);

		// map をバイナリデータとして受信
		ar & make_nvp("map", qv);

		// データを取り出す
		map.clear(); // 現在のデータを破棄
		qv.copy_to(back_inserter(map));

	};

};

std::ostream& operator<<(std::ostream &os, GridMap &map);

#endif //_GRID_MAP_H__
